/*
  CLAW - a C++ Library Absolutely Wonderful

  CLAW is a free library without any particular aim but being useful to 
  anyone.

  Copyright (C) 2005-2008 Julien Jorge

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: julien_jorge@yahoo.fr
*/
/**
 * \file multi_type_map.tpp
 * \brief Implementation of the claw::multi_type_map class.
 * \author Julien Jorge
 */
#include <claw/assert.hpp>

namespace claw
{
  /**
   * \brief This class provides types and methods to allow the call of methods
   *        recursively along the inherintance hierarchy.
   *
   * Suppose we have defined the following variable
   *
   * <tt>multi_type_map<int, type_list<std::string, type_list<int, no_type> > >
   * my_map;</tt>
   *
   * The call <tt>int i = my_map.get<int>( 24 );</tt> doesn't match the type
   * of the end class. So the call is repercuted on the parent class until the
   * types match. Then, we can get the values in m_data.
   *
   * \author Julien Jorge
   */
  template<typename ValueType, typename Map>
  class multi_type_map_wrapper
  {
  public:
    /**
     * \brief When \a ValueType matches Map::value_type, this class is defined
     *        as the method to call.
     */
    class last_call
    {
    public:
      const ValueType&
      get( const Map& self, const typename Map::key_type& k ) const;
      void set( Map& self, const typename Map::key_type& k,
		const ValueType& v ) const;
      bool exists( const Map& self, const typename Map::key_type& k ) const;

    }; // class last_call

    /**
     * \brief When \a ValueType doesn't match Map::value_type, this class is
     *        defined as the method to call.
     */
    class recursive_call
    {
    public:
      const ValueType&
      get( const Map& self, const typename Map::key_type& k ) const;
      void set( Map& self, const typename Map::key_type& k,
		const ValueType& v ) const;
      bool exists( const Map& self, const typename Map::key_type& k ) const;

    }; // class recursive_call
    
  public:
    /**
     * \brief Define the type of the method to call, according to the current
     *        template parameters.
     */
    typedef typename meta::if_then_else
    < meta::same_type<ValueType, typename Map::value_type>::result,
      last_call, recursive_call >::result method_type;

  }; // class multi_type_map_wrapper

} // namespace claw




/*----------------------------------------------------------------------------*/
/**
 * \brief Get a value from the map.
 * \param self The map in which we search the key.
 * \param k The key of the value to get.
 */
template<typename ValueType, typename Map>
const ValueType& claw::multi_type_map_wrapper<ValueType, Map>::last_call::get
( const Map& self, const typename Map::key_type& k ) const
{
  CLAW_PRECOND( exists(self, k) );

  return self.m_data.find(k)->second;
} // multi_type_map_wrapper::last_call::get()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a value in the map.
 * \param self The map in which we search the key.
 * \param k The key of the value to set.
 * \param v The value to set.
 */
template<typename ValueType, typename Map>
void claw::multi_type_map_wrapper<ValueType, Map>::last_call::set
( Map& self, const typename Map::key_type& k, const ValueType& v ) const
{
  self.m_data[k] = v;
} // multi_type_map_wrapper::last_call::set()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the map contains a value of a given type with a given key.
 * \param self The map in which we search the key.
 * \param k The key of the value to get.
 */
template<typename ValueType, typename Map>
bool claw::multi_type_map_wrapper<ValueType, Map>::last_call::exists
( const Map& self, const typename Map::key_type& k ) const
{
  return self.m_data.find(k) != self.m_data.end();
} // multi_type_map_wrapper::last_call::exists()




/*----------------------------------------------------------------------------*/
/**
 * \brief Get a value from the map.
 * \param self The map in which we search the key.
 * \param k The key of the value to get.
 */
template<typename ValueType, typename Map>
const ValueType&
claw::multi_type_map_wrapper<ValueType, Map>::recursive_call::get
( const Map& self, const typename Map::key_type& k ) const
{
  typename
    multi_type_map_wrapper<ValueType, typename Map::super>::method_type w;
  const typename Map::super& m = self;

  return w.get( m, k );
} // multi_type_map_wrapper::recursive_call::get()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a value in the map.
 * \param self The map in which we search the key.
 * \param k The key of the value to set.
 * \param v The value to set.
 */
template<typename ValueType, typename Map>
void claw::multi_type_map_wrapper<ValueType, Map>::recursive_call::set
( Map& self, const typename Map::key_type& k, const ValueType& v ) const
{
  typename
    multi_type_map_wrapper<ValueType, typename Map::super>::method_type w;
  typename Map::super& m = self;

  w.set( m, k, v );
} // multi_type_map_wrapper::recursive_call::set()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the map contains a value of a given type with a given key.
 * \param self The map in which we search the key.
 * \param k The key of the value to check.
 */
template<typename ValueType, typename Map>
bool claw::multi_type_map_wrapper<ValueType, Map>::recursive_call::exists
( const Map& self, const typename Map::key_type& k ) const
{
  typename
    multi_type_map_wrapper<ValueType, typename Map::super>::method_type w;
  const typename Map::super& m = self;

  return w.exists( m, k );
} // multi_type_map_wrapper::recursive_call::exists()





/*----------------------------------------------------------------------------*/
/**
 * \brief Get a value from the map.
 * \param k The key of the value to get.
 */
template<typename Key, typename TypeList>
template<typename ValueType>
const ValueType&
claw::multi_type_map<Key, TypeList>::get( const key_type& k ) const
{
  typename multi_type_map_wrapper<ValueType, self_type>::method_type w;
  return w.get( *this, k );
} // multi_type_map::get()

/*----------------------------------------------------------------------------*/
/**
 * \brief Set a value in the map.
 * \param k The key of the value to set.
 * \param v The value to set.
 */
template<typename Key, typename TypeList>
template<typename ValueType>
void claw::multi_type_map<Key, TypeList>::set
( const key_type& k, const ValueType& v )
{
  typename multi_type_map_wrapper<ValueType, self_type>::method_type w;
  w.set( *this, k, v);
} // multi_type_map::set()

/*----------------------------------------------------------------------------*/
/**
 * \brief Tell if the map contains a value of a given type with a given key.
 * \param k The key of the value to get.
 */
template<typename Key, typename TypeList>
template<typename ValueType>
bool claw::multi_type_map<Key, TypeList>::exists( const key_type& k ) const
{
  typename multi_type_map_wrapper<ValueType, self_type>::method_type w;
  return w.exists( *this, k );
} // multi_type_map::exists()
