package org.apache.commons.compress.utils;

import org.apache.commons.compress.archivers.ArchiveStreamFactory;
import org.apache.commons.compress.archivers.dump.DumpArchiveEntry;
import org.apache.commons.compress.archivers.zip.ZipLong;
import org.apache.commons.compress.compressors.FileNameUtil;
import shaded.org.apache.commons.io.FilenameUtils;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.SeekableByteChannel;
import java.nio.file.Files;
import java.nio.file.StandardOpenOption;
import java.util.*;
import java.util.regex.Pattern;

public class ZipSplitReadOnlySeekableByteChannel extends MultiReadOnlySeekableByteChannel {
    private final int ZIP_SPLIT_SIGNATURE_LENGTH = 4;
    private ByteBuffer zipSplitSignatureByteBuffer = ByteBuffer.allocate(ZIP_SPLIT_SIGNATURE_LENGTH);

    /**
     * Concatenates the given channels.
     * the channels should be add in ascending order, e.g. z01, z02, ... z99, zip
     * please note that the .zip file is the last segment and should be added as the last one in the channels
     *
     * @param channels the channels to concatenate
     * @throws NullPointerException if channels is null
     */
    public ZipSplitReadOnlySeekableByteChannel(List<SeekableByteChannel> channels) throws IOException {
        super(channels);

        // each split zip segment should begin with zip splite signature
        for(SeekableByteChannel channel : channels) {
            channel.position(0L);
            validSplitSignature(channel);
            channel.position(0L);
        }
    }

    @Override
    public synchronized int read(ByteBuffer dst) throws IOException {
        if (!isOpen()) {
            throw new ClosedChannelException();
        }
        if (!dst.hasRemaining()) {
            return 0;
        }

        SeekableByteChannel currentChannel;

        int totalBytesRead = 0;
        while (dst.hasRemaining() && currentChannelIdx < channels.size()) {
            currentChannel = channels.get(currentChannelIdx);
            // the first 4 bytes in split zip segment is the signature and need to be skipped
            if(currentChannel.position() < 4) {
                globalPosition += (4 - currentChannel.position());
                currentChannel.position(ZIP_SPLIT_SIGNATURE_LENGTH);
            }
            final int newBytesRead = currentChannel.read(dst);
            if (newBytesRead == -1) {
                // EOF for this channel -- advance to next channel idx
                currentChannelIdx += 1;
                continue;
            }
            if (currentChannel.position() >= currentChannel.size()) {
                // we are at the end of the current channel
                currentChannelIdx++;
            }
            totalBytesRead += newBytesRead;
        }
        if (totalBytesRead > 0) {
            globalPosition += totalBytesRead;
            return totalBytesRead;
        }
        return -1;
    }

    private void validSplitSignature(final SeekableByteChannel channel) throws IOException {
        // the zip split file signature is always at the beginning of the segment
        if (channel.position() > 0) {
            return;
        }

        channel.read(zipSplitSignatureByteBuffer);
        final ZipLong signature = new ZipLong(zipSplitSignatureByteBuffer.array());
        if(!signature.equals(ZipLong.DD_SIG)) {
            throw new IOException("No." + (currentChannelIdx + 1) +  " split zip file is not begin with split zip file signature");
        }
    }

    public synchronized SeekableByteChannel position(long diskNumber, long relativeOffset) throws IOException {
        long globalPosition = relativeOffset;
        for(int i = 0; i < diskNumber;i++) {
            globalPosition += channels.get(i).size();
        }

        return position(globalPosition);
    }

    /**
     * Concatenates the given channels.
     *
     * @param channels the channels to concatenate
     * @throws NullPointerException if channels is null
     * @return SeekableByteChannel that concatenates all provided channels
     */
    public static SeekableByteChannel forSeekableByteChannels(SeekableByteChannel... channels) throws IOException {
        if (Objects.requireNonNull(channels, "channels must not be null").length == 1) {
            return channels[0];
        }
        return new ZipSplitReadOnlySeekableByteChannel(Arrays.asList(channels));
    }

    public static SeekableByteChannel forSeekableByteChannels(SeekableByteChannel lastSegmentChannel, Iterable<SeekableByteChannel> channels) throws IOException {
        if(channels == null || lastSegmentChannel == null) {
            throw new IllegalArgumentException("channels must not be null");
        }

        List<SeekableByteChannel> channelsList = new ArrayList<>();
        for(SeekableByteChannel channel : channels) {

            channelsList.add(channel);
        }
        channelsList.add(lastSegmentChannel);

        if (channelsList.size() == 1) {
            return channelsList.get(0);
        }
        return new ZipSplitReadOnlySeekableByteChannel(channelsList);
    }

    public static SeekableByteChannel buildFromLastSplitSegment(File lastSegmentFile) throws IOException {
        String extension = FilenameUtils.getExtension(lastSegmentFile.getCanonicalPath());
        if(!extension.equals(ArchiveStreamFactory.ZIP)) {
            throw new IllegalArgumentException("The extension of last zip splite segment should be .zip");
        }

        File parent = lastSegmentFile.getParentFile();
        String fileBaseName = FilenameUtils.getBaseName(lastSegmentFile.getCanonicalPath());
        ArrayList<File> splitZipSegments = new ArrayList<>();

        // zip split segments should be like z01,z02....z(n-1) based on the zip specification
        String pattern = fileBaseName + ".z[0-9]+";
        for(File file : parent.listFiles()) {
            if(!Pattern.matches(pattern, file.getName())) {
                continue;
            }

            splitZipSegments.add(file);
        }

        Collections.sort(splitZipSegments, new ZipSplitSegmentComparator());
        return forFiles(lastSegmentFile, splitZipSegments);
    }

    /**
     * Concatenates the given files.
     *
     * @param files the files to concatenate
     * @throws NullPointerException if files is null
     * @throws IOException if opening a channel for one of the files fails
     * @return SeekableByteChannel that concatenates all provided files
     */
    public static SeekableByteChannel forFiles(File... files) throws IOException {
        List<SeekableByteChannel> channels = new ArrayList<>();
        for (File f : Objects.requireNonNull(files, "files must not be null")) {
            channels.add(Files.newByteChannel(f.toPath(), StandardOpenOption.READ));
        }
        if (channels.size() == 1) {
            return channels.get(0);
        }
        return new ZipSplitReadOnlySeekableByteChannel(channels);
    }

    public static SeekableByteChannel forFiles(File lastSegmentFile, Iterable<File> files) throws IOException {
        if(files == null || lastSegmentFile == null) {
            throw new IllegalArgumentException("files must not be null");
        }

        List<SeekableByteChannel> channelsList = new ArrayList<>();
        for (File f : files) {
            channelsList.add(Files.newByteChannel(f.toPath(), StandardOpenOption.READ));
        }
        channelsList.add(Files.newByteChannel(lastSegmentFile.toPath(), StandardOpenOption.READ));

        if (channelsList.size() == 1) {
            return channelsList.get(0);
        }
        return new ZipSplitReadOnlySeekableByteChannel(channelsList);
    }
}
