/***************************************************************************
                          cquerymanager.cpp  -  description
                             -------------------
    begin                : Sat Jun 8 2002
    copyright            : (C) 2002-2004 by Mathias Kster
    email                : mathen@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <stdio.h>

#ifndef WIN32
#include <unistd.h>
#endif

#include <dclib/dcos.h>
#include <dclib/core/cdir.h>
#include <dclib/core/cstringlist.h>
#include <dclib/cconnectionmanager.h>
#include <dclib/cdownloadqueue.h>
#include <dclib/cdownloadmanager.h>
#include <dclib/cfilemanager.h>
#include <dclib/cconfig.h>
#include <dclib/core/cmanager.h>
#include <dclib/csearchindex.h>

#include "cquerymanager.h"

#define MAX_SEARCH_RESULTS 		10
#define MAX_QUERY_QUEUE_COUNT		25
#define QUERY_TIMEOUT			10
#define MIN_QUERY_STRING_LENGTH		3

/** */
CQueryManager::CQueryManager()
{
	m_pQueryQueue    = new CThreadList<CQueryObject>;
	m_pQuerySendList = new CThreadList<CQuerySendObject>;

	m_nSearchCountActive  = 0;
	m_nSearchCountPassive = 0;
	m_nSearchCountReject  = 0;
	m_nSearchCountError   = 0;
	m_nResultCount        = 0;
	m_nResultCountError   = 0;

	m_pCallback = new CCallback<CQueryManager>( this, &CQueryManager::Callback );
	CManager::Instance()->Add( m_pCallback );

	SetInstance(this);
}

/** */
CQueryManager::~CQueryManager()
{
	SetInstance(0);

	CManager::Instance()->Remove( m_pCallback );

	if ( m_pCallback )
	{
		delete m_pCallback;
		m_pCallback = 0;
	}
	
	delete m_pQueryQueue;
	delete m_pQuerySendList;
}

/** */
bool CQueryManager::CheckSize( CQueryObject * queryobject, struct filebaseobject * fbo )
{
	bool res = TRUE;

	// check file size
	if ( queryobject->pMessageSearch->m_bSizeLimit == TRUE )
	{
		if ( queryobject->pMessageSearch->m_eSizeType == esstATMOST )
		{
			if ( fbo->m_nSize > queryobject->pMessageSearch->m_nSize )
			{
				res = FALSE;
			}
		}
		else
		{
			if ( fbo->m_nSize < queryobject->pMessageSearch->m_nSize )
			{
				res = FALSE;
			}
		}
	}

	return res;
}

/** */
bool CQueryManager::CheckType( CQueryObject * queryobject, struct filebaseobject * fbo )
{
	bool res = TRUE;

	// check file type
	switch(queryobject->pMessageSearch->m_eFileType)
	{
		case eftALL:
			break;
		case eftMP3:
		case eftARCHIVE:
		case eftDOCUMENT:
		case eftAPPLICATION:
		case eftPICTURE:
		case eftVIDEO:
			if ( queryobject->pMessageSearch->m_eFileType != eFileTypes(fbo->m_eFileType) )
			{
				res = FALSE;
			}
			break;
		case eftFOLDER:
			res = FALSE;
			break;
		case eftHASH:
			if ( queryobject->pMessageSearch->m_sString.Left(4) == "TTH:" )
				res = TRUE;
			else
				res = FALSE;
			break;
		default:
			res = FALSE;
			break;
	}

	return res;
}

/** */
bool CQueryManager::AddResult( CQuerySendObject * querysendobject, CQueryObject * queryobject, 
			struct filebaseobject * fbo, CString filename, CString hash )
{
	CString s, result;
	int slot_max,slot_free;

	// get slot info
	slot_max  = CConfig::Instance()->GetMaxUpload();
	slot_free = CDownloadManager::Instance()->DLM_UsedSlots();

	// unlimit slots
	if ( slot_max == 0 )
	{
		slot_max = 99+slot_free;
	}

	// check for extra slots
	if ( slot_free > slot_max )
	{
		slot_free = 0;
	}
	else
	{
		slot_free = slot_max-slot_free;
	}

	// create the searchresult
	s += filename;
	s  = s.Replace('/',"\\");
	result  = "$SR ";
	result += queryobject->sNick + " ";
	result += s + "\x5";
	result += CString().setNum(fbo->m_nSize) + " ";
	result += CString().setNum(slot_free) + "/" + CString().setNum(slot_max);
	result += "\x5";
	
	// add hash or hubname to searchresult
	if ( hash != "" )
		result += hash;
	else
		result += queryobject->sHubName;
	result += " ";

	result += "(" + queryobject->sHubHost + ")";

	// result found ! send it ...
	if ( queryobject->pMessageSearch->m_bLocal == TRUE )
	{
		result += "\x5" + queryobject->pMessageSearch->m_sSource + "|";
		querysendobject->m_pList->Add("r", new CString(result));
//		printf("ADD L RESULT: '%s'\n",result.Data());
	}
	else
	{
		querysendobject->m_pList->Add("r", new CString(result));
//		printf("ADD G RESULT: '%s'\n",result.Data());
	}

	return TRUE;
}

/** */
void CQueryManager::SendResults()
{
	CString * s;
	CQuerySendObject * querysendobject = 0;

	while ( (querysendobject=m_pQuerySendList->Next(querysendobject)) != 0 )
	{
		if ( querysendobject->m_pSocket )
		{
			eConnectState e;

			e = querysendobject->m_pSocket->Connect( querysendobject->m_sSource, querysendobject->m_nPort, TRUE );

			if ( e == ecsSUCCESS )
			{
				s = 0;

				while ( querysendobject->m_pList->Next( (CObject*&) s) == 1 )
				{
					if ( querysendobject->m_pSocket->Write( (const unsigned char*)s->Data(), s->Length(), 2 ) <= 0 )
					{
						m_nResultCountError++;
						
						break;
					}
					else
					{
						m_nResultCount++;
					}
				}

				querysendobject->m_pSocket->Disconnect();
				m_pQuerySendList->Del(querysendobject);
			}
			else if ( e == ecsERROR )
			{
				m_pQuerySendList->Del(querysendobject);
				
				m_nResultCountError++;
			}

			break;
		}
		else
		{
			s = 0;

			while ( querysendobject->m_pList->Next( (CObject*&) s) == 1 )
			{
				if ( CConnectionManager::Instance()->SendStringToConnectedServers(*s,querysendobject->m_sSource) == 0 )
				{
					m_nResultCountError++;
					
					break;
				}
				else
				{
					m_nResultCount++;
				}
			}

			m_pQuerySendList->Del(querysendobject);
			break;
		}
	}
}

/** */
void CQueryManager::GetResults( CString s, CStringList * resultlist )
{
	CString *id;
	CStringList * sl;
	CQueryResultObject * QueryResultObject;

	if ( !CFileManager::Instance() )
	{
		return;
	}

	if ( s.Left(4) == "TTH:" )
	{
		sl = CFileManager::Instance()->SearchHash(s);
	}
	else
	{
		sl = CFileManager::Instance()->Search(s);
	}
	
	if ( sl != 0 )
	{
		id = 0;

		while( sl->Next( (CObject*&)id ) != 0 )
		{
			if ( resultlist->Get( *id, (CObject*&)QueryResultObject ) != 0 )
			{
				// add new result
				QueryResultObject         = new CQueryResultObject();
				QueryResultObject->ID     = *id;
				QueryResultObject->iDepth = 1;

				resultlist->Add( *id, QueryResultObject );
			}
			else
			{
				// result allready exist
				QueryResultObject->iDepth++;
			}
		}

		delete sl;
	}
}

/** */
void CQueryManager::HandleQuery( CQueryObject * queryobject )
{
	CString search;
	CString *id, filename;
	CStringList resultlist;
	struct filebaseobject FileBaseObject;
	CQueryResultObject * QueryResultObject;
	CQuerySendObject * querysendobject;
	CString s,s1;
	long i;
	bool dummy = FALSE;
	int depth;
	int resultcount;

	QueryResultObject = 0;
	
	// get searchstring
	search = queryobject->pMessageSearch->m_sString.ToUpper();

	// sanity check
	if ( search == "" )
	{
		return;
	}

	// check for dummy search
	if ( search == "." )
	{
		dummy = TRUE;
	}

	depth = 0;

	if ( !dummy )
	{
		// search for the correct filename
		GetResults( search, &resultlist );

		// if file not found search for substrings
		if ( resultlist.Count() == 0 )
		{
			// convert special chars
			//search = search.Replace(' '," ");
			search = search.Replace('.'," ");
			search = search.Replace('-'," ");
			search = search.Replace('_'," ");
			search = search.Replace('('," ");
			search = search.Replace(')'," ");
			search = search.Replace('!'," ");

			s = search + " ";

			while( (i=s.Find(' ')) != -1 )
			{
				// get substring
				s1 = s.Left(i);
				// remove substring from string
				s  = s.Mid(i+1,s.Length()-i-1);

				i++;

				// min searchstring length is 3
				if ( s1.Length() < MIN_QUERY_STRING_LENGTH )
				{
					continue;
				}

				depth++;

				// get results
				GetResults( s1, &resultlist );
			}
		}
	}

	// printf("querymanager result for '%ld'\n",resultlist.Count());

	if ( (resultlist.Count() > 0) || (dummy == TRUE) )
	{
		// results found ...

		id = 0;
		resultcount = 0;

		// create resultobject
		querysendobject = new CQuerySendObject();

		// init passive or active search result
		if ( queryobject->pMessageSearch->m_bLocal == FALSE )
		{
			querysendobject->m_pSocket = new CSocket(estUDP);
			querysendobject->m_sSource = queryobject->pMessageSearch->m_sSource;
			querysendobject->m_nPort   = queryobject->pMessageSearch->m_nPort;
		}
		else
		{
			querysendobject->m_sSource = queryobject->sHubName;
		}

		// real query
		if ( dummy == FALSE )
		{
			while( (resultlist.Next( (CObject*&)QueryResultObject ) != 0) && (resultcount < MAX_SEARCH_RESULTS) )
			{
				// check for a good result
				// iDepth are the count of same results
				// depth are the count of searched substrings
				if ( queryobject->pMessageSearch->m_eFileType != eftHASH )
				{
					if ( QueryResultObject->iDepth < ((depth/2)+1) )
					{
						continue;
					}
				}

				// get the fileobject for this resultid
				if ( CFileManager::Instance()->GetFileBaseObject( QueryResultObject->ID, &FileBaseObject, filename ) == TRUE )
				{
					// check if filesize correct
					if ( CheckSize( queryobject, &FileBaseObject ) == FALSE )
					{
						continue;
					}

					// check if filetype correct
					if ( CheckType( queryobject, &FileBaseObject ) == FALSE )
					{
						continue;
					}

					CString hash;
					
					hash = CFileManager::Instance()->GetHash(FileBaseObject.m_nHashIndex);
					
					// add result
					if ( AddResult( querysendobject, queryobject, &FileBaseObject, filename, hash ) == FALSE )
					{
						break;
					}

					resultcount++;
				}
			}
		}
		else
		{
			// add dummy result
			struct filebaseobject fbo;
			fbo.m_nSize = 0;
			AddResult( querysendobject, queryobject, &fbo, "." );
		}

		if ( querysendobject->m_pList->Count() > 0 )
		{
			m_pQuerySendList->Add(querysendobject);
		}
		else
		{
			delete querysendobject;
		}
		
		resultlist.Clear();
	}
}

/** */
int CQueryManager::Callback( CObject *, CObject * )
{
	CQueryObject * queryobject;

	m_pQueryQueue->Lock();

	// get searches from the queue
	while ( (queryobject = m_pQueryQueue->Next(0)) != 0 )
	{
		// remove it from the list
		m_pQueryQueue->Remove(queryobject);

		// query timeout in seconds
		if ( (time(0)-queryobject->tTimeout) > QUERY_TIMEOUT )
		{
			// remove it
			delete queryobject;
			queryobject = 0;
			// reject statistic
			m_nSearchCountReject++;
		}
		else
		{
			break;
		}
	}

	m_pQueryQueue->UnLock();

	// sanity check / object not rejected
	if (queryobject)
	{
		// handle query
		HandleQuery(queryobject);
		// remove object
		delete queryobject;
	}

	// send any results from the result queue
	SendResults();

	return 0;
}

/** store all searches from the hub in a queue */
bool CQueryManager::SearchQuery( CString hubname, CString hubhost, CString nick, CMessageSearch * msg )
{
	bool res = FALSE;
	CQueryObject * queryobject;

	// sanity check
	if ( msg == 0)
	{
		return res;
	}

	m_pQueryQueue->Lock();

	// active/passive search statistic
	if ( msg->m_bLocal == FALSE )
	{
		m_nSearchCountActive++;
	}
	else
	{
		m_nSearchCountPassive++;
	}

	// queue overflow ?
	if ( m_pQueryQueue->Count() < MAX_QUERY_QUEUE_COUNT )
	{
		// sanity check
		if ( (hubname != "") && (hubhost != "") )
		{
			// create query object and add it to the queue
			queryobject = new CQueryObject();

			queryobject->sHubName = hubname;
			queryobject->sHubHost = hubhost;
			queryobject->sNick    = nick;
			queryobject->tTimeout = time(0);
			queryobject->pMessageSearch = new CMessageSearch();
			*queryobject->pMessageSearch = *msg;
			m_pQueryQueue->Add(queryobject);

			res = TRUE;
		}
		else
		{
			// error statistic
			m_nSearchCountError++;
		}
	}
	else
	{
		// reject statistic
		m_nSearchCountReject++;
	}

	m_pQueryQueue->UnLock();

	return res;
}

/** query mananger stats */
ulonglong CQueryManager::GetStat( enum eSearchStat e )
{
	ulonglong res = 0;
	
	switch(e)
	{
		case essCOUNTACTIVE:
			res = m_nSearchCountActive;
			break;
		case essCOUNTPASSIVE:
			res = m_nSearchCountPassive;
			break;
		case essCOUNTREJECT:
			res = m_nSearchCountReject;
			break;
		case essCOUNTERROR:
			res = m_nSearchCountError;
			break;
		case essRESULTCOUNT:
			res = m_nResultCount;
			break;
		case essRESULTCOUNTERROR:
			res = m_nResultCountError;
			break;
		default:
			break;
	}
	
	return res;
}
