/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/*
 * This file is part of the libepubgen project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 */

#include <cassert>
#include <iomanip>
#include <sstream>

#include "EPUBHTMLManager.h"
#include "EPUBManifest.h"

namespace libepubgen
{

EPUBHTMLManager::EPUBHTMLManager(EPUBManifest &manifest)
  : m_manifest(manifest)
  , m_paths()
  , m_contents()
  , m_ids()
  , m_number()
{
}

const EPUBHTMLGeneratorPtr_t EPUBHTMLManager::create(EPUBImageManager &imageManager, EPUBListStyleManager &listStyleManager, EPUBParagraphStyleManager &paragraphStyleManager, EPUBSpanStyleManager &spanStyleManager, EPUBTableStyleManager &tableStyleManager, const EPUBPath &stylesheetPath)
{
  std::ostringstream nameBuf;
  nameBuf << "section" << std::setw(4) << std::setfill('0') << m_number.next();
  m_ids.push_back(nameBuf.str());

  nameBuf << ".html";
  m_paths.push_back(EPUBPath("OEBPS/sections") / nameBuf.str());

  m_manifest.insert(m_paths.back(), "application/xhtml+xml", m_ids.back());

  m_contents.push_back(EPUBXMLSink());

  const EPUBHTMLGeneratorPtr_t gen(
    new EPUBHTMLGenerator(m_contents.back(), imageManager, listStyleManager, paragraphStyleManager, spanStyleManager, tableStyleManager, m_paths.back(), stylesheetPath));

  return gen;
}

void EPUBHTMLManager::writeTo(EPUBPackage &package)
{
  assert(m_contents.size() == m_paths.size());

  std::vector<EPUBPath>::const_iterator pathIt = m_paths.begin();
  std::vector<EPUBXMLSink>::iterator contentIt = m_contents.begin();

  for (; (m_paths.end() != pathIt) && (m_contents.end() != contentIt); ++pathIt, ++contentIt)
    contentIt->writeTo(package, pathIt->str().c_str());
}

void EPUBHTMLManager::writeSpineTo(EPUBXMLSink &sink)
{
  for (std::vector<std::string>::const_iterator it = m_ids.begin(); m_ids.end() != it; ++it)
  {
    librevenge::RVNGPropertyList itemrefAttrs;
    itemrefAttrs.insert("idref", it->c_str());
    sink.insertEmptyElement("itemref", itemrefAttrs);
  }
}

void EPUBHTMLManager::writeTocTo(EPUBXMLSink &sink, const EPUBPath &tocPath)
{
  librevenge::RVNGPropertyList navPointAttrs;
  for (std::vector<EPUBPath>::size_type i = 0; m_paths.size() != i; ++i)
  {
    std::ostringstream id;
    id << "section" << (i + 1);
    navPointAttrs.insert("id", id.str().c_str());
    navPointAttrs.insert("class", "document");
    navPointAttrs.insert("playOrder", "");
    sink.openElement("navPoint", navPointAttrs);
    sink.openElement("navLabel");
    sink.openElement("text");
    std::ostringstream label;
    label << "Section " << (i + 1);
    sink.insertCharacters(label.str().c_str());
    sink.closeElement("text");
    sink.closeElement("navLabel");
    librevenge::RVNGPropertyList contentAttrs;
    contentAttrs.insert("src", m_paths[i].relativeTo(tocPath).str().c_str());
    sink.insertEmptyElement("content", contentAttrs);
    sink.closeElement("navPoint");
  }
}

}

/* vim:set shiftwidth=2 softtabstop=2 expandtab: */
