/* secure.c  -  memory allocation from a secure heap
 * Copyright (C) 1998,1999 Free Software Foundation, Inc.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

/* This file was lifted from the GNUPG source and slightly modified by
 * Hans Petter Jansson (hpj@styx.net) to work better with the rest of the
 * library and API.
 */


#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <stdarg.h>
#include <unistd.h>
#if defined(HAVE_MLOCK) || defined(HAVE_MMAP)
#  include <sys/mman.h>
#  include <sys/types.h>
#  include <fcntl.h>
#endif


#include "types.h"

#if defined(MAP_ANON) && !defined(MAP_ANONYMOUS)
#  define MAP_ANONYMOUS MAP_ANON
#endif

#define DEFAULT_POOLSIZE 8196

typedef struct memblock_struct MEMBLOCK;
struct memblock_struct
{
  unsigned size;
  union
  {
    MEMBLOCK *next;
    PROPERLY_ALIGNED_TYPE aligned;
  } u;
};


void  *pool;
int   mem_pool_okay;
int   mem_pool_is_mmapped;
size_t mem_pool_size; /* allocated length */
size_t mem_pool_len;  /* used length */
MEMBLOCK *mem_blocks_unused;
unsigned mem_alloced_max;
unsigned mem_alloced_cur;
unsigned mem_blocks_max;
unsigned mem_blocks_cur;
int mem_disable_security;
int mem_warning_show;
int mem_warning_none;
int mem_warning_suspend;

void mem_sec_free(void *a);


void mem_warning_print(void)
{
  if (!mem_warning_none) /* log_info(_("Warning: using insecure memory!\n")) */ ;
}


void mem_pool_lock(void *p, size_t n)
{
#ifdef HAVE_MLOCK
  uid_t uid;
  int err;

  uid = getuid();

#  ifdef HAVE_BROKEN_MLOCK
  if (uid)
  {
    errno = EPERM;
    err = errno;
  }
  else
  {
    err = mlock(p, n);
    if (err && errno) err = errno;
  }
#  else
  err = mlock(p, n);
  if (err && errno) err = errno;
#  endif

  if( uid && !geteuid() )
  {
    if (setuid( uid ) || getuid() != geteuid())
      /* log_fatal("failed to reset uid: %s\n", strerror(errno)) */ ;
  }

  if (err)
  {
    if(errno != EPERM
#  ifdef EAGAIN  /* OpenBSD returns this */
       && errno != EAGAIN
#  endif
       )
    /* log_error("cant lock memory: %s\n", strerror(err)) */ ;
    mem_warning_show = 1;
  }

#else
  /* log_info("Please note that you don't have secure memory on this system\n"); */
#endif
}


void mem_pool_init(size_t n)
{
  size_t pgsize;
  
  mem_pool_size = n;
  
  if (mem_disable_security) /* log_bug("secure memory is disabled") */ ;

#ifdef HAVE_GETPAGESIZE
  pgsize = getpagesize();
#else
  pgsize = 4096;
#endif

#if HAVE_MMAP
  mem_pool_size = (mem_pool_size + pgsize -1 ) & ~(pgsize-1);
#  ifdef MAP_ANONYMOUS
  pool = mmap(0, mem_pool_size, PROT_READ | PROT_WRITE,
              MAP_PRIVATE | MAP_ANONYMOUS, -1, 0);
#  else /* map /dev/zero instead */
  {
    int fd;
    
    fd = open("/dev/zero", O_RDWR);
    if (fd == -1) 
    {
      /* log_error("can't open /dev/zero: %s\n", strerror(errno)); */
      pool = (void*) -1;
    }
    else
    {
      pool = mmap(0, mem_pool_size, PROT_READ|PROT_WRITE,
                  MAP_PRIVATE, fd, 0);
    }
  }
#  endif
  if (pool == (void*) -1)
    /* log_info("can't mmap pool of %u bytes: %s - using malloc\n",
             (unsigned)mem_pool_size, strerror(errno)) */ ;
  else
  {
    mem_pool_is_mmapped = 1;
    mem_pool_okay = 1;
  }

#endif
  if (!mem_pool_okay)
  {
    pool = malloc(mem_pool_size);
    if(!pool) /* log_fatal("can't allocate memory pool of %u bytes\n",
                        (unsigned) mem_pool_size) */ ;
    else
      mem_pool_okay = 1;
  }
  mem_pool_lock( pool, mem_pool_size );
  mem_pool_len = 0;
}


/* concatenate unused blocks */
void mem_sec_pool_compress(void)
{

}

void mem_sec_flags_set(unsigned flags)
{
  int was_susp = mem_warning_suspend;

  mem_warning_none = flags & 1;
  mem_warning_suspend = flags & 2;

  /* and now issue the warning if it is not longer suspended */
  if (was_susp && !mem_warning_suspend && mem_warning_show)
  {
    mem_warning_show = 0;
    mem_warning_print();
  }
}

unsigned mem_sec_flags_get(void)
{
  unsigned flags;

  flags  = mem_warning_none ? 1:0;
  flags |= mem_warning_suspend ? 2:0;
  return flags;
}

void mem_sec_init( size_t n )
{
  if (!n)
  {
#ifndef HAVE_DOSISH_SYSTEM
    uid_t uid;

    mem_disable_security=1;
    uid = getuid();
    if (uid != geteuid())
    {
      if (setuid(uid) || getuid() != geteuid())
        /* log_fatal("failed to drop setuid\n" ) */ ;
    }
#endif
  }
  else
  {
    if (n < DEFAULT_POOLSIZE) n = DEFAULT_POOLSIZE;
    if (!mem_pool_okay) mem_pool_init(n);
    else /* log_error("Oops, secure memory pool already initialized\n") */ ;
  }
}


void *mem_sec_alloc(size_t size)
{
  MEMBLOCK *mb, *mb2;
  int compressed = 0;

  if (!mem_pool_okay)
  {
    /* log_info(_("operation is not possible without initialized secure memory\n"));
    log_info(_("(you may have used the wrong program for this task)\n")); */
    exit(2);
  }

  if (mem_warning_show && !mem_warning_suspend)
  {
    mem_warning_show = 0;
    mem_warning_print();
  }

  /* blocks are always a multiple of 32 */
  size += sizeof(MEMBLOCK);
  size = ((size + 31) / 32) * 32;

  retry:
  
  /* try to get it from the used blocks */
  for (mb = mem_blocks_unused,mb2=NULL; mb; mb2=mb, mb = mb->u.next)
    if (mb->size >= size)
    {
      if (mb2) mb2->u.next = mb->u.next;
      else mem_blocks_unused = mb->u.next;
      goto leave;
    }
  
  /* allocate a new block */
  if ((mem_pool_len + size <= mem_pool_size)) 
  {
    mb = (void*) ((char*) pool + mem_pool_len);
    mem_pool_len += size;
    mb->size = size;
  }
  else if (!compressed)
  {
    compressed=1;
    mem_sec_pool_compress();
    goto retry;
  }
  else return NULL;

  leave:
  
  mem_alloced_cur += mb->size;
  mem_blocks_cur++;
  if (mem_alloced_cur > mem_alloced_max) mem_alloced_max = mem_alloced_cur;
  if (mem_blocks_cur > mem_blocks_max) mem_blocks_max = mem_blocks_cur;
  return &mb->u.aligned.c;
}


void *mem_sec_realloc(void *p, size_t newsize)
{
  MEMBLOCK *mb;
  size_t size;
  void *a;

  mb = (MEMBLOCK*)((char*)p - ((size_t) &((MEMBLOCK*)0)->u.aligned.c));
  size = mb->size;
  if (newsize < size) return p; /* it is easier not to shrink the memory */
  a = mem_sec_alloc( newsize );
  memcpy(a, p, size);
  memset((char*)a+size, 0, newsize-size);
  mem_sec_free(p);
  return a;
}


void mem_sec_free(void *a)
{
  MEMBLOCK *mb;
  size_t size;

  if (!a) return;

  mb = (MEMBLOCK*)((char*)a - ((size_t) &((MEMBLOCK*)0)->u.aligned.c));
  size = mb->size;
  memset(mb, 0xff, size );
  memset(mb, 0xaa, size );
  memset(mb, 0x55, size );
  memset(mb, 0x00, size );
  mb->size = size;
  mb->u.next = mem_blocks_unused;
  mem_blocks_unused = mb;
  mem_blocks_cur--;
  mem_alloced_cur -= size;
}

int m_is_secure(const void *p)
{
  return p >= pool && p < (void*)((char*)pool+mem_pool_size);
}

void mem_sec_term()
{
  if(!mem_pool_okay) return;

  memset( pool, 0xff, mem_pool_size);
  memset( pool, 0xaa, mem_pool_size);
  memset( pool, 0x55, mem_pool_size);
  memset( pool, 0x00, mem_pool_size);
#if HAVE_MMAP
  if (mem_pool_is_mmapped) munmap(pool, mem_pool_size);
#endif
  pool = NULL;
  mem_pool_okay = 0;
  mem_pool_size=0;
  mem_pool_len=0;
  mem_blocks_unused=NULL;
}


void mem_sec_stats_dump()
{
  if (mem_disable_security) return;

  fprintf(stderr, "secmem usage: %u/%u bytes in %u/%u blocks of pool %lu/%lu\n",
          mem_alloced_cur, mem_alloced_max, mem_blocks_cur, mem_blocks_max,
          (ulong) mem_pool_len, (ulong) mem_pool_size);
}

