/* Copyright (C) 1999 Hans Petter K. Jansson
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * You can contact the library's author by sending e-mail to <hpj@styx.net>.
 */

#include "config.h"
#include <stdlib.h>
#include "tn.h"


/* Allocates a new node structure, initialized with data of size, or
 * nothing if either data or size is 0. */

TN *tn_add(unsigned char *data, unsigned long size)
{
  TN *n;
  
  n = malloc(sizeof(TN));
  memset(n, 0, sizeof(TN));

  _tn_edges_step_set_in(n, 0);
  _tn_edges_step_set_out(n, 0);

  n->data = malloc(size + 1);
  memcpy(n->data, data, size);
  n->size = size;
  
  return(n);
}

/* Frees a node structure, paying no heed to net integrity. */

void tn_del_fast(TN *n)
{
  free(n->edge_in);
  free(n->edge_out);
  free(n->data);
  free(n);
}

/* Frees a node structure. Returns 1 if successful, or 0 if there were
 * still edges pointing to/from it. */

int tn_del_try(TN *n)
{
  if (n->edges_in_num || n->edges_out_num) return(0);
  
  tn_del_fast(n);
  return(1);
}

/* Always frees a node structure, disassociating it first if necessary. */

void tn_del(TN *n)
{
  long i;
  TN **edge;

  for (i = n->edges_in_num - 1, edge = n->edge_in; i > -1; i--)
    if (edge[i]) tn_edge_out_del(edge[i], n);

  for (i = n->edges_out_num - 1, edge = n->edge_out; i > -1; i--)
    if (edge[i]) tn_edge_in_del(edge[i], n);

  tn_del_fast(n);
}

/* Deletes all nodes (like tn_del) separated from source node by at least
 * dist_min edges and at most dist_max edges (see description of
 * tn_distance). Returns number of nodes freed.
 * 
 * Useful for punching holes in a network, separating out a sub-network or
 * limiting the extent of a network (removing all distant nodes). */

int tn_del_distance(TN *n, unsigned short dist_min, unsigned short dist_max, int dir)
{
  return(_tn_del_distance_r(n, dist_min, dist_max, dir, 0));
}

int _tn_del_distance_r(TN *n, unsigned short dist_min, unsigned short dist_max, int dir, unsigned short dist)
{
  unsigned short dist_next = dist + 1;
  unsigned int freed = 0;
  long i;

  tn_trav_set(n);

  if (dist < dist_max)
  {
    /* Mark neighbors as seen at this distance */

    if (!dir || dir == 1)
    {
      for (i = n->edges_out_num - 1; i > -1; i--)
        if (!tn_seen(n->edge_out[i])) tn_seen_set(n->edge_out[i], dist_next);
    }
    
    if (!dir || dir == -1)
    {
      for (i = n->edges_in_num - 1; i > -1; i--)
        if (!tn_seen(n->edge_in[i])) tn_seen_set(n->edge_in[i], dist_next);
    }

    /* Traverse my seen neighbors */
    
    tn_nopk_set(n);

    if (!dir || dir == 1)
    {
      for (i = n->edges_out_num - 1; i > -1; i--)
        if (tn_seen_dist(n->edge_out[i], dist_next) && !tn_trav(n->edge_out[i]))
          freed +=_tn_del_distance_r(n->edge_out[i], dist_min, dist_max, dir, dist_next);
    }

    if (!dir || dir == -1)
    {
      for (i = n->edges_in_num - 1; i > -1; i--)
        if (tn_seen_dist(n->edge_in[i], dist_next) && !tn_trav(n->edge_in[i]))
          freed +=_tn_del_distance_r(n->edge_in[i], dist_min, dist_max, dir, dist_next);
    }

    tn_nopk_clear(n);
    tn_edges_pack(n);
  }

  if (dist >= dist_min && dist <= dist_max) { tn_del(n); freed++; }
  else { tn_seen_clear(n); tn_trav_clear(n); }

  return(freed);
}

/* Checks if neighbors are already seen, and those that are not seen, are
 * marked as such. */

unsigned long tn_seen_neighbors(TN *n, int dir)
{
  long i;
  unsigned long seen = 0;

  if (!dir || dir == 1)
    for (i = n->edges_out_num - 1; i > -1; i--)
    {
      if (tn_seen(n->edge_out[i])) seen++;
      else n->edge_out[i]->flags |= TN_SEEN;
    }

  if (!dir || dir == -1)
    for (i = n->edges_in_num - 1; i > -1; i--)
    {
      if (tn_seen(n->edge_in[i])) seen++;
      else n->edge_in[i]->flags |= TN_SEEN;
    }

  return(seen);
}
