/* Copyright (C) 1999 Hans Petter K. Jansson
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * You can contact the library's author by sending e-mail to <hpj@styx.net>.
 */

#include "config.h"
#include "types.h"
#include "log.h"
#include "rmd160.h"
#include "tt.h"
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>


TT *tt_split(TT *tt, u32 pos)
{
  TT *tt_n;
  unsigned int len, len_first, len_last;

  len = tt_size(tt);

  if (pos > len) return(0);
  if (!tt_data_is_internal(tt)) return(0);  /* FIXME: Need to handle external data too */

  len_first = pos;
  len_last = len - pos;
  
  tt_n = tt_new();
  tt_add_after(tt, tt_n);
  
  if (tt->data && len_last)
  {
    tt_data_set_internal(tt_n, tt->data + len_first, len_last, TRUE);

    if (len_first) tt->data = realloc(tt->data, len_first);
    else
    {
      free(tt->data);
      tt->data = 0;
    }
    
    tt->size = len_first;
  }

  return(tt_n);
}


void tt_data_swap(TT *tt0, TT *tt1)
{
  void *data;
  short handle;
  u16 flag;
  
  data = tt0->data;
  tt0->data = tt1->data;
  tt1->data = data;

  handle = tt0->handle;
  tt0->handle = tt1->handle;
  tt1->handle = handle;

  flag = tt0->data_is_internal;
  tt0->data_is_internal = tt1->data_is_internal;
  tt1->data_is_internal = flag;

  flag = tt0->data_is_local;
  tt0->data_is_local = tt1->data_is_local;
  tt1->data_is_local = flag;

  flag = tt0->data_is_ready;
  tt0->data_is_ready = tt1->data_is_ready;
  tt1->data_is_ready = flag;
}


void tt_data_del(TT *tt)
{
  if (tt->handle >= 0)
  {
    close(tt->handle);
    tt->handle = -1;
  }

  if (tt_has_data(tt))
  {
    /* If data is local, we can free/remove stuff */
  
    if (tt_data_is_local(tt))
    {
      if (!tt_data_is_internal(tt)) remove(tt->data);
      free(tt->data);
    }
    else if (!tt_data_is_internal(tt)) free(tt->data);  /* Resource handle */

    /* In any case, we zero the pointers */
    
    tt->data = 0;
    tt->size = 0;
  }
}


void tt_data_set_internal(TT *tt, void *src, u32 len, unsigned int copy)
{
  tt_data_del(tt);

  if (copy)
  {
    tt->data = malloc(len);
    memcpy(tt->data, src, len);
    tt->data_is_local = TRUE;
  }
  else
  {
    tt->data = src;
    tt->data_is_local = FALSE;
  }
  
  tt->size = len;
  tt->data_is_internal = TRUE;
}


/* TODO: Relocate in code listing */

int tt_get_external_handle(TT *tt)
{
  if (!tt_has_data(tt)) return(-1);

  if (tt->handle >= 0) return(tt->handle);

  tt->handle = open(tt->data, O_RDWR | O_CREAT);
  return(tt->handle);
}


void tt_data_set_bytes(TT *tt, void *src, u32 start, u32 len)
{
  if (tt_has_data(tt))
  {
    if (tt_data_is_internal(tt))
    {
      if (tt_size(tt) < start + len)
      {
        tt->data = realloc(tt->data, start + len);
        tt->size = start + len;
      }

      memcpy(tt->data + start, src, len);
    }
    else  /* !tt_data_is_internal(tt) */
    {
      int handle;
      off_t offset;
      byte null = 0;
      
      handle = tt_get_external_handle(tt);
      if (handle >= 0)
      {
        offset = lseek(handle, start, SEEK_SET);

        /* Pad with zero */
        for (; offset < start; offset++) write(handle, &null, 1);

        write(handle, src, len);
        tt->handle = -1;
        close(handle);  /* TODO: Make more intelligent system for maintaining handles */
      }
      else
      {
        /* Out of filehandles? */
        
        log_put_abort("Unable to get handle for external data.");
      }
    }
  }
  else
  {
    /* TODO: Large datasets should be allocated externally */

    tt->size = start + len;
    tt->data = malloc(tt->size);
    memset(tt->data, 0, start);

    tt->data_is_local = TRUE;
    tt->data_is_internal = TRUE;
  }
}


void tt_data_append_bytes(TT *tt, void *src, u32 len)
{
  if (tt_has_data(tt))
  {
    if (tt_data_is_internal(tt))
    {
      tt->data = realloc(tt->data, tt->size + len);
      memcpy(tt->data + tt->size, src, len);
      tt->size += len;
    }
    else
    {
      int handle;
      
      handle = tt_get_external_handle(tt);
      if (handle >= 0)
      {
        lseek(handle, 0, SEEK_END);
        write(handle, src, len);
        tt->handle = -1;
        close(handle);  /* TODO: Make more intelligent system for maintaining handles */
        tt->size += len;
      }
      else
      {
        /* Out of filehandles? */
        
        log_put_abort("Unable to get handle for external data.");
      }
    }
  }
  else tt_data_set_internal(tt, src, len, TRUE);
}


/* TODO: Add error handling to this function */
/* TODO: This should be part of a more generic file library */

void tt_data_external_offset(int handle, off_t old_start, off_t offset)
{
  off_t old_end, new_start, new_end;
  long count;
  byte *block;

  
  block = malloc(TT_GENERIC_BLOCK_SIZE);

  old_end = lseek(handle, 0, SEEK_END);
  new_start = old_start + offset;
  new_end = old_end + offset;
  
  /* Pad end of file */
  
  for (count = offset; count >= TT_GENERIC_BLOCK_SIZE; count -= TT_GENERIC_BLOCK_SIZE)
    write(handle, block, TT_GENERIC_BLOCK_SIZE);

  if (count) write(handle, block, count);

  /* Push blocks */
  
  for (count = old_end - old_start; count >= TT_GENERIC_BLOCK_SIZE; count -= TT_GENERIC_BLOCK_SIZE)
  {
    lseek(handle, old_start + count - TT_GENERIC_BLOCK_SIZE, SEEK_SET);
    read(handle, block, TT_GENERIC_BLOCK_SIZE);
    lseek(handle, new_start + count - TT_GENERIC_BLOCK_SIZE, SEEK_SET);
    write(handle, block, TT_GENERIC_BLOCK_SIZE);
  }

  if (count)
  {
    lseek(handle, old_start, SEEK_SET);
    read(handle, block, count);
    lseek(handle, new_start, SEEK_SET);
    write(handle, block, count);
  }
}


/* Slow, slow, slow */

void tt_data_prepend_bytes(TT *tt, void *src, u32 len)
{
  if (tt_has_data(tt))
  {
    if (tt_data_is_internal(tt))
    {
      tt->data = realloc(tt->data, tt->size + len);
      memmove(tt->data + len, tt->data, tt->size);
      memcpy(tt->data, src, len);
      tt->size += len;
    }
    else
    {
      int handle;

      handle = tt_get_external_handle(tt);
      if (handle >= 0)
      {
        tt_data_external_offset(handle, 0, len);
        lseek(handle, 0, SEEK_SET);
        write(handle, src, len);
        tt->handle = -1;
        close(handle);  /* TODO: Make more intelligent system for maintaining handles */
        tt->size += len;
      }
      else
      {
        /* Out of filehandles? */
        
        log_put_abort("Unable to get handle for external data.");
      }
    }
  }
  else tt_data_set_internal(tt, src, len, TRUE);
}


void tt_data_set_int(TT *tt, int val)
{
  tt_data_set_internal(tt, &val, sizeof(val), TRUE);
}


void tt_data_set_ptr(TT *tt, void *ptr)
{
  tt_data_set_internal(tt, &ptr, sizeof(ptr), TRUE);
}


void tt_data_set_str(TT *tt, const char *str)
{
  tt_data_set_internal(tt, (char *) str, strlen(str), TRUE);
}


/* Returns success indicator */

int tt_data_set_file(TT *tt, const char *path, int local)
{
  int handle;

  tt_data_del(tt);

  tt->data = strdup(path);
  tt->data_is_internal = 0;
  tt->data_is_local = local;
  
  handle = tt_get_external_handle(tt);
  if (handle >= 0)
  {
    tt->size = lseek(handle, 0, SEEK_END);
    tt->handle = -1;
    close(handle);
  }
  
  return(handle >= 0 ? 1 : 0);
}


void *tt_data_get(TT *tt)
{
  return(tt->data);
}


u32 tt_data_get_bytes(TT *tt, void *dest, u32 start, u32 len)
{
  u32 real_len;
  int handle;
  
  if (!tt_has_data(tt)) return(0);
  
  if (tt_size(tt) <= start) return(0);
  else if (tt_size(tt) - start > len) real_len = len;
  else real_len = tt_size(tt) - start;

  if (tt_data_is_internal(tt)) memcpy(dest, tt->data + start, real_len);
  else
  {
    handle = tt_get_external_handle(tt);
    if (handle >= 0)
    {
      lseek(handle, start, SEEK_SET);
      read(handle, dest, real_len);
      tt->handle = -1;
      close(handle);
    }
  }
  
  return(real_len);
}


int tt_data_get_int(TT *tt)
{
  int val;
  
  tt_data_get_bytes(tt, &val, 0, sizeof(val));
  return(val);
}


void *tt_data_get_ptr(TT *tt)
{
  void *ptr;
  
  tt_data_get_bytes(tt, &ptr, 0, sizeof(ptr));
  return(ptr);
}


char *tt_data_get_str(TT *tt)
{
  char *str;

  str = malloc(tt_size(tt) + 1);
  tt_data_get_bytes(tt, str, 0, tt_size(tt));
  *(str + tt_size(tt)) = 0;

  return(str);
}
