/*
 * Copyright(C) 2012 Canonical Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Ken VaDine <ken.vandine@canonical.com>
 */


    /**
     * This is a thin wrapper around {@link Ag.Account}.
     *
     * @since 0.1
     */
    class FriendsAccount : GLib.Object {
        private Ag.Account _account = null;

        /**
         * String that uniquely identifies the account. It takes the
         * form of '#/protocol', where # is the {@link Ag.Account.id}
         * and protocol is the lowercase name of the social network
         * that the account is from. Possible values include
         * '6/facebook' or '2/twitter'. The number on it's own is
         * unique (so you'd never see '1/twitter' and '1/facebook' on
         * the same system), but we do support registering multiple
         * accounts per protocol, so finding '1/twitter', '2/twitter',
         * and '3/twitter' is perfectly acceptable.
         *
         * @since 0.1
         */
        public uint id {
            get { return unique_id; }
        }

        /**
         * This is the username that the social network uses to
         * identify the user. For example on twitter, this is the name
         * that people use to @mention you in their tweets.
         *
         * @since 0.1
         */
        public string username {
            get { return _account.get_display_name(); }
        }

        /**
         * This is the bare name of the protocol. Possible values are
         * ``facebook``, ``foursquare``, ``flickr``, ``identica``, and
         * ``twitter``.
         *
         * @since 0.1
         */
        public string service {
            get { return _account.get_provider_name(); }
        }

        /**
         * A single, solid color used to represent the given protocol.
         * This is stored as a familiar HTML style hex triplet, in the
         * form of 'RRGGBB'.
         *
         * @since 0.1
         */
        private string _color;
        public string color {
            get {
              GLib.Value t = "";
              _account.get_value("friends/color", ref t);
              _color = t.get_string();
              return _color;
            }
            set {
              GLib.Value t = value;
              try
              {
                  _account.set_value("friends/color", t);
                  _account.store_blocking();
              }
              catch (Ag.AccountsError e)
              {
                  warning (e.message);
              }
            }
        }

        /**
         * Controls whether this account should be included in the
         * global ``send_message`` call. If you want to send a
         * particular message to some, but not all accounts, you can
         * use this to effectively 'disable' the account.
         *
         * Set to the string "1" to enable and the string "0" to disable.
         *
         * @since 0.1
         * @see toggle_send_enabled
         */
        private string? _send_enabled;
        public string? send_enabled {
            get {
              GLib.Value t = false;
              var res = _account.get_value("friends/send_enabled", ref t);
              if (res == Ag.SettingSource.NONE) {
                _send_enabled = null;
                return _send_enabled;
              }
              if (t.get_boolean())
                _send_enabled = "1";
              else
                _send_enabled = "0";
              return _send_enabled;
            }
            set {
              GLib.Value t = false;
              if (value == "1")
                t = true;
              try
              {
                  _account.set_value("friends/send_enabled", t);
                  _account.store_blocking();
              }
              catch (Ag.AccountsError e)
              {
                  warning (e.message);
              }
            }
        }

        private Ag.AccountService account_service;
        private uint unique_id;

        /**
         * Emitted when the account has changed, most useful for being
         * notified of changes to {@link send_enabled}.
         *
         * @since 0.1
         */
        public signal void updated();

        /**
         * Emitted when the account has been deleted. This happens
         * when the user deletes the account from the Ubuntu Online
         * Accounts screen.
         *
         * @since 0.1
         */
        public signal void deleted();

        /**
         * Construct a new FriendsAccount object.
         *
         * @params account_service The {@link Ag.AccountService}
         * object that our FriendsAccount will be wrapping.
         *
         * @since 0.1
         */
        public FriendsAccount (Ag.AccountService account_service) {
            this.account_service = account_service;
            _account = account_service.get_account();
            unique_id = _account.id;

            account_service.changed.connect(() => {
                updated();
            });
            _account.deleted.connect(() => {
                deleted();
            });
        }

        /**
         * Toggle the value of send_enabled for the account.
         *
         * @since 0.1
         * @see send_enabled
         */
        public void toggle_send_enabled()
        {
            if (send_enabled == "1")
              send_enabled = "0";
            else
              send_enabled = "1";
        }

    }
