/*
 * Copyright (C) 2012 Canonical Ltd
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Ken VanDine <ken.vandine@canonical.com>
 */

class DispatcherSuite : Object
{
    private Friends.Dispatcher dispatcher;

    public DispatcherSuite ()
    {
        dispatcher = new Friends.Dispatcher ();

        // Synchronous tests...
        GLib.Test.add_data_func (
            "/Unit/Friends/Dispatcher/Refresh",
            test_refresh);
        GLib.Test.add_data_func (
            "/Unit/Friends/Dispatcher/Features",
            test_features);
        GLib.Test.add_data_func (
            "/Unit/Friends/Dispatcher/Upload",
            test_upload);
        // FIXME: Why won't this test pass?
        // GLib.Test.add_data_func (
        //     "/Unit/Friends/Dispatcher/SendMessage",
        //     test_send_message);
        GLib.Test.add_data_func (
            "/Unit/Friends/Dispatcher/SendMessageAccount",
            test_send_message_with_account);
        GLib.Test.add_data_func (
            "/Unit/Friends/Dispatcher/SendReply",
            test_send_reply);
        GLib.Test.add_data_func (
            "/Unit/Friends/Dispatcher/Retweet",
            test_retweet);
        GLib.Test.add_data_func (
            "/Unit/Friends/Dispatcher/Home",
            test_home);
        GLib.Test.add_data_func (
            "/Unit/Friends/Dispatcher/Wall",
            test_wall);
        GLib.Test.add_data_func (
            "/Unit/Friends/Dispatcher/Search",
            test_search);
        GLib.Test.add_data_func (
            "/Unit/Friends/Dispatcher/Like",
            test_like);
        GLib.Test.add_data_func (
            "/Unit/Friends/Dispatcher/Unlike",
            test_unlike);
        GLib.Test.add_data_func (
            "/Unit/Friends/Dispatcher/Delete",
            test_delete);
        GLib.Test.add_data_func (
            "/Unit/Friends/Dispatcher/Contacts",
            test_contacts);
        // FIXME: I need to figure out how to assert that an
        // exception gets raised before I can make this test work.
        // GLib.Test.add_data_func (
        //     "/Unit/Friends/Dispatcher/ClearIndicators",
        //     test_clear_indicators);
        GLib.Test.add_data_func (
            "/Unit/Friends/Dispatcher/URLShorten",
            test_url_shorten);

        // Asynchronous tests...
        GLib.Test.add_data_func (
            "/Unit/Friends/Dispatcher/UploadAsync",
            test_upload_async);
        GLib.Test.add_data_func (
            "/Unit/Friends/Dispatcher/SendMessageAsync",
            test_send_message_async);
        GLib.Test.add_data_func (
            "/Unit/Friends/Dispatcher/SendReplyAsync",
            test_send_reply_async);
        GLib.Test.add_data_func (
            "/Unit/Friends/Dispatcher/RetweetAsync",
            test_retweet_async);
        GLib.Test.add_data_func (
            "/Unit/Friends/Dispatcher/HomeAsync",
            test_home_async);
        GLib.Test.add_data_func (
            "/Unit/Friends/Dispatcher/WallAsync",
            test_wall_async);
        GLib.Test.add_data_func (
            "/Unit/Friends/Dispatcher/SearchAsync",
            test_search_async);
        GLib.Test.add_data_func (
            "/Unit/Friends/Dispatcher/LikeAsync",
            test_like_async);
        GLib.Test.add_data_func (
            "/Unit/Friends/Dispatcher/UnlikeAsync",
            test_unlike_async);
        GLib.Test.add_data_func (
            "/Unit/Friends/Dispatcher/DeleteAsync",
            test_delete_async);
        GLib.Test.add_data_func (
            "/Unit/Friends/Dispatcher/ContactsAsync",
            test_contacts_async);
    }

    internal void test_refresh ()
    {
        dispatcher.refresh ();
        // FIXME: find a way to test this...
    }

    internal void test_features ()
    {
        string[] features = dispatcher.features ("protocol name here");
        assert (features != null);
        assert (features[0] == "protocol");
        assert (features[1] == "name");
        assert (features[2] == "here");
        assert (features.length == 3);
    }

    internal void test_upload ()
    {
        uint acct = 1;
        string uri = "file:///tmp/foo.png";
        string desc = "Something interesting to say";
        var result = dispatcher.upload (acct, uri, desc);
        assert (result ==
                "Called with: account_id=1, uri=file:///tmp/foo.png, " +
                "description=Something interesting to say");
    }

    internal void test_upload_async ()
    {
        var loop = new MainLoop ();
        uint acct = 1;
        string uri = "file:///tmp/foo.png";
        string desc = "Something interesting to say";
        string result = "";
        dispatcher.upload_async.begin (
            acct, uri, desc, (obj, res) => {
                dispatcher.upload_async.end (res, out result);
                loop.quit ();
            }
        );
        loop.run ();

        assert (result ==
                "Called with: account_id=1, uri=file:///tmp/foo.png, " +
                "description=Something interesting to say");
    }

    internal void test_send_message ()
    {
        var result = dispatcher.send_message (null, "Hello");
        assert (result == "Called with: action=send, account_id=, arg=Hello");
    }

    internal void test_send_message_async ()
    {
        var loop = new MainLoop ();
        string result = "";
        dispatcher.send_message_async.begin (
            3, "Goodbye", (obj, res) => {
                dispatcher.send_message_async.end (res, out result);
                loop.quit ();
            }
        );
        loop.run ();

        assert (result == "Called with: action=send, account_id=3, arg=Goodbye");
    }

    internal void test_send_message_with_account ()
    {
        var result = dispatcher.send_message (12, "Testing...");
        assert (result ==
                "Called with: action=send, account_id=12, arg=Testing...");
    }

    internal void test_send_reply ()
    {
        var result = dispatcher.send_reply (1, "100B", "Response!");
        assert (result ==
                "Called with: account_id=1, message_id=100B, msg=Response!");
    }

    internal void test_send_reply_async ()
    {
        var loop = new MainLoop ();
        string result = "";
        dispatcher.send_reply_async.begin (
            4, "msgid42", "LOL", (obj, res) => {
                dispatcher.send_reply_async.end (res, out result);
                loop.quit ();
            }
        );
        loop.run ();

        assert (result ==
                "Called with: account_id=4, message_id=msgid42, msg=LOL");
    }

    internal void test_retweet ()
    {
        var result = dispatcher.retweet (1, "messageid1234");
        assert (result ==
                "Called with: action=retweet, account_id=1, arg=messageid1234");
    }

    internal void test_retweet_async ()
    {
        var loop = new MainLoop ();
        string result = "";
        dispatcher.retweet_async.begin (
            4, "msgid69", (obj, res) => {
                dispatcher.retweet_async.end (res, out result);
                loop.quit ();
            }
        );
        loop.run ();

        assert (result ==
                "Called with: action=retweet, account_id=4, arg=msgid69");
    }

    internal void test_home ()
    {
        var result = dispatcher.home (10);
        assert (result == "Called with: action=home, account_id=10, arg=");
        result = dispatcher.home (null);
        assert (result == "Called with: action=home, account_id=, arg=");
    }

    internal void test_home_async ()
    {
        var loop = new MainLoop ();
        string result = "";
        dispatcher.home_async.begin (
            23, (obj, res) => {
                dispatcher.home_async.end (res, out result);
                loop.quit ();
            }
        );
        loop.run ();

        assert (result == "Called with: action=home, account_id=23, arg=");
    }

    internal void test_wall ()
    {
        var result = dispatcher.wall (10);
        assert (result == "Called with: action=wall, account_id=10, arg=");
        result = dispatcher.wall (null);
        assert (result == "Called with: action=wall, account_id=, arg=");
    }

    internal void test_wall_async ()
    {
        var loop = new MainLoop ();
        string result = "";
        dispatcher.wall_async.begin (
            12, (obj, res) => {
                dispatcher.wall_async.end (res, out result);
                loop.quit ();
            }
        );
        loop.run ();

        assert (result == "Called with: action=wall, account_id=12, arg=");
    }

    internal void test_search ()
    {
        var result = dispatcher.search (10, "terms");
        assert (result ==
                "Called with: action=search, account_id=10, arg=terms");
        result = dispatcher.search (null, "terms");
        assert (result ==
                "Called with: action=search, account_id=, arg=terms");
    }

    internal void test_search_async ()
    {
        var loop = new MainLoop ();
        string result = "";
        dispatcher.search_async.begin (
            8, "what i want", (obj, res) => {
                dispatcher.search_async.end (res, out result);
                loop.quit ();
            }
        );
        loop.run ();

        assert (result ==
                "Called with: action=search, account_id=8, arg=what i want");
    }

    internal void test_like ()
    {
        var result = dispatcher.like (10, "msgid9876");
        assert (result ==
                "Called with: action=like, account_id=10, arg=msgid9876");
    }

    internal void test_like_async ()
    {
        var loop = new MainLoop ();
        string result = "";
        dispatcher.like_async.begin (
            7, "msgid9000", (obj, res) => {
                dispatcher.like_async.end (res, out result);
                loop.quit ();
            }
        );
        loop.run ();

        assert (result ==
                "Called with: action=like, account_id=7, arg=msgid9000");
    }

    internal void test_unlike ()
    {
        var result = dispatcher.unlike (5, "msgid9876");
        assert (result ==
                "Called with: action=unlike, account_id=5, arg=msgid9876");
    }

    internal void test_unlike_async ()
    {
        var loop = new MainLoop ();
        string result = "";
        dispatcher.unlike_async.begin (
            6, "msgid9001", (obj, res) => {
                dispatcher.unlike_async.end (res, out result);
                loop.quit ();
            }
        );
        loop.run ();

        assert (result ==
                "Called with: action=unlike, account_id=6, arg=msgid9001");
    }

    internal void test_delete ()
    {
        var result = dispatcher.delete (15, "msgid3456");
        assert (result ==
                "Called with: action=delete, account_id=15, arg=msgid3456");
    }

    internal void test_delete_async ()
    {
        var loop = new MainLoop ();
        string result = "";
        dispatcher.delete_async.begin (
            5, "msgid9002", (obj, res) => {
                dispatcher.delete_async.end (res, out result);
                loop.quit ();
            }
        );
        loop.run ();

        assert (result ==
                "Called with: action=delete, account_id=5, arg=msgid9002");
    }

    internal void test_contacts ()
    {
        var result = dispatcher.contacts (15);
        assert (result == "Called with: action=contacts, account_id=15, arg=");
        result = dispatcher.contacts (null);
        assert (result == "Called with: action=contacts, account_id=, arg=");
    }

    internal void test_contacts_async ()
    {
        var loop = new MainLoop ();
        string result = "";
        dispatcher.contacts_async.begin (
            5, (obj, res) => {
                dispatcher.contacts_async.end (res, out result);
                loop.quit ();
            }
        );
        loop.run ();

        assert (result == "Called with: action=contacts, account_id=5, arg=");
    }

    internal void test_clear_indicators ()
    {
        /*
         * In the mocked out friends-dispatcher, this method
         * switches a flip that causes everything else to
         * deliberately fail, so if we call this and then
         * everything breaks, we know the correct dbus method has
         * been called.
         */
        bool failure = false;
        try
        {
            dispatcher.messaging_menu_clear ();
            dispatcher.send_reply(1, "two", "three");
        }
        catch (GLib.Error e)
        {
            failure = true;
        }
        assert (failure);
    }

    internal void test_url_shorten ()
    {
        var result = dispatcher.shorten (
            "http://example.com/really/really/long");
        // The mock dispatcher returns the length of your URL as a string;
        assert (result == "37");
    }
}
