/**
 * @file libgalago/galago-avatar.c Galago Avatar API
 *
 * @Copyright (C) 2004-2005 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libgalago/galago-avatar.h>
#include <libgalago/galago-assert.h>
#include <libgalago/galago-context.h>
#include <libgalago/galago-core.h>
#include <libgalago/galago-object-utils.h>
#include <libgalago/galago-utils.h>
#include <libgalago/galago-utils-priv.h>
#include <stdio.h>
#include <string.h>

struct _GalagoAvatarPrivate
{
	GalagoAccount *account;
};


/**************************************************************************
 * Object/Class support
 **************************************************************************/
GALAGO_REGISTER_CLASS(galago_avatar, GalagoAvatar, GALAGO_CLASS_IMAGE,
					  GALAGO_DBUS_AVATAR_INTERFACE);
DEPRECATE_OLD_GET_CLASS(galago_avatars, galago_avatar);

static void
galago_avatar_object_init(GalagoAvatar *avatar)
{
	avatar->priv = galago_new0(GalagoAvatarPrivate, 1);
}

static void
galago_avatar_object_finalize(GalagoObject *object)
{
	GalagoAvatar *avatar = (GalagoAvatar *)object;
	GalagoAccount *account;

	account = galago_avatar_get_account(avatar);

	if (galago_account_get_avatar(account, FALSE) == avatar)
		galago_account_set_avatar(account, NULL);

	free(avatar->priv);
}

static void
galago_avatar_dbus_message_append(DBusMessageIter *iter,
								  const GalagoObject *object)
{
	GalagoAvatar *avatar = (GalagoAvatar *)object;
	unsigned char *image_data;
	size_t image_len;

	galago_dbus_message_iter_append_object(iter,
		galago_avatar_get_account(avatar));

	galago_avatar_get_image_data(avatar, &image_data, &image_len);
	galago_dbus_message_iter_append_byte_array(iter, image_data, image_len);
}

static void *
galago_avatar_dbus_message_get(DBusMessageIter *iter)
{
	GalagoAvatar *avatar;
	GalagoAccount *account;
	unsigned char *image_data;
	int image_len;

	account = galago_dbus_message_iter_get_object(iter, GALAGO_CLASS_ACCOUNT);
	dbus_message_iter_next(iter);

	galago_dbus_message_iter_get_byte_array(iter, &image_data, &image_len);

	avatar = galago_avatar_new(account, image_data, image_len);

	return avatar;
}

static void
galago_avatar_class_init(GalagoObjectClass *klass)
{
	klass->finalize            = galago_avatar_object_finalize;
	klass->dbus_message_append = galago_avatar_dbus_message_append;
	klass->dbus_message_get    = galago_avatar_dbus_message_get;
}


/**************************************************************************
 * Avatar API
 **************************************************************************/
GalagoAvatar *
galago_avatar_new(GalagoAccount *account, const unsigned char *data,
				  size_t len)
{
	GalagoAvatar *avatar;

	galago_return_val_if_fail(galago_is_initted(),        NULL);
	galago_return_val_if_fail(account != NULL,            NULL);
	galago_return_val_if_fail(GALAGO_IS_ACCOUNT(account), NULL);

	avatar = galago_account_get_avatar(account, FALSE);

	if (avatar == NULL)
	{
		const char *obj_prefix;

		galago_context_push(galago_object_get_context(account));
		avatar = galago_object_new(GALAGO_CLASS_AVATAR);
		galago_context_pop();

		avatar->priv->account = account;

		if ((obj_prefix = galago_object_get_dbus_path(account)) != NULL)
		{
			size_t len;
			char *obj_path;

			len = strlen(obj_prefix) + strlen("/avatar") + 1;

			obj_path = galago_new(char, len);
			snprintf(obj_path, len, "%s/avatar", obj_prefix);

			galago_object_set_dbus_path(avatar, obj_path);

			free(obj_path);
		}

		galago_avatar_set_image_data(avatar, data, len);
		galago_account_set_avatar(account, avatar);
	}
	else
	{
		galago_avatar_set_image_data(avatar, data, len);
	}

	return avatar;
}

void
galago_avatar_set_image_data(GalagoAvatar *avatar,
							 const unsigned char *data, size_t len)
{
	galago_return_if_fail(avatar != NULL);
	galago_return_if_fail(GALAGO_IS_AVATAR(avatar));
	galago_return_if_fail(data == NULL || len > 0);

	galago_image_set_image_data(GALAGO_IMAGE(avatar), data, len);

	if (galago_account_is_native(galago_avatar_get_account(avatar)) &&
		galago_is_connected() && galago_core_is_feed())
	{
		unsigned char *image_data;
		size_t image_len;

		galago_avatar_get_image_data(avatar, &image_data, &image_len);
		galago_dbus_send_message(avatar, "SetImage",
			galago_value_new_array(GALAGO_TYPE_CHAR, image_data, image_len,
								   NULL),
			NULL);
	}
}

GalagoAccount *
galago_avatar_get_account(const GalagoAvatar *avatar)
{
	galago_return_val_if_fail(avatar != NULL,           NULL);
	galago_return_val_if_fail(GALAGO_IS_AVATAR(avatar), NULL);

	return avatar->priv->account;
}

void
galago_avatar_get_image_data(const GalagoAvatar *avatar,
							 unsigned char **ret_data, size_t *ret_len)
{
	galago_return_if_fail(avatar != NULL);
	galago_return_if_fail(GALAGO_IS_AVATAR(avatar));
	galago_return_if_fail(ret_data != NULL || ret_len != NULL);

	galago_image_get_image_data(GALAGO_IMAGE(avatar), ret_data, ret_len);
}
