/**
 * @file libgalago/galago-core.h Galago Context API
 *
 * @Copyright (C) 2004-2005 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#ifndef _GALAGO_CORE_H_
#define _GALAGO_CORE_H_

#include <libgalago/galago-dbus.h>
#include <libgalago/galago-person.h>
#include <libgalago/galago-service.h>
#include <libgalago/galago-signals.h>
#include <libgalago/galago-types.h>

#ifdef __G_LIB_H__
# include <dbus/dbus-glib-lowlevel.h>
#endif

#define GALAGO_DBUS_CORE_INTERFACE "org.freedesktop.Galago.Core"
#define GALAGO_DBUS_CORE_OBJECT    "/org/freedesktop/Galago/Core"

typedef struct _GalagoCore GalagoCore;

typedef void (*GalagoMainLoopSetupFunc)(DBusConnection *, void *);

#ifdef __cplusplus
extern "C" {
#endif

/**
 * Initializes Galago.
 *
 * @param name The ID name used for this connection. It's often the
 *             name of the program, without spaces or special characters.
 * @param feed TRUE if this connection should feed presence data. FALSE
 *             if it will only be querying for data.
 *
 * @return TRUE if initialization was successful, or FALSE otherwise.
 */
galago_bool galago_init(const char *name, galago_bool feed);

/**
 * Initializes Galago, passing a mainloop setup function to call.
 *
 * @param name The ID name used for this connection. It's often the
 *             name of the program, without spaces or special characters.
 * @param feed TRUE if this connection should feed presence data. FALSE
 *             if it will only be querying for data.
 * @param func The mainloop setup function to call.
 * @param data User-specified data to pass to the function.
 *
 * @return TRUE if initialization was successful, or FALSE otherwise.
 */
galago_bool galago_init_with_mainloop(const char *name, galago_bool feed,
									  GalagoMainLoopSetupFunc func,
									  void *data);

/**
 * Uninitializes Galago.
 *
 * This will be called automatically on exit, but can be called manually.
 */
void galago_uninit(void);

/**
 * Returns whether or not Galago has been initialized.
 *
 * @return TRUE if Galago has been initialized, or FALSE otherwise.
 */
galago_bool galago_is_initted(void);

/**
 * Returns whether or not Galago is connected to the daemon.
 *
 * @return TRUE if Galago is connected, or FALSE otherwise.
 */
galago_bool galago_is_connected(void);

/**
 * Returns whether or not Galago is successfully registered to the daemon.
 *
 * @return TRUE if this Galago connection is registered, or FALSE otherwise.
 */
galago_bool galago_is_registered(void);

/**
 * Returns whether or not the Galago daemon is active.
 *
 * @return TRUE if the daemon is active, or FALSE.
 */
galago_bool galago_is_daemon_active(void);

/**
 * Returns whether or not this instance of Galago is the daemon.
 *
 * @return TRUE if this is the daemon, or FALSE.
 */
galago_bool galago_is_daemon(void);

/**
 * Returns Galago's D-BUS connection.
 *
 * @return The D-BUS connection.
 */
DBusConnection *galago_core_get_dbus_conn(void);

/**
 * Returns the local Galago connection's unique ID.
 *
 * This ID is used in D-BUS object paths. It's really only needed
 * internally.
 *
 * @return This connection's unique ID.
 */
const char *galago_core_get_uid(void);

/**
 * Returns the D-BUS object path representing this connection.
 *
 * @return This connection's D-BUS object path.
 */
const char *galago_core_get_obj_path(void);

/**
 * Returns whether or not this Galago connection works as a feed.
 *
 * @return TRUE if this connection works as a feed, or FALSE if not.
 */
galago_bool galago_core_is_feed(void);

/**
 * Sets whether or not all services, accounts, etc. should be watched for
 * property updates.
 *
 * This will only add accounts, persons, presences, etc. that have a known
 * parent object. That is, if a signal arrives for an account on a service
 * that the local program does not know about, it will be ignored. If you
 * wish to get information on every single object, call the list query
 * functions first.
 *
 * @param watch_all TRUE if everything should be watched, or FALSE.
 */
void galago_core_set_watch_all(galago_bool watch_all);

/**
 * Returns whether or not all services, accounts, etc. is being watched
 * for property updates.
 *
 * @return TRUE if everything is being watched, or FALSE.
 */
galago_bool galago_core_get_watch_all(void);

/**
 * Sets whether or not the application should exit when the Galago
 * daemon disconnects.
 *
 * This is mainly only used for auto-launched feeds.
 *
 * @param exit_with_daemon TRUE if the application should exit with the
 *                         daemon, or FALSE.
 */
void galago_core_set_exit_with_daemon(galago_bool exit_with_daemon);

/**
 * Returns whether or not the application should exit when the Galago
 * daemon disconnects.
 *
 * @return TRUE if the application should exit with the daemon, or FALSE.
 */
galago_bool galago_core_get_exit_with_daemon(void);

/**
 * Adds a service to Galago's services cache.
 *
 * @param service The service to add.
 */
void galago_core_add_service(GalagoService *service);

/**
 * Removes a service from Galago's services cache.
 *
 * @param service The service to remove.
 */
void galago_core_remove_service(GalagoService *service);

/**
 * Returns the service with the specified ID.
 *
 * @param id     The service ID.
 * @param native TRUE if the desired service is native.
 * @param query  TRUE if a remote query should be done if there is no
 *               local service found, or FALSE.
 *
 * @return The service, if found, or NULL.
 */
GalagoService *galago_core_get_service(const char *id, galago_bool native,
									   galago_bool query);

/**
 * Returns a list of known services.
 *
 * @param native TRUE if the desired services are native.
 * @param query  TRUE if a remote query should be done if there are no
 *               local services found, or FALSE.
 *
 * @return The list of services, or NULL.
 */
const GalagoList *galago_core_get_services(galago_bool native,
										   galago_bool query);

/**
 * Adds a person to Galago's person cache.
 *
 * This is used internally. You don't need to call this.
 *
 * @param person The person to add.
 */
void galago_core_add_person(GalagoPerson *person);

/**
 * Removes a person from Galago's person cache.
 *
 * This is used internally. You don't need to call this.
 *
 * @param person The person to remove.
 */
void galago_core_remove_person(GalagoPerson *person);

/**
 * Returns the person with the specified ID.
 *
 * @param id     The person ID.
 * @param native TRUE if the desired person is native or not.
 * @param query  TRUE if a remote query should be done if there is no
 *               local person found, or FALSE.
 *
 * @return The person, if found, or NULL.
 */
GalagoPerson *galago_core_get_person(const char *id, galago_bool native,
									 galago_bool query);

/**
 * Returns a list of known people.
 *
 * @param native TRUE if the desired people are native or not.
 * @param query  TRUE if a remote query should be done if there is no
 *               local people found, or FALSE.
 *
 * @return The people, if found, or NULL.
 */
const GalagoList *galago_core_get_people(galago_bool native,
										 galago_bool query);

/**
 * Returns the "Me" person.
 *
 * @param native TRUE if the desired person is native or not.
 * @param query  TRUE if a remote query should be done if there is no
 *               local person found, or FALSE.
 *
 * @return The "Me" person, if found, or NULL.
 */
GalagoPerson *galago_core_get_me(galago_bool native, galago_bool query);

/**
 * Returns the core object.
 *
 * This is needed for signals.
 *
 * @return The core object.
 */
GalagoCore *galago_get_core(void);

/**
 * Initializes Galago and sets the D-BUS connection to use the GLib mainloop.
 *
 * @param name    The name of the connection.
 * @param feed    TRUE if this connection will be feeding presence.
 * @param context The optional glib context. NULL for the default one.
 *
 * @return TRUE if initialized, or FALSE otherwise.
 */
#define galago_glib_init(name, feed, context) \
	galago_init_with_mainloop((name), (feed), \
		(GalagoMainLoopSetupFunc)dbus_connection_setup_with_g_main, (context))

#ifdef __cplusplus
}
#endif

#endif /* _GALAGO_CORE_H_ */
