/**
 * @file libgalago/galago-dbus.c D-BUS Support API
 *
 * @Copyright (C) 2004-2005 Christian Hammond.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libgalago/galago-dbus.h>
#include <libgalago/galago-assert.h>
#include <libgalago/galago-core.h>
#include <libgalago/galago-status.h>
#include <libgalago/galago-utils-priv.h>
#include <string.h>
#include <stdio.h>

void
galago_dbus_message_iter_append_string_or_nil(DBusMessageIter *iter,
											  const char *str)
{
	galago_return_if_fail(iter != NULL);

	if (str == NULL)
		str = "";

	galago_dbus_message_iter_append_string(iter, str);
}

char *
galago_dbus_message_iter_get_string_or_nil(DBusMessageIter *iter)
{
	char *str;

	galago_return_val_if_fail(iter != NULL, NULL);

	galago_dbus_message_iter_get_string(iter, str);

	if (*str == '\0')
	{
#if !GALAGO_CHECK_DBUS_VERSION(0, 30)
		dbus_free(str);
#endif

		str = NULL;
	}

	return str;
}

void
galago_dbus_message_iter_append_object(DBusMessageIter *iter,
									   const void *object)
{
	GalagoObjectClass *klass;

	galago_return_if_fail(iter != NULL);
	galago_return_if_fail(object != NULL && GALAGO_IS_OBJECT(object));

	klass = GALAGO_OBJECT_CLASS(object);

	if (klass->dbus_message_append != NULL)
	{
#if GALAGO_CHECK_DBUS_VERSION(0, 30)
		DBusMessageIter struct_iter;

		dbus_message_iter_open_container(iter, DBUS_TYPE_STRUCT, NULL,
										 &struct_iter);
		klass->dbus_message_append(&struct_iter, object);
		dbus_message_iter_close_container(iter, &struct_iter);
#else
		klass->dbus_message_append(iter, object);
#endif
	}
	else
	{
		galago_log_error("Class type %s passed to "
						 "galago_dbus_message_iter_append_object does not "
						 "implement dbus_message_append!\n",
						 galago_class_get_name(klass));
	}
}

void
galago_dbus_message_iter_append_object_list(DBusMessageIter *iter,
											const GalagoList *list)
{
	const GalagoList *l;
	size_t num_objects;
	DBusMessageIter *append_iter;
#if 0 && GALAGO_CHECK_DBUS_VERSION(0, 30)
	DBusMessageIter array_iter;
#endif

	galago_return_if_fail(iter != NULL);

	num_objects = galago_list_get_count(list);

#if 0 && GALAGO_CHECK_DBUS_VERSION(0, 30)
	dbus_message_iter_open_container(iter, DBUS_TYPE_ARRAY,
									 DBUS_STRUCT_BEGIN_CHAR_AS_STRING
									 DBUS_STRUCT_END_CHAR_AS_STRING,
									 &array_iter);
	append_iter = &array_iter;
#else
	galago_dbus_message_iter_append_uint32(iter, num_objects);
	append_iter = iter;
#endif

	/*
	 * NOTE: There's a possible problem of objects in a list being
	 *       of different classes. If this is the case, oops, screwed.
	 *       Up to the developer to fix.
	 */
	for (l = list; l != NULL; l = l->next)
	{
		GalagoObject *object = (GalagoObject *)l->data;

		galago_dbus_message_iter_append_object(append_iter, object);
	}

#if 0 && GALAGO_CHECK_DBUS_VERSION(0, 30)
	dbus_message_iter_close_container(iter, &array_iter);
#endif
}

void *
galago_dbus_message_iter_get_object(DBusMessageIter *iter,
									const GalagoObjectClass *klass)
{
	GalagoObject *object;
#if GALAGO_CHECK_DBUS_VERSION(0, 30)
	DBusMessageIter temp_iter;
#endif

	galago_return_val_if_fail(iter  != NULL, NULL);
	galago_return_val_if_fail(klass != NULL, NULL);

	if (klass->dbus_message_get == NULL)
	{
		galago_log_error("Class type %s passed to "
						 "galago_dbus_message_iter_get_object does not "
						 "implement dbus_message_get!\n",
						 galago_class_get_name(klass));
		return NULL;
	}

#if GALAGO_CHECK_DBUS_VERSION(0, 30)
	dbus_message_iter_recurse(iter, &temp_iter);
	iter = &temp_iter;
#endif

	object = klass->dbus_message_get(iter);

	return object;
}

GalagoList *
galago_dbus_message_iter_get_object_list(DBusMessageIter *iter,
										 const GalagoObjectClass *klass)
{
	GalagoList *list = NULL;
#if 0 && GALAGO_CHECK_DBUS_VERSION(0, 30)
	DBusMessageIter array_iter;
#else
	dbus_uint32_t num_objects, i;
#endif

	galago_return_val_if_fail(iter  != NULL, NULL);
	galago_return_val_if_fail(klass != NULL, NULL);

#if 0 && GALAGO_CHECK_DBUS_VERSION(0, 30)
	dbus_message_iter_recurse(iter, &array_iter);

	while (dbus_message_iter_get_arg_type(&array_iter) == DBUS_TYPE_STRUCT)
	{
		list = galago_list_append(list,
			galago_dbus_message_iter_get_object(&array_iter, klass));

		dbus_message_iter_next(&array_iter);
	}
#else
	galago_dbus_message_iter_get_uint32(iter, num_objects);

	for (i = 0; i < num_objects; i++)
	{
		dbus_message_iter_next(iter);

		list = galago_list_append(list,
			galago_dbus_message_iter_get_object(iter, klass));
	}
#endif

	return list;
}

static void
galago_dbus_message_iter_append_value_list(DBusMessageIter *iter,
										   GalagoValue *value)
{
	GalagoList *list = NULL;
	const GalagoList *l;

	switch (galago_value_get_subtype(value))
	{
		case GALAGO_TYPE_OBJECT:
			for (l = galago_value_get_list(value); l != NULL; l = l->next)
			{
				GalagoValue *temp_value = (GalagoValue *)l->data;

				list = galago_list_append(list,
					galago_value_get_object(temp_value));
			}

			galago_dbus_message_iter_append_object_list(iter, list);
			galago_list_destroy(list);
			break;

		default:
			galago_log_warning("Unsupported list type %d appended to "
							   "message\n",
							   galago_value_get_type(value));
			break;
	}
}

static void
galago_dbus_message_iter_append_value_array(DBusMessageIter *iter,
											GalagoValue *value)
{
	const void *array;
	size_t array_size;

	galago_value_get_array(value, &array, &array_size);

	switch (galago_value_get_subtype(value))
	{
		case GALAGO_TYPE_CHAR:
		case GALAGO_TYPE_UCHAR:
			galago_dbus_message_iter_append_byte_array(iter, array,
													   array_size);
			break;

		case GALAGO_TYPE_BOOLEAN:
			galago_dbus_message_iter_append_boolean_array(iter, array,
														  array_size);
			break;

		case GALAGO_TYPE_SHORT:
		case GALAGO_TYPE_INT:
		case GALAGO_TYPE_LONG:
			galago_dbus_message_iter_append_int32_array(iter, array,
														array_size);
			break;

		case GALAGO_TYPE_USHORT:
		case GALAGO_TYPE_UINT:
		case GALAGO_TYPE_ULONG:
			galago_dbus_message_iter_append_uint32_array(iter, array,
														 array_size);
			break;

		case GALAGO_TYPE_STRING:
			galago_dbus_message_iter_append_string_array(iter, array,
														 array_size);
			break;

		default:
			galago_log_warning("Invalid array type %d appended to message\n",
							   galago_value_get_subtype(value));
			break;
	}
}

void
galago_dbus_message_iter_append_value(DBusMessageIter *iter,
									  GalagoValue *value)
{
	galago_return_if_fail(iter  != NULL);
	galago_return_if_fail(value != NULL);

#define CHECK_APPEND_VALUE(type, dbusname, galagoname, vartype) \
	case type: \
	{ \
		vartype var = galago_value_get_##galagoname(value); \
		galago_dbus_message_iter_append_##dbusname(iter, var); \
		break; \
	}

	switch (galago_value_get_type(value))
	{
		CHECK_APPEND_VALUE(GALAGO_TYPE_CHAR,    byte,    char,
						   char);
		CHECK_APPEND_VALUE(GALAGO_TYPE_UCHAR,   byte,    uchar,
						   unsigned char);
		CHECK_APPEND_VALUE(GALAGO_TYPE_BOOLEAN, boolean, boolean,
						   galago_bool);
		CHECK_APPEND_VALUE(GALAGO_TYPE_SHORT,   int32,   short,
						   short);
		CHECK_APPEND_VALUE(GALAGO_TYPE_USHORT,  uint32,  ushort,
						   unsigned short);
		CHECK_APPEND_VALUE(GALAGO_TYPE_INT,     int32,   int,
						   int);
		CHECK_APPEND_VALUE(GALAGO_TYPE_UINT,    uint32,  uint,
						   unsigned int);
		CHECK_APPEND_VALUE(GALAGO_TYPE_LONG,    int32,   long,
						   long);
		CHECK_APPEND_VALUE(GALAGO_TYPE_ULONG,   uint32,  ulong,
						   unsigned long);

		case GALAGO_TYPE_STRING:
			galago_dbus_message_iter_append_string_or_nil(iter,
				galago_value_get_string(value));
			break;

		case GALAGO_TYPE_OBJECT:
			galago_dbus_message_iter_append_object(iter,
				galago_value_get_object(value));
			break;

		case GALAGO_TYPE_LIST:
			galago_dbus_message_iter_append_value_list(iter, value);
			break;

		case GALAGO_TYPE_ARRAY:
			galago_dbus_message_iter_append_value_array(iter, value);
			break;

		default:
			galago_log_warning("Invalid type %d appended to message\n",
							   galago_value_get_type(value));
			break;
	}

#undef CHECK_APPEND_VALUE
}

GalagoValue *
galago_dbus_message_iter_get_value(DBusMessageIter *iter)
{
	GalagoValue *value = NULL;

	galago_return_val_if_fail(iter != NULL, NULL);

#define CHECK_GET_VALUE(dbustype, galagotype, dbusname, galagoname, vartype) \
	case dbustype: \
	{ \
		vartype var; \
		value = galago_value_new(galagotype, NULL, NULL); \
		galago_dbus_message_iter_get_##dbusname(iter, var); \
		galago_value_set_##galagoname(value, var); \
		break; \
	}

	switch (dbus_message_iter_get_arg_type(iter))
	{
		CHECK_GET_VALUE(DBUS_TYPE_BYTE,    GALAGO_TYPE_CHAR,   byte,   char,
						char);
		CHECK_GET_VALUE(DBUS_TYPE_INT32,   GALAGO_TYPE_INT,    int32,  int,
						int);
		CHECK_GET_VALUE(DBUS_TYPE_UINT32,  GALAGO_TYPE_UINT,   uint32, uint,
						unsigned int);
		CHECK_GET_VALUE(DBUS_TYPE_BOOLEAN, GALAGO_TYPE_BOOLEAN,
						boolean, boolean, galago_bool);

		case DBUS_TYPE_STRING:
		{
			char *str;
			value = galago_value_new(GALAGO_TYPE_STRING, NULL, NULL);
			str = galago_dbus_message_iter_get_string_or_nil(iter);
			galago_value_set_string(value, str);

#if !GALAGO_CHECK_DBUS_VERSION(0, 30)
			if (str != NULL)
				dbus_free(str);
#endif

			break;
		}

		default:
			galago_log_warning("Unsupported type %d retrieved from message\n",
							   dbus_message_iter_get_arg_type(iter));
			break;
	}

	return value;

#undef CHECK_GET_VALUE
}

DBusMessage *
galago_dbus_message_new_method_call(const void *object, const char *name,
									galago_bool reply, DBusMessageIter *iter)
{
	DBusMessage *message;
	GalagoObjectClass *klass;
	const char *obj_path;
	const char *iface;

	galago_return_val_if_fail(object != NULL, NULL);
	galago_return_val_if_fail(name   != NULL, NULL);
	galago_return_val_if_fail(*name  != '\0', NULL);
	galago_return_val_if_fail(GALAGO_IS_OBJECT(object), NULL);

	klass = GALAGO_OBJECT_CLASS(object);

	obj_path = galago_object_get_dbus_path(object);
	iface    = galago_class_get_dbus_iface(klass);

	if (obj_path == NULL)
	{
		galago_log_error(
			"No object path was registered for class '%s'. "
			"Please report this.\n",
			galago_class_get_name(klass));

		return NULL;
	}

	if (iface == NULL)
	{
		galago_log_error(
			"No D-BUS interface was registered for class '%s'. "
			"Please report this.\n",
			galago_class_get_name(klass));

		return NULL;
	}

	message = dbus_message_new_method_call(GALAGO_DBUS_SERVICE,
										   obj_path, iface, name);

	galago_return_val_if_fail(message != NULL, NULL);

	dbus_message_set_no_reply(message, !reply);

	if (iter != NULL)
		dbus_message_iter_init_append(message, iter);

	return message;
}

DBusMessage *
galago_dbus_message_new_method_call_vargs(const void *object,
										  const char *name,
										  galago_bool reply,
										  va_list args)
{
	DBusMessage *message;
	DBusMessageIter iter;
	GalagoValue *value;

	galago_return_val_if_fail(object != NULL, NULL);
	galago_return_val_if_fail(name   != NULL, NULL);
	galago_return_val_if_fail(*name  != '\0', NULL);
	galago_return_val_if_fail(GALAGO_IS_OBJECT(object), NULL);

	message = galago_dbus_message_new_method_call(object, name, reply, &iter);

	galago_return_val_if_fail(message != NULL, NULL);

	while ((value = (GalagoValue *)va_arg(args, GalagoValue *)) != NULL)
	{
		galago_dbus_message_iter_append_value(&iter, value);
		galago_value_destroy(value);
	}

	return message;
}

DBusMessage *
galago_dbus_message_new_method_call_args(const void *object,
										 const char *name,
										 galago_bool reply, ...)
{
	va_list args;
	DBusMessage *message;

	galago_return_val_if_fail(object != NULL, NULL);
	galago_return_val_if_fail(name   != NULL, NULL);
	galago_return_val_if_fail(*name  != '\0', NULL);
	galago_return_val_if_fail(GALAGO_IS_OBJECT(object), NULL);

	va_start(args, reply);
	message = galago_dbus_message_new_method_call_vargs(object, name,
														reply, args);
	va_end(args);

	return message;
}

void
galago_dbus_send_message(const void *object, const char *name, ...)
{
	va_list args;
	DBusMessage *message;

	galago_return_if_fail(object != NULL);
	galago_return_if_fail(name   != NULL);
	galago_return_if_fail(*name  != '\0');
	galago_return_if_fail(GALAGO_IS_OBJECT(object));

	if (!galago_is_connected())
		return;

	va_start(args, name);
	message = galago_dbus_message_new_method_call_vargs(object, name,
														FALSE, args);
	va_end(args);

	galago_return_if_fail(message != NULL);

	dbus_connection_send(galago_core_get_dbus_conn(), message, NULL);
	dbus_message_unref(message);
}

static void *
get_ret_val_from_iter(DBusMessageIter *iter, GalagoValue *value)
{
	void *retval = NULL;

	switch (galago_value_get_type(value))
	{
		case GALAGO_TYPE_LIST:
			switch (galago_value_get_subtype(value))
			{
				case GALAGO_TYPE_OBJECT:
					retval = galago_dbus_message_iter_get_object_list(iter,
						galago_value_get_object_class(value));
					break;

				default:
					galago_log_warning(
						"Unsupported list type %d returned from message\n",
						galago_value_get_subtype(value));
					break;
			}
			break;

		case GALAGO_TYPE_OBJECT:
			retval = galago_dbus_message_iter_get_object(iter,
				galago_value_get_object_class(value));
			break;

		default:
#define CHECK_GET_VALUE(type, name, vartype) \
	case type: \
	{ \
		vartype var; \
		galago_dbus_message_iter_get_##name(iter, var); \
		retval = (void *)var; \
		break; \
	}

			switch (dbus_message_iter_get_arg_type(iter))
			{
				CHECK_GET_VALUE(DBUS_TYPE_INT32,   int32,   int);
				CHECK_GET_VALUE(DBUS_TYPE_UINT32,  uint32,  unsigned int);
				CHECK_GET_VALUE(DBUS_TYPE_BOOLEAN, boolean, galago_bool);

				case DBUS_TYPE_STRING:
					retval = galago_dbus_message_iter_get_string_or_nil(iter);
					break;

				default:
					galago_log_warning(
						"Unsupported type %d retrieved from message\n",
						dbus_message_iter_get_arg_type(iter));
					break;
			}
#undef CHECK_GET_VALUE
			break;
	}

	return retval;
}

static GalagoList *
galago_dbus_send_message_with_reply_list_vargs(const void *object,
											   const char *name,
											   GalagoList *return_types,
											   va_list args)
{
	DBusMessage *message;
	DBusMessage *reply;
	DBusMessageIter iter;
	DBusError error;
	GalagoList *ret_list = NULL, *l;

	galago_goto_if_fail(object != NULL,           exit);
	galago_goto_if_fail(name   != NULL,           exit);
	galago_goto_if_fail(*name  != '\0',           exit);
	galago_goto_if_fail(GALAGO_IS_OBJECT(object), exit);
	galago_goto_if_fail(return_types != NULL,     exit);

	if (!galago_is_connected())
		goto exit;

	message = galago_dbus_message_new_method_call_vargs(object, name,
														TRUE, args);

	galago_goto_if_fail(message != NULL, exit);

	dbus_error_init(&error);

	reply = dbus_connection_send_with_reply_and_block(
		galago_core_get_dbus_conn(), message, -1, &error);

	dbus_message_unref(message);

	if (dbus_error_is_set(&error))
	{
		if (!dbus_error_has_name(&error, GALAGO_DBUS_ERROR_OBJECT_NOT_FOUND))
		{
			galago_log_error("Error sending %s.%s: %s\n",
							 galago_class_get_name(GALAGO_OBJECT_CLASS(object)),
							 name, error.message);
		}

		goto exit;
	}

	dbus_message_iter_init(reply, &iter);

	for (l = return_types; l != NULL; l = l->next)
	{
		ret_list = galago_list_append(ret_list,
			get_ret_val_from_iter(&iter, (GalagoValue *)l->data));

		dbus_message_iter_next(&iter);
	}

	dbus_message_unref(reply);

exit:
	dbus_error_free(&error);

	for (l = return_types; l != NULL; l = l->next)
		galago_value_destroy((GalagoValue *)l->data);

	galago_list_destroy(return_types);

	return ret_list;
}

GalagoList *
galago_dbus_send_message_with_reply_list(const void *object, const char *name,
										 GalagoList *return_types, ...)
{
	va_list args;
	GalagoList *list;

	va_start(args, return_types);
	list = galago_dbus_send_message_with_reply_list_vargs(object, name,
														  return_types, args);
	va_end(args);

	return list;
}

void *
galago_dbus_send_message_with_reply(const void *object, const char *name,
									GalagoValue *return_type, ...)
{
	va_list args;
	void *retval = NULL;
	GalagoList *list;

	va_start(args, return_type);
	list = galago_dbus_send_message_with_reply_list_vargs(
		object, name, galago_list_append(NULL, return_type), args);
	va_end(args);

	if (list != NULL)
	{
		retval = list->data;

		galago_list_destroy(list);
	}

	return retval;
}

void
galago_dbus_object_push_full(void *object)
{
	GalagoObjectClass *klass;

	galago_return_if_fail(object != NULL);
	galago_return_if_fail(GALAGO_IS_OBJECT(object));

	klass = GALAGO_OBJECT_CLASS(object);

	if (klass->dbus_push_full != NULL)
		klass->dbus_push_full(object);
	else
	{
		galago_log_error("Class type %s passed to "
						 "galago_dbus_object_push_full does not "
						 "implement dbus_push_full!\n",
						 galago_class_get_name(klass));
	}
}

#define IS_VALID_DBUS_NAME_CHAR(c) \
	(((c) >= '0' && (c) <= '9') || \
	 ((c) >= 'A' && (c) <= 'Z') || \
	 ((c) >= 'a' && (c) <= 'z'))

const char *
galago_dbus_normalize_name(const char *name)
{
	static char buffer[BUFSIZ];
	const char *c;
	char *d;

	galago_return_val_if_fail(name != NULL, NULL);

	for (c = name, d = buffer; *c != '\0'; c++)
	{
		if (!IS_VALID_DBUS_NAME_CHAR(*c))
		{
			char escaped_c[9];
			snprintf(escaped_c, sizeof(escaped_c), "_0x%x_", *c);

			strncpy(d, escaped_c, BUFSIZ - (d - buffer));
			d += strlen(escaped_c);
		}
		else
		{
			*d++ = *c;
		}
	}

	*d = '\0';

	return buffer;
}
