/**
 * @file libgalago/galago-hashtable.c Hash Table API
 *
 * @Copyright (C) 2004-2005 Christian Hammond.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libgalago/galago-hashtable.h>
#include <libgalago/galago-assert.h>
#include <libgalago/galago-utils.h>
#include <string.h>
#include <stdlib.h>

#define HASH_TABLE_MIN_SIZE 11

typedef struct _GalagoBucket GalagoBucket;

struct _GalagoBucket
{
	void *key;
	void *value;

	GalagoBucket *next;
};

struct _GalagoHashTable
{
	size_t bucket_count;
	size_t item_count;

	GalagoBucket **buckets;
	size_t *bucket_counts;

	GalagoHashFunc hash_func;
	GalagoEqualFunc key_equal_func;
	GalagoFreeFunc destroy_key_func;
	GalagoFreeFunc destroy_value_func;
};

GalagoHashTable *
galago_hash_table_new(GalagoHashFunc hash_func,
					  GalagoEqualFunc key_equal_func)
{
	return galago_hash_table_new_full(hash_func, key_equal_func, NULL, NULL);
}

GalagoHashTable *
galago_hash_table_new_full(GalagoHashFunc hash_func,
						   GalagoEqualFunc key_equal_func,
						   GalagoFreeFunc destroy_key_func,
						   GalagoFreeFunc destroy_value_func)
{
	GalagoHashTable *hash_table;

	hash_table = galago_new0(GalagoHashTable, 1);

	hash_table->bucket_count       = HASH_TABLE_MIN_SIZE;
	hash_table->hash_func          = (hash_func == NULL
									  ? galago_direct_hash : hash_func);
	hash_table->key_equal_func     = (key_equal_func == NULL
									  ? galago_direct_equal : key_equal_func);
	hash_table->destroy_key_func   = destroy_key_func;
	hash_table->destroy_value_func = destroy_value_func;

	hash_table->buckets = galago_new0(GalagoBucket *, hash_table->bucket_count);
	hash_table->bucket_counts = galago_new0(size_t, hash_table->bucket_count);

	return hash_table;
}

void
galago_hash_table_destroy(GalagoHashTable *hash_table)
{
	GalagoBucket *bucket, *bucket_next = NULL;
	size_t i;

	galago_return_if_fail(hash_table != NULL);

	for (i = 0; i < hash_table->bucket_count; i++)
	{
		for (bucket = hash_table->buckets[i];
			 bucket != NULL;
			 bucket = bucket_next)
		{
			bucket_next = bucket->next;

			if (hash_table->destroy_key_func != NULL)
				hash_table->destroy_key_func(bucket->key);

			if (hash_table->destroy_value_func != NULL &&
				bucket->value != NULL)
			{
				hash_table->destroy_value_func(bucket->value);
			}

			free(bucket);
		}
	}

	free(hash_table->buckets);
	free(hash_table->bucket_counts);
	free(hash_table);
}

void
galago_hash_table_clear(GalagoHashTable *hash_table)
{
	GalagoBucket *bucket, *bucket_next = NULL;
	size_t i;

	galago_return_if_fail(hash_table != NULL);

	for (i = 0; i < hash_table->bucket_count; i++)
	{
		for (bucket = hash_table->buckets[i];
			 bucket != NULL;
			 bucket = bucket_next)
		{
			bucket_next = bucket->next;

			if (hash_table->destroy_key_func != NULL)
				hash_table->destroy_key_func(bucket->key);

			if (hash_table->destroy_value_func != NULL &&
				bucket->value != NULL)
			{
				hash_table->destroy_value_func(bucket->value);
			}
		}
	}
}

void
galago_hash_table_insert(GalagoHashTable *hash_table, void *key, void *value)
{
	GalagoBucket *bucket, *bucket_next = NULL;
	size_t index;

	galago_return_if_fail(hash_table != NULL);
	galago_return_if_fail(key        != NULL);

	index = hash_table->hash_func(key) % hash_table->bucket_count;

	if (hash_table->buckets[index] != NULL)
	{
		for (bucket = hash_table->buckets[index];
			 bucket != NULL;
			 bucket = bucket->next)
		{
			if (hash_table->key_equal_func(key, bucket->key))
			{
				if (hash_table->destroy_key_func != NULL)
					hash_table->destroy_key_func(key);

				if (hash_table->destroy_value_func != NULL &&
					bucket->value != value)
				{
					hash_table->destroy_value_func(bucket->value);
				}

				bucket->value = value;

				return;
			}

			bucket_next = hash_table->buckets[index];
		}
	}

	bucket = galago_new(GalagoBucket, 1);

	bucket->key   = key;
	bucket->value = value;
	bucket->next  = bucket_next;

	hash_table->buckets[index] = bucket;
	hash_table->bucket_counts[index]++;
	hash_table->item_count++;
}

void
galago_hash_table_replace(GalagoHashTable *hash_table, void *key, void *value)
{
	GalagoBucket *bucket, *bucket_next = NULL;
	size_t index;

	galago_return_if_fail(hash_table != NULL);
	galago_return_if_fail(key        != NULL);

	index = hash_table->hash_func(key) % hash_table->bucket_count;

	if (hash_table->buckets[index] != NULL)
	{
		for (bucket = hash_table->buckets[index];
			 bucket != NULL;
			 bucket = bucket->next)
		{
			if (hash_table->key_equal_func(key, bucket->key))
			{
				if (hash_table->destroy_key_func != NULL &&
					bucket->key != key)
				{
					hash_table->destroy_key_func(bucket->key);
				}

				if (hash_table->destroy_value_func != NULL &&
					bucket->value != value)
				{
					hash_table->destroy_value_func(bucket->value);
				}

				bucket->key   = key;
				bucket->value = value;

				return;
			}

			bucket_next = hash_table->buckets[index];
		}
	}

	bucket = galago_new(GalagoBucket, 1);

	bucket->key   = key;
	bucket->value = value;
	bucket->next  = bucket_next;

	hash_table->buckets[index] = bucket;
	hash_table->bucket_counts[index]++;
	hash_table->item_count++;
}

void
galago_hash_table_remove(GalagoHashTable *hash_table, const void *key)
{
	size_t index;
	GalagoBucket *bucket, *prev_bucket = NULL;

	galago_return_if_fail(hash_table != NULL);
	galago_return_if_fail(key        != NULL);

	index = hash_table->hash_func(key) % hash_table->bucket_count;

	for (bucket = hash_table->buckets[index];
		 bucket != NULL;
		 bucket = bucket->next)
	{
		if (hash_table->key_equal_func(key, bucket->key))
		{
			if (prev_bucket != NULL)
				prev_bucket->next = bucket->next;
			else
				hash_table->buckets[index] = bucket->next;

			hash_table->item_count--;
			hash_table->bucket_counts[index]--;

			if (hash_table->destroy_key_func != NULL)
				hash_table->destroy_key_func(bucket->key);

			if (hash_table->destroy_value_func != NULL &&
				bucket->value != NULL)
			{
				hash_table->destroy_value_func(bucket->value);
			}

			free(bucket);
		}

		prev_bucket = bucket;
	}
}

void *
galago_hash_table_lookup(const GalagoHashTable *hash_table, const void *key)
{
	size_t index;
	GalagoBucket *bucket;

	galago_return_val_if_fail(hash_table != NULL, NULL);
	galago_return_val_if_fail(key        != NULL, NULL);

	index = hash_table->hash_func(key) % hash_table->bucket_count;

	for (bucket = hash_table->buckets[index];
		 bucket != NULL;
		 bucket = bucket->next)
	{
		if (hash_table->key_equal_func(key, bucket->key))
			return bucket->value;
	}

	return NULL;
}

galago_bool
galago_hash_table_exists(const GalagoHashTable *hash_table, const void *key)
{
	galago_return_val_if_fail(hash_table != NULL, FALSE);
	galago_return_val_if_fail(key        != NULL, FALSE);

	return (galago_hash_table_lookup(hash_table, key) != NULL);
}

GalagoList *
galago_hash_table_get_keys(const GalagoHashTable *hash_table)
{
	GalagoList *list = NULL;
	GalagoBucket *bucket;
	size_t i;

	galago_return_val_if_fail(hash_table != NULL, NULL);

	for (i = 0; i < hash_table->bucket_count; i++)
	{
		for (bucket = hash_table->buckets[i];
			 bucket != NULL;
			 bucket = bucket->next)
		{
			list = galago_list_append(list, bucket->key);
		}
	}

	return list;
}

GalagoList *
galago_hash_table_get_values(const GalagoHashTable *hash_table)
{
	GalagoList *list = NULL;
	GalagoBucket *bucket;
	size_t i;

	galago_return_val_if_fail(hash_table != NULL, NULL);

	for (i = 0; i < hash_table->bucket_count; i++)
	{
		for (bucket = hash_table->buckets[i];
			 bucket != NULL;
			 bucket = bucket->next)
		{
			list = galago_list_append(list, bucket->value);
		}
	}

	return list;
}

void
galago_hash_table_foreach(const GalagoHashTable *hash_table,
						  GalagoHForEachFunc func, void *user_data)
{
	GalagoBucket *bucket;
	size_t i;

	galago_return_if_fail(hash_table != NULL);
	galago_return_if_fail(func       != NULL);

	for (i = 0; i < hash_table->bucket_count; i++)
	{
		for (bucket = hash_table->buckets[i];
			 bucket != NULL;
			 bucket = bucket->next)
		{
			func(bucket->key, bucket->value, user_data);
		}
	}
}

void
galago_hash_table_foreach_key(const GalagoHashTable *hash_table,
							  GalagoForEachFunc func, void *user_data)
{
	GalagoBucket *bucket;
	size_t i;

	galago_return_if_fail(hash_table != NULL);
	galago_return_if_fail(func       != NULL);

	for (i = 0; i < hash_table->bucket_count; i++)
	{
		for (bucket = hash_table->buckets[i];
			 bucket != NULL;
			 bucket = bucket->next)
		{
			func(bucket->key, user_data);
		}
	}
}

void
galago_hash_table_foreach_value(const GalagoHashTable *hash_table,
								GalagoForEachFunc func, void *user_data)
{
	GalagoBucket *bucket;
	size_t i;

	galago_return_if_fail(hash_table != NULL);
	galago_return_if_fail(func       != NULL);

	for (i = 0; i < hash_table->bucket_count; i++)
	{
		for (bucket = hash_table->buckets[i];
			 bucket != NULL;
			 bucket = bucket->next)
		{
			func(bucket->value, user_data);
		}
	}
}
size_t
galago_hash_table_get_size(const GalagoHashTable *hash_table)
{
	galago_return_val_if_fail(hash_table != NULL, 0);

	return hash_table->item_count;
}

galago_bool
galago_str_equal(const void *a, const void *b)
{
	if (a == NULL && b == NULL)
		return TRUE;

	return (a != NULL && b != NULL &&
			!strcmp((const char *)a, (const char *)b));
}

unsigned int
galago_str_hash(const void *str)
{
	const char *p = str;
	unsigned int h = *p;

	if (h != 0)
	{
		for (++p; *p != '\0'; p++)
			h = (h << 5) - h + *p;
	}

	return h;
}

galago_bool
galago_int_equal(const void *a, const void *b)
{
	int i1 = (int)a;
	int i2 = (int)b;

	return (i1 == i2);
}

unsigned int
galago_int_hash(const void *i)
{
	return (unsigned int)i;
}

galago_bool
galago_direct_equal(const void *a, const void *b)
{
	return (a == b);
}

unsigned int
galago_direct_hash(const void *ptr)
{
	return (unsigned int)ptr;
}
