/**
 * @file libgalago/galago-image.c Galago Image API
 *
 * @Copyright (C) 2004-2005 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libgalago/galago-image.h>
#include <libgalago/galago-assert.h>
#include <libgalago/galago-object-utils.h>
#include <libgalago/galago-utils.h>
#include <libgalago/galago-utils-priv.h>
#include <string.h>

struct _GalagoImagePrivate
{
	unsigned char *image_data;
	size_t image_length;
};


/**************************************************************************
 * Object/Class support
 **************************************************************************/
GALAGO_REGISTER_CLASS(galago_image, GalagoImage, NULL, NULL);
DEPRECATE_OLD_GET_CLASS(galago_images, galago_image);

static void
galago_image_object_init(GalagoImage *image)
{
	image->priv = galago_new0(GalagoImagePrivate, 1);
}

static void
galago_image_object_finalize(GalagoObject *object)
{
	GalagoImage *image = (GalagoImage *)object;

	if (image->priv->image_data != NULL)
		free(image->priv->image_data);

	free(image->priv);
}

static void
galago_image_class_init(GalagoObjectClass *klass)
{
	klass->finalize = galago_image_object_finalize;
}


/**************************************************************************
 * Image API
 **************************************************************************/
void
galago_image_set_image_data(GalagoImage *image,
							const unsigned char *data, size_t len)
{
	galago_return_if_fail(image != NULL);
	galago_return_if_fail(GALAGO_IS_IMAGE(image));
	galago_return_if_fail(data == NULL || len > 0);

	if (image->priv->image_data != NULL)
		free(image->priv->image_data);

	image->priv->image_data = (unsigned char *)malloc(len);
	memcpy(image->priv->image_data, data, len);

	image->priv->image_length = len;
}

void
galago_image_get_image_data(const GalagoImage *image,
							unsigned char **ret_data, size_t *ret_len)
{
	galago_return_if_fail(image != NULL);
	galago_return_if_fail(GALAGO_IS_IMAGE(image));
	galago_return_if_fail(ret_data != NULL || ret_len != NULL);

	if (ret_data != NULL)
		*ret_data = image->priv->image_data;

	if (ret_len != NULL)
		*ret_len = image->priv->image_length;
}
