/**
 * @file libgalago/galago-photo.h Galago Photo API
 *
 * @Copyright (C) 2004-2005 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#ifndef _GALAGO_PHOTO_H_
#define _GALAGO_PHOTO_H_

#include <stdlib.h>

typedef struct _GalagoPhoto        GalagoPhoto;
typedef struct _GalagoPhotoClass   GalagoPhotoClass;
typedef struct _GalagoPhotoPrivate GalagoPhotoPrivate;

#include <libgalago/galago-image.h>
#include <libgalago/galago-person.h>

/**
 * An photo for a user.
 *
 * The photo is a visual representation of the user, usually in
 * PNG, GIF, or JPEG format.
 */
struct _GalagoPhoto
{
	GalagoImage parent_object;

	GalagoPhotoPrivate *priv;

	void (*reserved_1)(void);
	void (*reserved_2)(void);
	void (*reserved_3)(void);
	void (*reserved_4)(void);
};

/**
 * GalagoPhoto's class.
 */
struct _GalagoPhotoClass
{
	GalagoImageClass parent_class;

	void (*reserved_1)(void);
	void (*reserved_2)(void);
	void (*reserved_3)(void);
	void (*reserved_4)(void);
};

#define GALAGO_CLASS_PHOTO (galago_photo_get_class())
#define GALAGO_DBUS_PHOTO_INTERFACE "org.freedesktop.Galago.Photo"
#define GALAGO_IS_PHOTO(obj) (GALAGO_IS_OBJECT(obj) && galago_object_check_cast((obj), GALAGO_CLASS_PHOTO))

#ifdef __cplusplus
extern "C" {
#endif

/**************************************************************************/
/** @name Photo API                                                       */
/**************************************************************************/
/*@{*/

/**
 * Returns the class for a GalagoPhoto.
 *
 * @return The GalagoPhoto class.
 */
GalagoObjectClass *galago_photo_get_class(void);

/**
 * Creates a new photo.
 *
 * @param person The person the photo is for.
 * @param data   The image data.
 * @param len    The image data length.
 *
 * @return The photo.
 */
GalagoPhoto *galago_photo_new(GalagoPerson *person,
							  const unsigned char *data, size_t len);

/**
 * Sets the photo's image data.
 *
 * @param photo  The photo.
 * @param data   The image data.
 * @param len    The image data length.
 */
void galago_photo_set_image_data(GalagoPhoto *photo,
								 const unsigned char *data, size_t len);

/**
 * Returns an photo's person.
 *
 * @param photo The photo.
 *
 * @return The person.
 */
GalagoPerson *galago_photo_get_person(const GalagoPhoto *photo);

/**
 * Returns an photo's image data.
 *
 * @param photo   The photo.
 * @param ret_data The returned image data.
 * @param ret_len  The returned image data length.
 */
void galago_photo_get_image_data(const GalagoPhoto *photo,
								 unsigned char **ret_data,
								 size_t *ret_len);

/*@}*/

#ifdef __cplusplus
}
#endif

#endif /* _GALAGO_PHOTO_H_ */
