/**
 * @file libgalago/galago-presence.c Galago Presence API
 *
 * @Copyright (C) 2004-2005 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libgalago/galago-presence.h>
#include <libgalago/galago-assert.h>
#include <libgalago/galago-context.h>
#include <libgalago/galago-core.h>
#include <libgalago/galago-marshal.h>
#include <libgalago/galago-object-utils.h>
#include <libgalago/galago-utils.h>
#include <libgalago/galago-utils-priv.h>
#include <stdio.h>
#include <string.h>

struct _GalagoPresencePrivate
{
	GalagoAccount *account;

	galago_bool idle;
	time_t idle_time;

	GalagoList *statuses;

	GalagoStatus *active_status;
};

static void _galago_presence_remove_status(GalagoPresence *presence,
										   GalagoStatus *status);

static unsigned int status_type_scores[] =
{
	0,     /* unset         */
	-500,  /* offline       */
	100,   /* available     */
	-50,   /* hidden        */
	-100,  /* away          */
	-200,  /* extended away */
};

/**************************************************************************
 * Object/Class support
 **************************************************************************/
GALAGO_REGISTER_CLASS(galago_presence, GalagoPresence, NULL,
					  GALAGO_DBUS_PRESENCE_INTERFACE);
DEPRECATE_OLD_GET_CLASS(galago_presences, galago_presence);

static void
galago_presence_object_init(GalagoPresence *presence)
{
	presence->priv = galago_new0(GalagoPresencePrivate, 1);
}

static void
galago_presence_object_finalize(GalagoObject *object)
{
	GalagoPresence *presence = (GalagoPresence *)object;
	GalagoAccount *account;

	account = galago_presence_get_account(presence);

	if (galago_account_get_presence(account, FALSE) == presence)
		galago_account_set_presence(account, NULL);

	if (presence->priv->statuses != NULL)
	{
		galago_list_foreach(presence->priv->statuses,
							(GalagoForEachFunc)galago_object_unref, NULL);
		galago_list_destroy(presence->priv->statuses);
	}

	free(presence->priv);
}

static void
galago_presence_dbus_message_append(DBusMessageIter *iter,
								   const GalagoObject *object)
{
	GalagoPresence *presence = (GalagoPresence *)object;
	galago_bool is_idle;
	dbus_uint32_t idle_time;

	galago_dbus_message_iter_append_object(iter,
		galago_presence_get_account(presence));

	is_idle = galago_presence_is_idle(presence);
	galago_dbus_message_iter_append_boolean(iter, is_idle);

	idle_time = galago_presence_get_idle_time(presence);
	galago_dbus_message_iter_append_uint32(iter, idle_time);

	galago_dbus_message_iter_append_object_list(iter,
		galago_presence_get_statuses(presence));
}

static void *
galago_presence_dbus_message_get(DBusMessageIter *iter)
{
	GalagoPresence *presence;
	GalagoAccount *account;
	const GalagoList *l, *statuses;
	galago_bool idle;
	time_t idle_time = 0;

	account = galago_dbus_message_iter_get_object(iter, GALAGO_CLASS_ACCOUNT);
	dbus_message_iter_next(iter);

	galago_dbus_message_iter_get_boolean(iter, idle);
	dbus_message_iter_next(iter);
	galago_dbus_message_iter_get_uint32(iter, idle_time);
	dbus_message_iter_next(iter);

	presence = galago_presence_new(account);
	galago_presence_set_idle(presence, idle, idle_time);

	statuses = galago_dbus_message_iter_get_object_list(iter,
														GALAGO_CLASS_STATUS);

	for (l = statuses; l != NULL; l = l->next)
		galago_presence_add_status(presence, (GalagoStatus *)l->data);

	return presence;
}


static void
galago_presence_class_init(GalagoObjectClass *klass)
{
	klass->finalize            = galago_presence_object_finalize;
	klass->dbus_message_append = galago_presence_dbus_message_append;
	klass->dbus_message_get    = galago_presence_dbus_message_get;

	galago_signal_register(klass->signal_context, "idle-changed",
						   galago_marshal_VOID__BOOLEAN_ULONG, 2,
						   GALAGO_TYPE_BOOLEAN,
						   GALAGO_TYPE_ULONG);
	galago_signal_register(klass->signal_context, "status-added",
						   galago_marshal_VOID__POINTER, 1,
						   GALAGO_TYPE_OBJECT);
	galago_signal_register(klass->signal_context, "status-removed",
						   galago_marshal_VOID__POINTER, 1,
						   GALAGO_TYPE_OBJECT);
	galago_signal_register(klass->signal_context, "status-updated",
						   galago_marshal_VOID__POINTER_STRING, 2,
						   GALAGO_TYPE_OBJECT,
						   GALAGO_TYPE_STRING);
	galago_signal_register(klass->signal_context, "updated",
						   galago_marshal_VOID, 0);
}


/**************************************************************************
 * GalagoPresence API
 **************************************************************************/
GalagoPresence *
galago_presence_new(GalagoAccount *account)
{
	GalagoPresence *presence;

	galago_return_val_if_fail(galago_is_initted(),        NULL);
	galago_return_val_if_fail(account != NULL,            NULL);
	galago_return_val_if_fail(GALAGO_IS_ACCOUNT(account), NULL);

	presence = galago_account_get_presence(account, FALSE);

	if (presence == NULL)
	{
		const char *obj_prefix;

		galago_context_push(galago_object_get_context(account));
		presence = galago_object_new(GALAGO_CLASS_PRESENCE);
		galago_context_pop();

		presence->priv->account = account;

		if ((obj_prefix = galago_object_get_dbus_path(account)) != NULL)
		{
			size_t len;
			char *obj_path;

			len = strlen(obj_prefix) + strlen("/presence") + 1;

			obj_path = galago_new(char, len);
			snprintf(obj_path, len, "%s/presence", obj_prefix);

			galago_object_set_dbus_path(presence, obj_path);

			free(obj_path);
		}

		galago_account_set_presence(account, presence);
	}

	return presence;
}

void
galago_presence_set_idle(GalagoPresence *presence, galago_bool idle,
						 time_t idle_time)
{
	GalagoPerson *person;
	GalagoAccount *account;

	galago_return_if_fail(presence != NULL);
	galago_return_if_fail(GALAGO_IS_PRESENCE(presence));

	if (presence->priv->idle == idle &&
		presence->priv->idle_time == idle_time)
	{
		return;
	}

	presence->priv->idle = idle;

	presence->priv->idle_time = (idle ? idle_time : 0);

	account = galago_presence_get_account(presence);
	person  = galago_account_get_person(account);

	if (galago_person_is_native(person) && galago_is_connected() &&
		galago_core_is_feed())
	{
		galago_dbus_send_message(presence, "SetIdle",
			galago_value_new(GALAGO_TYPE_BOOLEAN, &idle, NULL),
			galago_value_new(GALAGO_TYPE_ULONG, &idle_time, NULL),
			NULL);
	}

	galago_signal_emit(presence, "idle-changed", idle, idle_time);
	galago_signal_emit(presence, "updated");
}

void
galago_presence_set_statuses(GalagoPresence *presence, GalagoList *statuses)
{
	const GalagoList *l;
	GalagoAccount *account;

	galago_return_if_fail(presence != NULL);
	galago_return_if_fail(GALAGO_IS_PRESENCE(presence));

	if (presence->priv->statuses == statuses)
		return;

	if (presence->priv->statuses != NULL)
	{
		for (l = presence->priv->statuses; l != NULL; l = l->next)
		{
			GalagoStatus *status = (GalagoStatus *)l->data;

			galago_signal_emit(presence, "status-removed", status);

			galago_object_unref(status);
		}

		galago_list_destroy(presence->priv->statuses);

		presence->priv->active_status = NULL;
	}

	presence->priv->statuses = statuses;

	account = galago_presence_get_account(presence);

	if (galago_person_is_native(galago_account_get_person(account)) &&
		galago_is_connected() && galago_core_is_feed())
	{
		galago_dbus_send_message(presence, "SetStatuses",
			galago_value_new_list(GALAGO_TYPE_OBJECT,
								  galago_presence_get_statuses(presence),
								  GALAGO_CLASS_STATUS),
			NULL);
	}

	for (l = statuses; l != NULL; l = l->next)
		galago_signal_emit(presence, "status-added", l->data);

	galago_signal_emit(presence, "updated");
}

void
galago_presence_add_status(GalagoPresence *presence, GalagoStatus *status)
{
	GalagoAccount *account;
	GalagoPerson *person;
	const char *status_id;

	galago_return_if_fail(presence != NULL);
	galago_return_if_fail(status   != NULL);
	galago_return_if_fail(GALAGO_IS_PRESENCE(presence));
	galago_return_if_fail(GALAGO_IS_STATUS(status));

	status_id = galago_status_get_id(status);

	account = galago_presence_get_account(presence);
	person  = galago_account_get_person(account);

	if (galago_presence_get_status(presence, status_id) != NULL)
	{
		GalagoAccount *account = galago_presence_get_account(presence);

		galago_log_warning("A status with ID %s has already been added "
						   "to the presence for account %s\n",
						   status_id, galago_account_get_username(account));

		galago_object_unref(status);

		return;
	}

	if (galago_person_is_native(person) && galago_is_connected())
	{
		char *obj_path;
		size_t len;

		len = strlen(galago_object_get_dbus_path(presence)) +
		      strlen(status_id) + 2;

		obj_path = galago_new(char, len);
		snprintf(obj_path, len, "%s/%s",
				 galago_object_get_dbus_path(presence), status_id);

		galago_object_set_dbus_path(status, obj_path);

		free(obj_path);
	}

	if (galago_status_is_exclusive(status))
	{
		GalagoStatus *active_status;

		active_status = galago_presence_get_active_status(presence);

		if (active_status != NULL)
			_galago_presence_remove_status(presence, active_status);

		presence->priv->active_status = status;
	}

	presence->priv->statuses = galago_list_append(presence->priv->statuses,
												  status);

	galago_status_set_presence(status, presence);

	if (galago_person_is_native(person) && galago_is_connected() &&
		galago_core_is_feed())
	{
		galago_dbus_send_message(presence, "AddStatus",
			galago_value_new(GALAGO_TYPE_OBJECT, &status, GALAGO_CLASS_STATUS),
			NULL);
	}

	galago_signal_emit(presence, "status-added", status);
	galago_signal_emit(presence, "updated");
}

static void
_galago_presence_remove_status(GalagoPresence *presence, GalagoStatus *status)
{
	GalagoAccount *account;

	galago_return_if_fail(presence != NULL);
	galago_return_if_fail(status   != NULL);
	galago_return_if_fail(GALAGO_IS_PRESENCE(presence));
	galago_return_if_fail(GALAGO_IS_STATUS(status));

	presence->priv->statuses = galago_list_remove(presence->priv->statuses,
												  status);

	account = galago_presence_get_account(presence);

	if (galago_person_is_native(galago_account_get_person(account)) &&
		!galago_status_is_exclusive(status) && galago_is_connected() &&
		galago_core_is_feed())
	{
		const char *id = galago_status_get_id(status);

		galago_dbus_send_message(presence, "RemoveStatus",
			galago_value_new(GALAGO_TYPE_STRING, &id, NULL),
			NULL);
	}

	galago_signal_emit(presence, "status-removed", status);
	galago_signal_emit(presence, "updated");

	galago_object_unref(status);
}

void
galago_presence_remove_status(GalagoPresence *presence, const char *status_id)
{
	GalagoStatus *status;

	galago_return_if_fail(presence  != NULL);
	galago_return_if_fail(status_id != NULL && *status_id != '\0');
	galago_return_if_fail(GALAGO_IS_PRESENCE(presence));

	status = galago_presence_get_status(presence, status_id);

	if (status == NULL)
	{
		GalagoAccount *account = galago_presence_get_account(presence);

		galago_log_warning("Attempted to remove an unknown status %s from "
						   "the presence for account %s\n",
						   status_id, galago_account_get_username(account));

		return;
	}

	galago_return_if_fail(!galago_status_is_exclusive(status));

	_galago_presence_remove_status(presence, status);
}

void
galago_presence_clear_statuses(GalagoPresence *presence)
{
	galago_return_if_fail(presence != NULL);
	galago_return_if_fail(GALAGO_IS_PRESENCE(presence));

	galago_presence_set_statuses(presence, NULL);
}

GalagoAccount *
galago_presence_get_account(const GalagoPresence *presence)
{
	galago_return_val_if_fail(presence != NULL,             NULL);
	galago_return_val_if_fail(GALAGO_IS_PRESENCE(presence), NULL);

	return presence->priv->account;
}

galago_bool
galago_presence_is_idle(const GalagoPresence *presence)
{
	galago_return_val_if_fail(presence != NULL,             FALSE);
	galago_return_val_if_fail(GALAGO_IS_PRESENCE(presence), FALSE);

	return presence->priv->idle;
}

time_t
galago_presence_get_idle_time(const GalagoPresence *presence)
{
	galago_return_val_if_fail(presence != NULL,             0);
	galago_return_val_if_fail(GALAGO_IS_PRESENCE(presence), 0);

	return presence->priv->idle_time;
}

galago_bool
galago_presence_is_discarded(const GalagoPresence *presence)
{
	galago_return_val_if_fail(presence != NULL,             TRUE);
	galago_return_val_if_fail(GALAGO_IS_PRESENCE(presence), TRUE);

	return (galago_presence_get_statuses(presence) == NULL);
}

galago_bool
galago_presence_is_available(const GalagoPresence *presence)
{
	GalagoStatus *status;

	galago_return_val_if_fail(presence != NULL,             FALSE);
	galago_return_val_if_fail(GALAGO_IS_PRESENCE(presence), FALSE);

	status = galago_presence_get_active_status(presence);

	return ((status != NULL && galago_status_is_available(status)) &&
			!galago_presence_is_idle(presence));
}

const GalagoList *
galago_presence_get_statuses(const GalagoPresence *presence)
{
	galago_return_val_if_fail(presence != NULL,             NULL);
	galago_return_val_if_fail(GALAGO_IS_PRESENCE(presence), NULL);

	return presence->priv->statuses;
}

GalagoStatus *
galago_presence_get_active_status(const GalagoPresence *presence)
{
	galago_return_val_if_fail(presence != NULL,             NULL);
	galago_return_val_if_fail(GALAGO_IS_PRESENCE(presence), NULL);

	return presence->priv->active_status;
}

galago_bool
galago_presence_is_status_exclusive(const GalagoPresence *presence,
									const char *status_id)
{
	GalagoStatus *status;

	galago_return_val_if_fail(presence  != NULL,                       FALSE);
	galago_return_val_if_fail(status_id != NULL && *status_id != '\0', FALSE);
	galago_return_val_if_fail(GALAGO_IS_PRESENCE(presence),            FALSE);

	status = galago_presence_get_status(presence, status_id);

	if (status == NULL)
		return FALSE;

	return galago_status_is_exclusive(status);
}

GalagoStatus *
galago_presence_get_status(const GalagoPresence *presence,
						   const char *status_id)
{
	const GalagoList *l;

	galago_return_val_if_fail(presence  != NULL,            NULL);
	galago_return_val_if_fail(status_id != NULL,            NULL);
	galago_return_val_if_fail(GALAGO_IS_PRESENCE(presence), NULL);

	for (l = galago_presence_get_statuses(presence);
		 l != NULL;
		 l = l->next)
	{
		GalagoStatus *status = (GalagoStatus *)l->data;

		if (!strcmp(galago_status_get_id(status), status_id))
			return status;
	}

	return NULL;
}

galago_bool
galago_presence_has_status(const GalagoPresence *presence,
						   const char *status_id)
{
	galago_return_val_if_fail(presence  != NULL,            FALSE);
	galago_return_val_if_fail(status_id != NULL,            FALSE);
	galago_return_val_if_fail(GALAGO_IS_PRESENCE(presence), FALSE);

	return (galago_presence_get_status(presence, status_id) != NULL);
}

galago_bool
galago_presence_has_status_type(const GalagoPresence *presence,
								GalagoStatusType type)
{
	const GalagoList *l;

	galago_return_val_if_fail(presence != NULL,             FALSE);
	galago_return_val_if_fail(GALAGO_IS_PRESENCE(presence), FALSE);
	galago_return_val_if_fail(type != GALAGO_STATUS_UNSET,  FALSE);

	for (l = galago_presence_get_statuses(presence);
		 l != NULL;
		 l = l->next)
	{
		GalagoStatus *status = (GalagoStatus *)l->data;

		if (galago_status_get_type(status) == type)
			return TRUE;
	}

	return FALSE;
}

int
galago_presence_compare(const GalagoPresence *presence1,
						const GalagoPresence *presence2)
{
	int score1 = 0, score2 = 0;
	const GalagoList *l;

	galago_return_val_if_fail(presence1 == NULL ||
							  GALAGO_IS_PRESENCE(presence1), 1);
	galago_return_val_if_fail(presence2 == NULL ||
							  GALAGO_IS_PRESENCE(presence2), -1);

	if (presence1 == presence2)
		return 0;
	else if (presence1 == NULL)
		return 1;
	else if (presence2 == NULL)
		return -1;

	/* Compute the score of the first set of statuses. */
	for (l = galago_presence_get_statuses(presence1); l != NULL; l = l->next)
	{
		GalagoStatus *status = (GalagoStatus *)l->data;
		score1 += status_type_scores[galago_status_get_type(status)];
	}

	/* Compute the score of the second set of statuses. */
	for (l = galago_presence_get_statuses(presence2); l != NULL; l = l->next)
	{
		GalagoStatus *status = (GalagoStatus *)l->data;
		score2 += status_type_scores[galago_status_get_type(status)];
	}

	if (score1 > score2)
		return 1;
	else if (score1 < score2)
		return -1;
	else
	{
		galago_bool idle1 = galago_presence_is_idle(presence1);
		galago_bool idle2 = galago_presence_is_idle(presence2);

		if (!idle1 && !idle2)
			return 0;
		else if (idle1 && !idle2)
			return 1;
		else if (!idle1 && idle2)
			return -1;
		else
		{
			time_t idle_time_1;
			time_t idle_time_2;

			idle_time_1 = galago_presence_get_idle_time(presence1);
			idle_time_2 = galago_presence_get_idle_time(presence2);

			if (idle_time_1 < idle_time_2)
				return 1;
			else if (idle_time_1 > idle_time_2)
				return -1;
			else
				return 0;
		}
	}
}
