/**
 * @file libgalago/galago-signals.c Galago Signals API
 *
 * @Copyright (C) 2004-2005 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libgalago/galago-signals.h>
#include <libgalago/galago-assert.h>
#include <libgalago/galago-core.h>
#include <libgalago/galago-utils.h>
#include <libgalago/galago-utils-priv.h>
#include <string.h>

struct _GalagoSignalHandler
{
	size_t id;

	GalagoSignalContext *signal_context;
	void *object;
	char *signal;
	GalagoCallback cb;
	void *user_data;
	size_t ref_count;
};

struct _GalagoSignalContext
{
	GalagoHashTable *signals;

	size_t freeze_count;
};

typedef struct
{
	GalagoMarshalFunc marshal;
	int num_values;
	GalagoType *value_types;

	GalagoList *handlers;

} GalagoSignalData;

static GalagoHashTable *signal_handlers = NULL;
static galago_bool initted = FALSE;
static size_t last_signal_id = 0;


/**************************************************************************
 * Property Watch API
 **************************************************************************/
static void galago_signal_handler_ref(GalagoSignalHandler *signal_handler);
static void galago_signal_handler_unref(GalagoSignalHandler *signal_handler);

static GalagoSignalHandler *
galago_signal_handler_new(GalagoSignalContext *signal_context, void *obj,
						  const char *signal, GalagoCallback cb,
						  void *user_data)
{
	GalagoSignalHandler *signal_handler;
	GalagoSignalData *signal_data;

	galago_return_val_if_fail(signal_context != NULL, NULL);
	galago_return_val_if_fail(signal         != NULL, NULL);
	galago_return_val_if_fail(cb             != NULL, NULL);

	signal_handler = galago_new0(GalagoSignalHandler, 1);
	signal_handler->id             = ++last_signal_id;
	signal_handler->signal_context = signal_context;
	signal_handler->object         = obj;
	signal_handler->cb             = cb;
	signal_handler->user_data      = user_data;
	signal_handler->signal         = strdup(signal);
	signal_handler->ref_count      = 1;

	galago_hash_table_insert(signal_handlers,
							 (void *)signal_handler->id,
							 signal_handler);

	signal_data = galago_hash_table_lookup(signal_context->signals,
										   signal_handler->signal);

	galago_return_val_if_fail(signal_data != NULL, FALSE);

	signal_data->handlers = galago_list_append(signal_data->handlers,
											   signal_handler);

	return signal_handler;
}

static void
galago_signal_handler_destroy(GalagoSignalHandler *signal_handler)
{
	GalagoSignalData *signal_data;

	galago_return_if_fail(signal_handler != NULL);

	if (signal_handler->ref_count > 0)
	{
		galago_signal_handler_unref(signal_handler);
		return;
	}

	signal_data = galago_hash_table_lookup(
		signal_handler->signal_context->signals, signal_handler->signal);

	if (signal_data != NULL)
	{
		signal_data->handlers = galago_list_remove(signal_data->handlers,
												   signal_handler);
	}

	galago_hash_table_remove(signal_handlers, (void *)signal_handler->id);

	if (signal_handler->signal != NULL)
		free(signal_handler->signal);

	free(signal_handler);
}

static void
galago_signal_handler_ref(GalagoSignalHandler *signal_handler)
{
	galago_return_if_fail(signal_handler != NULL);

	signal_handler->ref_count++;
}

static void
galago_signal_handler_unref(GalagoSignalHandler *signal_handler)
{
	galago_return_if_fail(signal_handler != NULL);

	signal_handler->ref_count--;

	if (signal_handler->ref_count == 0)
		galago_signal_handler_destroy(signal_handler);
}

static void
destroy_signal(GalagoSignalData *signal_data)
{
	galago_list_foreach(signal_data->handlers,
						(GalagoForEachFunc)galago_signal_handler_destroy, NULL);
	galago_list_destroy(signal_data->handlers);

	if (signal_data->value_types != NULL)
		free(signal_data->value_types);
}

/**************************************************************************
 * Signal Context API
 **************************************************************************/
static void
galago_signal_context_remove_handler(GalagoSignalContext *signal_context,
									 GalagoSignalHandler *signal_handler)
{
	GalagoSignalData *signal_data;

	galago_return_if_fail(signal_context != NULL);
	galago_return_if_fail(signal_handler != NULL);

	signal_data = galago_hash_table_lookup(signal_context->signals,
										   signal_handler->signal);

	galago_return_if_fail(signal_data != NULL);

	galago_signal_handler_unref(signal_handler);
}

GalagoSignalContext *
galago_signal_context_new(void)
{
	GalagoSignalContext *signal_context;

	signal_context = galago_new0(GalagoSignalContext, 1);

	signal_context->signals =
		galago_hash_table_new_full(galago_str_hash, galago_str_equal,
								   free, (GalagoFreeFunc)destroy_signal);

	return signal_context;
}

void
galago_signal_context_destroy(GalagoSignalContext *signal_context)
{
	galago_return_if_fail(signal_context != NULL);

	galago_hash_table_destroy(signal_context->signals);

	free(signal_context);
}

void
galago_signal_context_freeze(GalagoSignalContext *signal_context)
{
	galago_return_if_fail(signal_context != NULL);

	signal_context->freeze_count++;
}

void
galago_signal_context_thaw(GalagoSignalContext *signal_context)
{
	galago_return_if_fail(signal_context != NULL);

	signal_context->freeze_count--;
}

galago_bool
galago_signal_context_is_frozen(const GalagoSignalContext *signal_context)
{
	galago_return_val_if_fail(signal_context != NULL, FALSE);

	return (signal_context->freeze_count > 0);
}

GalagoSignalHandler *
galago_signal_context_find_handler(const GalagoSignalContext *signal_context,
								   const void *object, const char *signal,
								   GalagoCallback cb)
{
	GalagoSignalData *signal_data;
	GalagoList *l;

	galago_return_val_if_fail(signal_context != NULL, NULL);
	galago_return_val_if_fail(signal         != NULL, NULL);
	galago_return_val_if_fail(cb             != NULL, NULL);

	signal_data = galago_hash_table_lookup(signal_context->signals, signal);

	galago_return_val_if_fail(signal_data != NULL, NULL);

	for (l = signal_data->handlers; l != NULL; l = l->next)
	{
		GalagoSignalHandler *signal_handler = (GalagoSignalHandler *)l->data;

		if (signal_handler->object == object && signal_handler->cb == cb)
			return signal_handler;
	}

	return NULL;
}

GalagoSignalHandler *
galago_signal_context_find_handler_with_id(unsigned long id)
{
	galago_return_val_if_fail(id > 0, NULL);

	return (GalagoSignalHandler *)galago_hash_table_lookup(signal_handlers,
														   (void *)id);
}


/**************************************************************************
 * Signals Subsystem API
 **************************************************************************/
galago_bool
galago_signal_register(GalagoSignalContext *signal_context,
					   const char *signal, GalagoMarshalFunc marshal,
					   size_t num_values, ...)
{
	GalagoSignalData *signal_data;

	galago_return_val_if_fail(signal_context != NULL, FALSE);
	galago_return_val_if_fail(signal         != NULL, FALSE);
	galago_return_val_if_fail(marshal        != NULL, FALSE);

	if (galago_hash_table_exists(signal_context->signals, signal))
		return FALSE;

	signal_data = galago_new0(GalagoSignalData, 1);

	signal_data->marshal    = marshal;
	signal_data->num_values = num_values;

	if (num_values > 0)
	{
		va_list args;
		int i;

		signal_data->value_types = galago_new0(GalagoType, num_values);

		va_start(args, num_values);

		for (i = 0; i < num_values; i++)
			signal_data->value_types[i] = va_arg(args, GalagoType);

		va_end(args);
	}

	galago_hash_table_insert(signal_context->signals,
							 strdup(signal), signal_data);

	return TRUE;
}

void
galago_signal_unregister(GalagoSignalContext *signal_context,
						 const char *signal)
{
	galago_return_if_fail(signal_context != NULL);
	galago_return_if_fail(signal         != NULL);

	galago_hash_table_remove(signal_context->signals, signal);
}

static unsigned long
_galago_signal_connect_common(GalagoSignalContext *signal_context, void *obj,
							  const char *signal, GalagoCallback cb,
							  void *user_data)
{
	GalagoSignalHandler *signal_handler;

	signal_handler = galago_signal_handler_new(signal_context, obj, signal,
											   cb, user_data);

	if (obj != NULL)
		galago_object_set_watch(obj, TRUE);
	else
		galago_core_set_watch_all(TRUE);

	return signal_handler->id;
}

unsigned long
galago_signal_connect(void *obj, const char *signal, GalagoCallback cb,
					  void *user_data)
{
	GalagoSignalContext *signal_context;

	galago_return_val_if_fail(obj     != NULL, 0);
	galago_return_val_if_fail(signal  != NULL, 0);
	galago_return_val_if_fail(*signal != '\0', 0);
	galago_return_val_if_fail(cb      != NULL, 0);
	galago_return_val_if_fail(GALAGO_IS_OBJECT(obj), 0);

	signal_context = galago_class_get_signal_context(GALAGO_OBJECT_CLASS(obj));

	return _galago_signal_connect_common(signal_context, obj, signal, cb,
										 user_data);
}

unsigned long
galago_signal_connect_class(GalagoObjectClass *klass, const char *signal,
							GalagoCallback cb, void *user_data)
{
	GalagoSignalContext *signal_context;

	galago_return_val_if_fail(klass   != NULL, 0);
	galago_return_val_if_fail(signal  != NULL, 0);
	galago_return_val_if_fail(*signal != '\0', 0);
	galago_return_val_if_fail(cb      != NULL, 0);

	signal_context = galago_class_get_signal_context(klass);

	return _galago_signal_connect_common(signal_context, NULL, signal, cb,
										 user_data);
}

void
galago_signals_disconnect_by_id(unsigned long id)
{
	GalagoSignalHandler *signal_handler;

	galago_return_if_fail(id > 0);

	signal_handler = galago_signal_context_find_handler_with_id(id);

	galago_return_if_fail(signal_handler != NULL);

	galago_signal_context_remove_handler(signal_handler->signal_context,
										 signal_handler);
}

void
galago_signal_emit(void *object, const char *signal, ...)
{
	GalagoObjectClass *klass;
	GalagoSignalContext *signal_context;
	va_list args;

	galago_return_if_fail(object != NULL);
	galago_return_if_fail(signal != NULL);
	galago_return_if_fail(GALAGO_IS_OBJECT(object));

	klass = GALAGO_OBJECT_CLASS(object);
	signal_context = galago_class_get_signal_context(klass);

	if (galago_signal_context_is_frozen(signal_context))
		return;

	va_start(args, signal);
	galago_signal_emit_vargs(object, signal, args);
	va_end(args);
}

void
galago_signal_emit_vargs(void *object, const char *signal, va_list args)
{
	GalagoSignalData *signal_data;
	GalagoSignalHandler *signal_handler = NULL;
	GalagoObjectClass *klass;
	GalagoSignalContext *signal_context;
	GalagoList *l;
	va_list tmp;

	galago_return_if_fail(object         != NULL);
	galago_return_if_fail(signal         != NULL);
	galago_return_if_fail(GALAGO_IS_OBJECT(object));

	klass = GALAGO_OBJECT_CLASS(object);
	signal_context = galago_class_get_signal_context(klass);

	if (galago_signal_context_is_frozen(signal_context))
		return;

	signal_data = galago_hash_table_lookup(signal_context->signals, signal);

	if (signal_data == NULL)
	{
		galago_log_error(
			"The signal '%s' for class '%s' was not registered. "
		    "This is probably a bug in libgalago. Please report this.\n",
			signal, galago_class_get_name(klass));

		return;
	}

	for (l = signal_data->handlers; l != NULL; l = l->next)
	{
		if (signal_handler != NULL)
			galago_signal_handler_unref(signal_handler);

		signal_handler = (GalagoSignalHandler *)l->data;
		galago_signal_handler_ref(signal_handler);

		GALAGO_VA_COPY(tmp, args);

		if (signal_handler->object == NULL || signal_handler->object == object)
		{
			signal_data->marshal(signal_handler->cb, object, signal, tmp,
								 signal_handler->user_data);
		}

		va_end(tmp);
	}

	if (signal_handler != NULL)
		galago_signal_handler_unref(signal_handler);
}

void
galago_signals_init(void)
{
	galago_return_if_fail(!initted);

	last_signal_id = 0;
	signal_handlers = galago_hash_table_new(galago_int_hash, galago_int_equal);

	initted = TRUE;
}

void
galago_signals_uninit(void)
{
	galago_return_if_fail(initted);

	galago_hash_table_destroy(signal_handlers);
	signal_handlers = NULL;

	initted = FALSE;
}
