/*******************************************************************************************************************************************
 cxml.c
*******************************************************************************************************************************************/

#include "cxml.h"
#include "cserialized.h"

//------------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//------------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_CAPSULE_METACLASS (CXMLDocument);

//------------------------------------------------------------------------------------------------------------------------------------------
// xml document constructor
//------------------------------------------------------------------------------------------------------------------------------------------
CXMLDocument::CXMLDocument     (const CString &inFileName, const int inMode, const CString &inRoot) THROWABLE
	     :m_XMLDocument    (NULL),
	      m_XMLCurrentNode (NULL),
	      m_XMLMode	       (inMode),
	      m_XMLFileName    (inFileName),
	      m_XMLElementNode (NULL)
{ 
	// file load requested ?
	if (inMode & XML_READ)
	{
		// does the file exist ?
		if (!CFile::Exists (inFileName))
			throw new CException ("file \"" + inFileName + "\" does not exist.", __exception(XMLPARSE));
	
		// parse the document
		m_XMLDocument = ::xmlParseFile (inFileName.Get());

		// be sure it was a xml one...
		if (m_XMLDocument == NULL)
			throw new CException (CString("xml document not parsed successfully."), __exception(XMLPARSE));

		// get the root node
		m_XMLCurrentNode = ::xmlDocGetRootElement (m_XMLDocument);
	
		// root type check in if requested
		if (inRoot != CString())
			if (xmlStrcmp (m_XMLCurrentNode -> name, reinterpret_cast <const xmlChar *> (inRoot.Get()))) 
				throw new CException ("wrong xml document type, requested root \"" + inRoot + "\" differs from root \"" + 
						      CString((char *)m_XMLCurrentNode -> name) + "\" document.", __exception(XMLTYPE));
	}
	// new file requested ?
	else if (inMode & XML_WRITE)
	{
		// check the root node name
		if (inRoot == CString())
			throw new CException (CString("root node cannot be null."), __exception(XMLTYPE));

		// create a new doc
		m_XMLDocument = ::xmlNewDoc (reinterpret_cast <const xmlChar *> ("1.0"));

		// be sure it has been created...
		if (m_XMLDocument == NULL)
			throw new CException (CString("xml document not created successfully."), __exception(XMLPARSE));

		// create the xml root element
		m_XMLCurrentNode = ::xmlNewNode (NULL, reinterpret_cast <const xmlChar *> (inRoot.Get()));

		// set the xml root element
		::xmlDocSetRootElement (m_XMLDocument, m_XMLCurrentNode);
	}
}

//------------------------------------------------------------------------------------------------------------------------------------------
// xml document destructor
//------------------------------------------------------------------------------------------------------------------------------------------
CXMLDocument::~CXMLDocument ()
{
	// file access analyse
	if (m_XMLMode & XML_WRITE)
	{
//		::xmlThrDefIndentTreeOutput (1);
//		::xmlKeepBlanksDefault (0);
		::xmlSaveFormatFileEnc (m_XMLFileName.Get(), m_XMLDocument, "UTF-8", 1);
	}

	// free doc
	if (m_XMLDocument != NULL) ::xmlFreeDoc (m_XMLDocument);

	// local hierarchy (used for serialization purpose) ?
	if (m_XMLElementNode != NULL) delete m_XMLElementNode;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// document root access
//------------------------------------------------------------------------------------------------------------------------------------------
CXMLElementNode * CXMLDocument::GetRootElement ()
{
	// get the root node
	m_XMLCurrentNode = ::xmlDocGetRootElement (m_XMLDocument);

	// allocate the output root node
	CXMLElementNode *outXMLElementNode = new CXMLElementNode (NULL, new CXMLElement (m_XMLCurrentNode));

	// build the tree
	CXMLDocument::BuildXMLNodes (outXMLElementNode, m_XMLCurrentNode->children);

	// ok
	return outXMLElementNode;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// xml access mode
//------------------------------------------------------------------------------------------------------------------------------------------
int CXMLDocument::GetMode () const
{
	return m_XMLMode;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// recursive xml tree build 
//------------------------------------------------------------------------------------------------------------------------------------------
void CXMLDocument::BuildXMLNodes (CXMLElementNode *inParent, xmlNodePtr inXMLNodePtr)
{
	for (xmlNodePtr curXMLNodePtr=inXMLNodePtr; curXMLNodePtr; curXMLNodePtr=curXMLNodePtr->next)
	{
		if (curXMLNodePtr->type == XML_ELEMENT_NODE)
		{
			CXMLElement *newXMLElement = new CXMLElement (curXMLNodePtr);
			CXMLElementNode *newXMLElementNode = new CXMLElementNode (inParent, newXMLElement);
			newXMLElement -> m_XMLElementNode = newXMLElementNode;
			CXMLDocument::BuildXMLNodes (newXMLElementNode, curXMLNodePtr->children);
		}
	}
}

//------------------------------------------------------------------------------------------------------------------------------------------
// xml serialization
//------------------------------------------------------------------------------------------------------------------------------------------
CXMLDocument & CXMLDocument::operator << (CSerialized &inSerialized)
{
	if (m_XMLElementNode == NULL)
		m_XMLElementNode = GetRootElement ();
	CXMLElementNode *outXMLElementNode = m_XMLElementNode;
	inSerialized.Serialize (outXMLElementNode, XML_WRITE);
	return *this;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// xml serialization
//------------------------------------------------------------------------------------------------------------------------------------------
CXMLDocument & CXMLDocument::operator >> (CSerialized &inSerialized)
{
	if (m_XMLElementNode == NULL)
		m_XMLElementNode = GetRootElement ();
	CXMLElementNode *outXMLElementNode = m_XMLElementNode;
	inSerialized.Serialize (outXMLElementNode, XML_READ);
	return *this;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// xml serialization
//------------------------------------------------------------------------------------------------------------------------------------------
CXMLDocument & CXMLDocument::operator << (CSerialized *inSerialized)
{
	return operator << (*inSerialized);
}

//------------------------------------------------------------------------------------------------------------------------------------------
// xml serialization
//------------------------------------------------------------------------------------------------------------------------------------------
CXMLDocument & CXMLDocument::operator >> (CSerialized *inSerialized)
{
	return operator >> (*inSerialized);
}

//------------------------------------------------------------------------------------------------------------------------------------------
// xml serialization
//------------------------------------------------------------------------------------------------------------------------------------------
CXMLDocument & operator << (CSerialized &ioSerialized, CXMLDocument &inXMLDocument)
{
	return inXMLDocument >> ioSerialized;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// xml serialization
//------------------------------------------------------------------------------------------------------------------------------------------
CXMLDocument & operator >> (CSerialized &inSerialized, CXMLDocument &ioXMLDocument)
{
	return ioXMLDocument << inSerialized;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// xml serialization
//------------------------------------------------------------------------------------------------------------------------------------------
CXMLDocument & operator << (CSerialized *ioSerialized, CXMLDocument &inXMLDocument)
{
	return inXMLDocument >> (*ioSerialized);
}

//------------------------------------------------------------------------------------------------------------------------------------------
// xml serialization
//------------------------------------------------------------------------------------------------------------------------------------------
CXMLDocument & operator >> (CSerialized *inSerialized, CXMLDocument &ioXMLDocument)
{
	return ioXMLDocument << (*inSerialized);
}

//------------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//------------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_CAPSULE_METACLASS (CXMLElement);

//------------------------------------------------------------------------------------------------------------------------------------------
// CXMLElement constructor
//------------------------------------------------------------------------------------------------------------------------------------------
CXMLElement::CXMLElement      (xmlNodePtr inXMLNodePtr)
	    :m_XMLNodePtr     (inXMLNodePtr),
	     m_XMLElementNode (NULL)
{ } 

//------------------------------------------------------------------------------------------------------------------------------------------
// CXMLElement constructor
//------------------------------------------------------------------------------------------------------------------------------------------
CXMLElement::CXMLElement      (CXMLElementNode *inParent, const CString &inName, const CString &inValue)
	    :m_XMLNodePtr     (NULL),
	     m_XMLElementNode (new CXMLElementNode (inParent, this))
{
	m_XMLNodePtr = ::xmlNewChild (inParent->GetValue()->m_XMLNodePtr, NULL, reinterpret_cast <const xmlChar *> (inName.Get()),
				      reinterpret_cast <const xmlChar *> (inValue.Get()));
}

//------------------------------------------------------------------------------------------------------------------------------------------
// CXMLElement destructor
//------------------------------------------------------------------------------------------------------------------------------------------
CXMLElement::~CXMLElement ()
{ }

//------------------------------------------------------------------------------------------------------------------------------------------
// XMLElementNode association
//------------------------------------------------------------------------------------------------------------------------------------------
CXMLElementNode * CXMLElement::GetXMLElementNode () const
{
	return m_XMLElementNode;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// reparent the xml element node
//------------------------------------------------------------------------------------------------------------------------------------------
void CXMLElement::SetParentXMLElementNode (CXMLElementNode *inParent)
{
	if (inParent == NULL || m_XMLElementNode == NULL) return;
	m_XMLElementNode -> RemoveParent ();
	m_XMLElementNode -> SetParent (inParent);
	::xmlUnlinkNode (m_XMLNodePtr);
	::xmlAddChild (inParent->GetValue()->m_XMLNodePtr, m_XMLNodePtr);
}

//------------------------------------------------------------------------------------------------------------------------------------------
// element name access
//------------------------------------------------------------------------------------------------------------------------------------------
CString CXMLElement::GetName () const
{
	return CString (reinterpret_cast <const char *> (m_XMLNodePtr->name));
}

//------------------------------------------------------------------------------------------------------------------------------------------
// element value
//------------------------------------------------------------------------------------------------------------------------------------------
void CXMLElement::SetValue (const CString &inValue)
{
	if (m_XMLNodePtr != NULL) ::xmlNodeSetContent (m_XMLNodePtr, reinterpret_cast <const xmlChar *> (inValue.Get()));
}

//------------------------------------------------------------------------------------------------------------------------------------------
// element value
//------------------------------------------------------------------------------------------------------------------------------------------
CString CXMLElement::GetValue () const
{
	return m_XMLNodePtr->children != NULL ? CString (reinterpret_cast <const char *> (m_XMLNodePtr->children->content)) : CString();
}

//------------------------------------------------------------------------------------------------------------------------------------------
// add attribute
//------------------------------------------------------------------------------------------------------------------------------------------
CXMLAttributes CXMLElement::AddAttribute (const CString &inName, const CString &inValue)
{
	::xmlNewProp (m_XMLNodePtr, reinterpret_cast <const xmlChar *> (inName.Get()), reinterpret_cast <const xmlChar *> (inValue.Get()));
	return GetAttributes();
}

//------------------------------------------------------------------------------------------------------------------------------------------
// element attributes
//------------------------------------------------------------------------------------------------------------------------------------------
CXMLAttributes CXMLElement::GetAttributes () const
{
	CXMLAttributes outAttributes;
	if (m_XMLNodePtr->properties)
	{
		for (xmlAttrPtr curXMLAttrPtr=m_XMLNodePtr->properties; curXMLAttrPtr; curXMLAttrPtr=curXMLAttrPtr->next)
			outAttributes += CXMLAttribute (m_XMLNodePtr, curXMLAttrPtr);
	}
	return outAttributes;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// element specified attribute
//------------------------------------------------------------------------------------------------------------------------------------------
CXMLAttribute CXMLElement::GetAttribute (const CString &inName) const
{
	CXMLAttributes inAttributes (GetAttributes ());
	for (size_t i=0; i<inAttributes.GetLength(); i++)
		if (inAttributes[i]->GetName() == inName)
			return *inAttributes[i];
	return CXMLAttribute ();
}

//------------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//------------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_CAPSULE_METACLASS (CXMLAttribute);

//------------------------------------------------------------------------------------------------------------------------------------------
// CXMLAttribute constructor
//------------------------------------------------------------------------------------------------------------------------------------------
CXMLAttribute::CXMLAttribute   (xmlNodePtr inXMLNodePtr, xmlAttrPtr inXMLAttrPtr)
	      :m_XMLNodePtr    (inXMLNodePtr),
	       m_XMLAttrPtr    (inXMLAttrPtr)
{ }

//------------------------------------------------------------------------------------------------------------------------------------------
// CXMLAttribute destructor
//------------------------------------------------------------------------------------------------------------------------------------------
CXMLAttribute::~CXMLAttribute ()
{ }

//------------------------------------------------------------------------------------------------------------------------------------------
// attribute name
//------------------------------------------------------------------------------------------------------------------------------------------
CString CXMLAttribute::GetName () const
{
	return m_XMLAttrPtr != NULL ? CString (reinterpret_cast <const char *> (m_XMLAttrPtr->name)) : CString();
}

//------------------------------------------------------------------------------------------------------------------------------------------
// attribute value
//------------------------------------------------------------------------------------------------------------------------------------------
CString CXMLAttribute::GetValue () const
{
	return m_XMLAttrPtr != NULL ? CString (reinterpret_cast  <char *> (m_XMLAttrPtr->children->content)) : CString();
}

//------------------------------------------------------------------------------------------------------------------------------------------
// attribute value
//------------------------------------------------------------------------------------------------------------------------------------------
void CXMLAttribute::SetValue (const CString &inValue)
{
	if (m_XMLAttrPtr != NULL && m_XMLNodePtr != NULL) 
		::xmlSetProp (m_XMLNodePtr, m_XMLAttrPtr->name, reinterpret_cast <const xmlChar *> (inValue.Get()));
}




