=begin
  rmsgmerge.rb - Merge old .po to new .po

  Copyright (C) 2005 Masao Mutoh
  Copyright (C) 2005 speakillof

  You may redistribute it and/or modify it under the same
  license terms as Ruby.
=end

require 'optparse'
require 'gettext'
require 'gettext/poparser'

module GetText
    
  module RMsgMerge
    
    class PoData       
      
      attr_reader :msgids
      
      def initialize
        @msgid2msgstr = {}
        @msgid2comment = {}
        @msgids = []
      end
      
      def set_comment(msgid_or_sym, comment)
        @msgid2comment[msgid_or_sym] = comment      
      end
      
      def msgstr(msgid)
        @msgid2msgstr[msgid]
      end
      
      def comment(msgid)
        @msgid2comment[msgid]
      end
      
      def [](msgid)
        @msgid2msgstr[msgid]
      end
      
      def []=(msgid, msgstr)
        # Retain the order
        unless @msgid2msgstr[msgid]
          @msgids << msgid
        end
        
        @msgid2msgstr[msgid] = msgstr
      end
      
      def each_msgid
        arr = @msgids.delete_if{|i| Symbol === i or i == ''}
        arr.each do |i|
          yield i
        end
      end
      
      def msgid?(msgid)
        !(Symbol === msgid) and  @msgid2msgstr[msgid] and (msgid != '')
      end
      
      # Is it necessary to implement this method?
      def search_msgid_fuzzy(msgid, used_msgids)
        nil
      end
      
      def nplural
        unless @msgid2msgstr['']
          return 0
        else
          if /\s*nplural\s*=\s*(\d+)/ =~ @msgid2msgstr['']
            return $1.to_i
          else
            return 0
          end
          
        end
      end
      
      def generate_po
        str = ''
        str << generate_po_header
        
        self.each_msgid do |id|
          str << self.generate_po_entry(id)          
        end
        
        str << @msgid2comment[:last]        
        str        
      end
      
      def generate_po_header
        str = ""
        
        str << @msgid2comment[''].strip << "\n"
        str << 'msgid ""'  << "\n"                
        str << 'msgstr ""' << "\n"
        msgstr = @msgid2msgstr[''].gsub(/"/, '\"').gsub(/\r/, '')
        msgstr = msgstr.gsub(/^(.*)$/, '"\1\n"')
        str << msgstr
        str << "\n"
        
        str
      end
      
      def generate_po_entry(msgid)
        str = ""
        str << @msgid2comment[msgid] << "\n"
        
        id = msgid.gsub(/"/, '\"').gsub(/\r/, '')
        msgstr = @msgid2msgstr[msgid].gsub(/"/, '\"').gsub(/\r/, '')

        if id.include?("\000")
          ids = id.split(/\000/)          
          str << "msgid " << __conv(ids[0]) << "\n"
          ids[1..-1].each do |single_id|
            str << "msgid_plural " << __conv(single_id) << "\n"
          end
          
          msgstr.split("\000").each_with_index do |m, n|
            str << "msgstr[#{n}] " << __conv(m) << "\n"
          end
        else
          str << "msgid "  << __conv(id) << "\n"
          str << "msgstr " << __conv(msgstr) << "\n"
        end
        
        str << "\n"
        str
      end
      
      def __conv(str)
        s = ''
        
        if str.count("\n") > 1
          s << '""' << "\n"
          s << str.gsub(/^(.*)$/, '"\1\n"')
        else
          s << str.gsub(/^(.*)$/, '"\1\n"')
        end
        
        s.rstrip
      end
      
    end
    
    class Merger__
      
      # From gettext source.
      # 
      # Merge the reference with the definition: take the #. and
      #	#: comments from the reference, take the # comments from
	  # the definition, take the msgstr from the definition.  Add
	  # this merged entry to the output message list.      
      COMMENT_RE = /\A#\.|\A#\:/
      
      CRLF_RE = /\r?\n/
      
      def initialize
      end
      
      def merge(definition, reference)
        # deep copy
        result = Marshal.load( Marshal.dump(reference) )        
        
        used = []        
        merge_header(result, definition)
        
        result.each_msgid do |msgid|
          if definition.msgid?(msgid)
            used << msgid
            merge_message(msgid, result, msgid, definition)
          elsif other_msgid = definition.search_msgid_fuzzy(msgid, used)
            used << other_msgid
            merge_fuzzy_message(msgid, result, other_msgid, definition) 
          else
            if msgid.index("\000") and (reference.msgstr(msgid).gsub("\000", '') == '')
              #plural              
              result[msgid] = "\000" * definition.nplural
            end
          end          
        end
        
        ###################################################################
        # msgids which are not used in reference are handled as obsolete. #
        ################################################################### 
        last_comment = result.comment(:last) || ''
        definition.each_msgid do |msgid|
          unless used.include?(msgid)
            last_comment << "\n"
            last_comment << definition.generate_po_entry(msgid).strip.gsub(/^/, '#~ ')
            last_comment << "\n"
          end
        end
        result.set_comment(:last, last_comment)
        
        result
      end
      
      def merge_message(msgid, target, def_msgid, definition)
        merge_comment(msgid, target, def_msgid, definition)
        
        ############################################
        # check mismatch of msgid and msgid_plural #
        ############################################
        def_msgstr = definition[def_msgid]
        if msgid.index("\000")
          if def_msgstr.index("\000")
            # OK
            target[msgid] = def_msgstr			
          else
            # NG
            s = ''            
            definition.nplural.times {
              s << def_msgstr
              s << "\000"
            }
            target[msgid] = s
          end
        else
          if def_msgstr.index("\000")
            # NG
            target[msgid] = def_msgstr.split("\000")[0]
          else
            # OK
            target[msgid] = def_msgstr
          end
        end
      end
      
      # for the future
      def merge_fuzzy_message(msgid, target, def_msgid, definition)
        merge_message(msgid, target, def_msgid, definition)
      end
      
      def merge_comment(msgid, target, def_msgid, definition)
        comment = target.comment(msgid)
        def_comment = definition.comment(def_msgid)
        new_comment = []
        
        def_comment.split(CRLF_RE).each do |l| 
          unless COMMENT_RE =~ l
            new_comment << l
          end           
        end
        
        comment.split(CRLF_RE).each do |l|
          if COMMENT_RE =~ l
            new_comment << l
          end
        end
        
        target.set_comment(msgid, new_comment.join("\n"))        
      end
      
      def merge_header(target, definition)
        merge_comment('', target, '', definition)
        
        msg = target.msgstr('')
        def_msg = definition.msgstr('')
        if /POT-Creation-Date:\s*(.*)?\s*\\n/ =~ msg
          time = $1
          def_msg = def_msg.sub(/POT-Creation-Date:.*\\n/, "POT-Creation-Date: #{time}" + '\n')
        end
        
        target[''] = def_msg
      end
      
    end
    
  end  
  
end



#
# commands 
# 
module GetText
  
  module RMsgMerge
    extend GetText

    bindtextdomain("rgettext")
    
    # constant values
    VERSION = GetText::VERSION
    DATE = %w($Date: 2005/11/27 15:35:55 $)[1]
    
    module_function

    def check_options
      output = STDOUT
      
      opts = OptionParser.new
      opts.banner = _("Usage: %s def.po ref.pot [-o output.pot]") % $0
      opts.separator("")
      opts.separator(_("Merges two Uniforum style .po files together. The def.po file is an existing PO file with translations. The ref.pot file is the last created PO file with up-to-date source references. ref.pot is generally created by rgettext."))
      opts.separator("")
      opts.separator(_("Specific options:"))
      
      opts.on("-o", "--output=FILE", _("write output to specified file")) do |out|
        unless FileTest.exist? out
          output = File.new(File.expand_path(out), "w+")
        else
          #$stderr.puts(_("File '%s' has already existed.") % out)
          #exit 1
        end
      end
      
      opts.on_tail("--version", _("display version information and exit")) do
        puts "#{$0} #{VERSION} (#{DATE})"
        exit
      end
      
      opts.parse!(ARGV)
      
      if ARGV.size != 2
        puts opts.help
        exit 1
      end
      
      [ARGV[0], ARGV[1], output]
    end
    
    def run(reference = nil, definition = nil, out = STDOUT)
      if reference.nil? or definition.nil?
        definition, reference, out = check_options()
      end
      
      if definition.nil?
        raise ArgumentError, _("definition po is not given.")
      elsif reference.nil? 
        raise ArgumentError, _("reference pot is not given.")
      end
      
      parser = PoParser.new
      defstr = nil; refstr = nil
      File.open(definition){|f| defstr = f.read}; File.open(reference){|f| refstr = f.read}
      defpo = parser.parse(defstr, PoData.new)
      refpot = parser.parse(refstr, PoData.new)
      
      m = Merger__.new
      result = m.merge(defpo, refpot)      
      pp result if $DEBUG
      print result.generate_po if $DEBUG
      
      begin
        if out.is_a? String
          File.open(File.expand_path(out), "w+") do |file|
            file.write(result.generate_po)
          end
        else
          out.puts(result.generate_po)
        end
      ensure
        out.close
      end
    end
  end

  module_function
  def rmsgmerge(reference = nil, definition = nil, out = STDOUT)
    RMsgMerge.run(reference, definition, out)
  end
end


if $0 == __FILE__ then
  require 'pp'
  
  #parser = GetText::RMsgMerge::PoParser.new;
  #parser = GetText::PoParser.new;
  #pp parser.parse(ARGF.read)
  
  GetText.rmsgmerge
end
