# Check for the presence of C99 features.  Generally the check will fail
# if the feature isn't present (a C99 compiler isn't that much to ask,
# right?).

# Copyright (C) 2004-2005 Jason Short.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.

# Originally written by Jason Short for Freeciv.  Imported and updated for
# GGZ.  The canonical copy of this resides in the libggz module.
# $Id: c99.m4 7750 2006-01-07 22:20:47Z jdorje $

# Check C99-style variadic macros (required):
#
#  #define PRINTF(msg, ...) (printf(msg, __VA_ARGS__)
#
AC_DEFUN([AC_C99_VARIADIC_MACROS],
[
  dnl Check for variadic macros
  AC_CACHE_CHECK([for C99 variadic macros],
    [ac_cv_c99_variadic_macros],
     [AC_TRY_COMPILE(
          [#include <stdio.h>
           #define MSG(...) fprintf(stderr, __VA_ARGS__)
          ],
          [MSG("foo");
           MSG("%s", "foo");
           MSG("%s%d", "foo", 1);],
          ac_cv_c99_variadic_macros=yes,
          ac_cv_c99_variadic_macros=no)])
  if test "x${ac_cv_c99_variadic_macros}" != "xyes"; then
    AC_MSG_ERROR([A compiler supporting C99 variadic macros is required])
  fi
])

# Check C99-style variable-sized arrays (required):
#
#   char concat_str[strlen(s1) + strlen(s2) + 1];
#
AC_DEFUN([AC_C99_VARIABLE_ARRAYS],
[
  dnl Check for variable arrays
  AC_CACHE_CHECK([for C99 variable arrays],
    [ac_cv_c99_variable_arrays],
    [AC_TRY_COMPILE(
        [],
        [char *s1 = "foo", *s2 = "bar";
         char s3[strlen(s1) + strlen(s2) + 1];
         sprintf(s3, "%s%s", s1, s2);],
        ac_cv_c99_variable_arrays=yes,
        ac_cv_c99_variable_arrays=no)])
  if test "x${ac_cv_c99_variable_arrays}" != "xyes"; then
    AC_MSG_ERROR([A compiler supporting C99 variable arrays is required])
  fi
])

# Check C99-style initializers (required):
#
# Examples:
#   struct timeval tv = {.tv_sec = 0, .tv_usec = 500000};
#   int fibonacci[6] = {[0] = 0, [1] = 1, [2] = 1, [3] = 2, [4] = 3, [5] = 5};
# Note we do not check for multi-field initializers like
#   struct { struct { int b; } a; } = {.a.b = 5}
# which are not supported by many compilers.  It is best to avoid this
# problem by writing these using nesting.  The above case becomes
#   struct { struct { int b; } a; } = {.a = {.b = 5}}
AC_DEFUN([AC_C99_INITIALIZERS],
[
  dnl Check for C99 initializers
  AC_CACHE_CHECK([for C99 initializers],
    [ac_cv_c99_initializers],
    [AC_TRY_COMPILE(
        [struct foo {
           int an_integer;
           char *a_string;
           int an_array[5];
           union {int x, y;} a_union;
         };
        ],
        [struct foo bar = {.an_array = {0, [3] = 2, [2] = 1, [4] = 3},
                           .an_integer = 999,
                           .a_string = "does it work?",
                           .a_union = {.y = 243}};],
        [ac_cv_c99_initializers=yes],
        [ac_cv_c99_initializers=no])])
  if test "${ac_cv_c99_initializers}" != "yes"; then
    AC_MSG_ERROR([A compiler supporting C99 initializers is required])
  fi
])

# Check C99-style stdint.h (required)
AC_DEFUN([AC_C99_STDINT_H],
[
  AC_CHECK_HEADERS([stdint.h])
  dnl Check for C99 stdint.h
  AC_CACHE_CHECK([for C99 stdint.h],
    [ac_cv_c99_stdint_h],
    [ac_cv_c99_stdint_h=$ac_cv_header_stdint_h])
  if test "${ac_cv_c99_stdint_h}" != "yes"; then
    AC_MSG_ERROR([A compiler supporting C99's stdint.h is required])
  fi
])
# AC_GGZ_DEBUG
# $Id: ggz-debug.m4 5748 2004-01-27 08:44:46Z josef $
#
# Check for debugging choices.  Sets $enable_debug, $enable_debug_gdb,
# and $enable_debug_mem.
#
# This file may be used in many GGZ modules; the master copy should reside
# in libggz.

AC_DEFUN([AC_GGZ_DEBUG],
[
  # This defines either DEBUG or NDEBUG.  We could provide a third option
  # where neither are defined.
  AC_ARG_ENABLE([debug],
		AC_HELP_STRING([--disable-debug], [disable basic debugging]),
		[enable_debug=no],
		[enable_debug=yes])

  if test "$enable_debug" = yes; then
    AC_DEFINE([DEBUG], 1, [Define to enable basic debugging])
    DFLAGS="DEBUG"
  else
    AC_DEFINE([NDEBUG], 1, [Define to disable basic debugging])
    DFLAGS="NDEBUG"
  fi

  # Turn on GDB debugging, via a command-line argument for gcc
  AC_ARG_ENABLE([debug-gdb],
		AC_HELP_STRING([--enable-debug-gdb], [enable GDB debugging]),
		[enable_debug_gdb=yes],
		[enable_debug_gdb=no])
  if test "$enable_debug_gdb" = yes; then
    CFLAGS="$CFLAGS -ggdb"
  fi

  # Turn on malloc debugging via dmalloc (if specified)
  AC_ARG_ENABLE([debug-mem],
		AC_HELP_STRING([--enable-debug-mem],[enable mem debugging]),
		[enable_debug_mem=yes],
		[enable_debug_mem=no])
  if test "$enable_debug_mem" = yes; then
    AC_DEFINE([DEBUG_MEM], 1, [Define to use dmalloc memory debugging])

    # We count on this code coming after DEBUG_MEM is defined...
    AH_VERBATIM([_DEBUG_MEM],
[/* Use dmalloc for memory debugging */
#ifdef DEBUG_MEM
# include <dmalloc.h>
#endif])
    DFLAGS="$DFLAGS DEBUG_MEM"
    LDADD="$LDADD -ldmalloc"
  fi

  # Use -Wall if we have gcc.
  # FIXME: Rewrite this to be comprehensible.
  changequote(,)dnl
  if test "x$GCC" = "xyes"; then
    case " $CFLAGS " in
    *[\ \	]-Wall[\ \	]*) ;;
    *) CFLAGS="$CFLAGS -Wall" ;;
    esac
  fi
  changequote([,])dnl
])
dnl ======================================
dnl GGZ Gaming Zone - Configuration Macros
dnl ======================================
dnl
dnl Copyright (C) 2001 - 2004 Josef Spillner, josef@ggzgamingzone.org
dnl This file has heavily been inspired by KDE's acinclude :)
dnl It is published under the conditions of the GNU General Public License.
dnl
dnl ======================================
dnl
dnl This file is common to most GGZ modules, and should be kept in sync
dnl between them all.  The master copy resides with libggz.
dnl Currently the following modules use it:
dnl   kde-games, kde-client, gtk-games, gtk-client, utils, grubby,
dnl   ggz-client-libs, ggzd, gnome-client, txt-client, sdl-games, libggz
dnl See /docs/ggz-project/buildsystem for documentation.
dnl
dnl ======================================
dnl
dnl History:
dnl   See the CVS log for a full history.
dnl
dnl ------------------------------------------------------------------------
dnl Content of this file:
dnl ------------------------------------------------------------------------
dnl High-level macros:
dnl   AC_GGZ_CHECK - Checks for presence of GGZ client and server libraries.
dnl                  GGZ users can call this macro to determine at compile
dnl                  time whether to include GGZ support.  Server and client
dnl                  are checked separately.  GGZ_SERVER and GGZ_CLIENT are
dnl                  defined in config.h, and created as conditionals in
dnl                  the Makefiles.
dnl
dnl Low-level macros:
dnl   AC_GGZ_INIT - initialization and paths/options setup
dnl   AC_GGZ_VERSION - ensure a minimum version of GGZ
dnl   AC_GGZ_LIBGGZ - find the libggz headers and libraries
dnl   AC_GGZ_GGZCORE - find the ggzcore headers and libraries
dnl   AC_GGZ_CONFIG - find the ggz-config tool and set up configuration
dnl   AC_GGZ_GGZMOD - find the ggzmod library
dnl   AC_GGZ_GGZDMOD - find the ggzdmod library
dnl   AC_GGZ_SERVER - set up game and room path for ggzd game servers
dnl   AC_GGZ_INTL - ensure proper i18n tools installation
dnl
dnl   Each macro takes two arguments:
dnl     1.  Action-if-found (or empty for no action).
dnl     2.  Action-if-not-found (or empty for error, or "ignore" to ignore).
dnl
dnl Internal functions:
dnl   AC_GGZ_ERROR - user-friendly error messages
dnl   AC_GGZ_FIND_FILE - macro for convenience (thanks kde)
dnl   AC_GGZ_REMOVEDUPS - eliminate duplicate list elements
dnl

dnl ------------------------------------------------------------------------
dnl Find a directory containing a single file
dnl Synopsis: AC_GGZ_FIND_FILE(file, directorylist, <returnvar>)
dnl ------------------------------------------------------------------------
dnl
AC_DEFUN([AC_GGZ_FIND_FILE],
[
$3=NO
for i in $2;
do
  for j in $1;
  do
    echo "configure: __oline__: $i/$j" >&AC_FD_CC
    if test -r "$i/$j"; then
      echo "taking that" >&AC_FD_CC
      $3=$i
      break 2
    fi
  done
done
])

dnl ------------------------------------------------------------------------
dnl Remove duplicate entries in a list, and remove all NO's
dnl Synopsis: AC_GGZ_REMOVEDUPS(list, <returnlist>)
dnl ------------------------------------------------------------------------
dnl
AC_DEFUN([AC_GGZ_REMOVEDUPS],
[
ret=""
for i in $1; do
  add=yes
  for j in $ret; do
    if test "x$i" = "x$j"; then
      add=no
    fi
  done
  if test "x$i" = "xNO"; then
    add=no
  fi
  if test "x$add" = "xyes"; then
  ret="$ret $i"
  fi
done
$2=$ret
])

dnl ------------------------------------------------------------------------
dnl User-friendly error messages
dnl Synopsis: AC_GGZ_ERROR(libraryname, headerdirlist, libdirlist)
dnl ------------------------------------------------------------------------
dnl
AC_DEFUN([AC_GGZ_ERROR],
[
  AC_MSG_WARN([no
  The library '$1' does not seem to be installed correctly.
  Headers searched in: $2
  Libraries searched in: $3
  Please read QuickStart.GGZ in order to fix this.
  ])
  exit 1
])

dnl ------------------------------------------------------------------------
dnl Initialization, common values and such
dnl Synopsis: AC_GGZ_INIT([export], [defaults])
dnl ------------------------------------------------------------------------
dnl
AC_DEFUN([AC_GGZ_INIT],
[
if test "x$prefix" = "xNONE"; then
  prefix="${ac_default_prefix}"
fi
AC_DEFINE_UNQUOTED([PREFIX], "${prefix}", [The installation prefix])

if test "x${prefix}" = "xNONE"; then
   ac_ggz_prefix_incdir="${ac_default_prefix}/include"
   ac_ggz_prefix_libdir="${ac_default_prefix}/lib"
   ac_ggz_prefix_bindir="${ac_default_prefix}/bin"
   ac_ggz_prefix_etcdir="${ac_default_prefix}/etc"
else
   ac_ggz_prefix_incdir="${prefix}/include"
   ac_ggz_prefix_libdir="${prefix}/lib"
   ac_ggz_prefix_bindir="${prefix}/bin"
   ac_ggz_prefix_etcdir="${prefix}/etc"
fi
ac_ggz_stdinc="$ac_ggz_prefix_incdir"
ac_ggz_stdlib="$ac_ggz_prefix_libdir"
ac_ggz_stdbin="$ac_ggz_prefix_bindir"
ac_ggz_stdetc="$ac_ggz_prefix_etcdir"
if test "x$1" = "xdefaults" || test "x$2" = "xdefaults"; then
  ac_ggz_stdinc="$ac_ggz_stdinc /usr/local/include /usr/include"
  ac_ggz_stdlib="$ac_ggz_stdlib /usr/local/lib /usr/lib"
  ac_ggz_stdbin="$ac_ggz_stdbin /usr/local/bin /usr/bin"
  ac_ggz_stdetc="$ac_ggz_stdetc/ggzd /usr/local/etc/ggzd /etc/ggzd"
fi
if test "x$1" = "xexport" || test "x$2" = "xexport"; then
  CPPFLAGS="$CPPFLAGS -isystem ${ac_ggz_prefix_incdir} -isystem /usr/local/include"
  LDFLAGS="$LDFLAGS -L${ac_ggz_prefix_libdir} -L/usr/local/lib"
fi

save_cflags=$CFLAGS
save_cxxflags=$CXXFLAGS
CFLAGS="-Wall -Werror"
AC_COMPILE_IFELSE([AC_LANG_PROGRAM(
	[[void signedness(void){char c;if(c==-1)c=0;}]])],
	[],
	[save_cflags="$save_cflags -fsigned-char"
	 save_cxxflags="$save_cxxflags -fsigned-char"])
CFLAGS=$save_cflags
CXXFLAGS=$save_cxxflags
])

dnl ------------------------------------------------------------------------
dnl Ensure that a minimum version of GGZ is present
dnl Synopsis: AC_GGZ_VERSION(major, minor, micro)
dnl ------------------------------------------------------------------------
dnl
AC_DEFUN([AC_GGZ_VERSION],
[
	major=$1
	minor=$2
	micro=$3

	testprologue="#include <ggz.h>"
	testbody=""
	testbody="$testbody if(LIBGGZ_VERSION_MAJOR > $major) return 0;"
	testbody="$testbody if(LIBGGZ_VERSION_MAJOR < $major) return -1;"
	testbody="$testbody if(LIBGGZ_VERSION_MINOR > $minor) return 0;"
	testbody="$testbody if(LIBGGZ_VERSION_MINOR < $minor) return -1;"
	testbody="$testbody if(LIBGGZ_VERSION_MICRO > $micro) return 0;"
	testbody="$testbody if(LIBGGZ_VERSION_MICRO < $micro) return -1;"
	testbody="$testbody return 0;"

	AC_RUN_IFELSE(
		[AC_LANG_PROGRAM([[$testprologue]], [[$testbody]])],
		[],
		[AC_MSG_ERROR([The GGZ version is too old. Version $major.$minor.$micro is required.])]
	)
])

dnl ------------------------------------------------------------------------
dnl Try to find the libggz headers and libraries.
dnl $(LIBGGZ_LDFLAGS) will be -L ... (if needed)
dnl and $(LIBGGZ_INCLUDES) will be -I ... (if needed)
dnl ------------------------------------------------------------------------
dnl
AC_DEFUN([AC_GGZ_LIBGGZ],
[
AC_MSG_CHECKING([for GGZ library: libggz])

ac_libggz_includes=NO ac_libggz_libraries=NO
libggz_libraries=""
libggz_includes=""

AC_ARG_WITH(libggz-dir,
    AC_HELP_STRING([--with-libggz-dir=DIR],[libggz installation prefix]),
    [  ac_libggz_includes="$withval"/include
       ac_libggz_libraries="$withval"/lib
    ])
AC_ARG_WITH(libggz-includes,
    AC_HELP_STRING([--with-libggz-includes=DIR],
                   [where the libggz includes are]),
    [  ac_libggz_includes="$withval"
    ])
AC_ARG_WITH(libggz-libraries,
    AC_HELP_STRING([--with-libggz-libraries=DIR],[where the libggz libs are]),
    [  ac_libggz_libraries="$withval"
    ])

AC_CACHE_VAL(ac_cv_have_libggz,
[
libggz_incdirs="$ac_libggz_includes $ac_ggz_stdinc"
AC_GGZ_REMOVEDUPS($libggz_incdirs, libggz_incdirs)
libggz_header=ggz.h

AC_GGZ_FIND_FILE($libggz_header, $libggz_incdirs, libggz_incdir)
ac_libggz_includes="$libggz_incdir"

libggz_libdirs="$ac_libggz_libraries $ac_ggz_stdlib"
AC_GGZ_REMOVEDUPS($libggz_libdirs, libggz_libdirs)

libggz_libdir=NO
for dir in $libggz_libdirs; do
  try="ls -1 $dir/libggz.la $dir/libggz.so"
  if test -n "`$try 2> /dev/null`"; then libggz_libdir=$dir; break; else echo "tried $dir" >&AC_FD_CC ; fi
done

ac_libggz_libraries="$libggz_libdir"

if test "$ac_libggz_includes" = NO || test "$ac_libggz_libraries" = NO; then
  ac_cv_have_libggz="have_libggz=no"
  ac_libggz_notfound=""
else
  have_libggz="yes"
fi
])

eval "$ac_cv_have_libggz"

if test "$have_libggz" != yes; then
  if test "x$2" = "xignore"; then
    AC_MSG_RESULT([$have_libggz (ignored)])
  else
    AC_MSG_RESULT([$have_libggz])
    if test "x$2" = "x"; then
      AC_GGZ_ERROR(libggz, $libggz_incdirs, $libggz_libdirs)
    fi

    # perform actions given by argument 2.
    $2
  fi
else
  ac_cv_have_libggz="have_libggz=yes \
    ac_libggz_includes=$ac_libggz_includes ac_libggz_libraries=$ac_libggz_libraries"
  AC_MSG_RESULT([$have_libggz (libraries $ac_libggz_libraries, headers $ac_libggz_includes)])

  libggz_libraries="$ac_libggz_libraries"
  libggz_includes="$ac_libggz_includes"

  AC_SUBST(libggz_libraries)
  AC_SUBST(libggz_includes)

  LIBGGZ_INCLUDES="-isystem $libggz_includes"
  LIBGGZ_LDFLAGS="-L$libggz_libraries"

  AC_SUBST(LIBGGZ_INCLUDES)
  AC_SUBST(LIBGGZ_LDFLAGS)

  LIB_GGZ='-lggz'
  AC_SUBST(LIB_GGZ)

  # perform actions given by argument 1.
  $1
fi

])


dnl ------------------------------------------------------------------------
dnl Try to find the ggzcore headers and libraries.
dnl $(GGZCORE_LDFLAGS) will be -L ... (if needed)
dnl and $(GGZCORE_INCLUDES) will be -I ... (if needed)
dnl ------------------------------------------------------------------------
dnl
AC_DEFUN([AC_GGZ_GGZCORE],
[
AC_MSG_CHECKING([for GGZ library: ggzcore])

ac_ggzcore_includes=NO ac_ggzcore_libraries=NO
ggzcore_libraries=""
ggzcore_includes=""

AC_ARG_WITH(ggzcore-dir,
    AC_HELP_STRING([--with-ggzcore-dir=DIR],[ggzcore installation prefix]),
    [  ac_ggzcore_includes="$withval"/include
       ac_ggzcore_libraries="$withval"/lib
    ])
AC_ARG_WITH(ggzcore-includes,
    AC_HELP_STRING([--with-ggzcore-includes=DIR],
                   [where the ggzcore includes are]),
    [  ac_ggzcore_includes="$withval"
    ])
AC_ARG_WITH(ggzcore-libraries,
    AC_HELP_STRING([--with-ggzcore-libraries=DIR],
                   [where the ggzcore libs are]),
    [  ac_ggzcore_libraries="$withval"
    ])

AC_CACHE_VAL(ac_cv_have_ggzcore,
[
ggzcore_incdirs="$ac_ggzcore_includes $ac_ggz_stdinc"
AC_GGZ_REMOVEDUPS($ggzcore_incdirs, ggzcore_incdirs)
ggzcore_header=ggzcore.h

AC_GGZ_FIND_FILE($ggzcore_header, $ggzcore_incdirs, ggzcore_incdir)
ac_ggzcore_includes="$ggzcore_incdir"

ggzcore_libdirs="$ac_ggzcore_libraries $ac_ggz_stdlib"
AC_GGZ_REMOVEDUPS($ggzcore_libdirs, ggzcore_libdirs)

ggzcore_libdir=NO
for dir in $ggzcore_libdirs; do
  try="ls -1 $dir/libggzcore.la $dir/libggzcore.so"
  if test -n "`$try 2> /dev/null`"; then ggzcore_libdir=$dir; break; else echo "tried $dir" >&AC_FD_CC ; fi
done

ac_ggzcore_libraries="$ggzcore_libdir"

if test "$ac_ggzcore_includes" = NO || test "$ac_ggzcore_libraries" = NO; then
  ac_cv_have_ggzcore="have_ggzcore=no"
  ac_ggzcore_notfound=""
else
  have_ggzcore="yes"
fi
])

eval "$ac_cv_have_ggzcore"

if test "$have_ggzcore" != yes; then
  if test "x$2" = "xignore"; then
    AC_MSG_RESULT([$have_ggzcore (intentionally ignored)])
  else
    AC_MSG_RESULT([$have_ggzcore])
    if test "x$2" = "x"; then
      AC_GGZ_ERROR(ggzcore, $ggzcore_incdirs, $ggzcore_libdirs)
    fi

    # Perform actions given by argument 2.
    $2
  fi
else
  ac_cv_have_ggzcore="have_ggzcore=yes \
    ac_ggzcore_includes=$ac_ggzcore_includes ac_ggzcore_libraries=$ac_ggzcore_libraries"
  AC_MSG_RESULT([$have_ggzcore (libraries $ac_ggzcore_libraries, headers $ac_ggzcore_includes)])

  ggzcore_libraries="$ac_ggzcore_libraries"
  ggzcore_includes="$ac_ggzcore_includes"

  AC_SUBST(ggzcore_libraries)
  AC_SUBST(ggzcore_includes)

  GGZCORE_INCLUDES="-isystem $ggzcore_includes"
  GGZCORE_LDFLAGS="-L$ggzcore_libraries"

  AC_SUBST(GGZCORE_INCLUDES)
  AC_SUBST(GGZCORE_LDFLAGS)

  LIB_GGZCORE='-lggzcore'
  AC_SUBST(LIB_GGZCORE)

  # Perform actions given by argument 1.
  $1
fi

])

dnl ------------------------------------------------------------------------
dnl Try to find the ggz-config binary.
dnl Sets GGZ_CONFIG to the path/name of the program.
dnl Sets also: ggz_gamedir, ggz_datadir etc.
dnl ------------------------------------------------------------------------
dnl
AC_DEFUN([AC_GGZ_CONFIG],
[
AC_MSG_CHECKING([for GGZ configuration tool: ggz-config])

ac_ggz_config=NO
ggz_config=""

AC_ARG_WITH(ggzconfig,
    AC_HELP_STRING([--with-ggzconfig=DIR],[path to ggz-config]),
    [  ac_ggz_config="$withval"
    ])

AC_CACHE_VAL(ac_cv_have_ggzconfig,
[
ggz_config_dirs="$ac_ggz_config $ac_ggz_stdbin"

AC_GGZ_FIND_FILE(ggz-config, $ggz_config_dirs, ggz_config_dir)
ac_ggz_config="$ggz_config_dir"

if test "$ac_ggz_config" = NO; then
  ac_cv_have_ggzcore="have_ggz_config=no"
  ac_ggz_config_notfound=""
  have_ggz_config="no"
else
  have_ggz_config="yes"
fi
])

eval "$ac_cv_have_ggz_config"

if test "$have_ggz_config" != yes; then
  if test "x$2" = "xignore"; then
    AC_MSG_RESULT([$have_ggz_config (intentionally ignored)])
    GGZ_CONFIG="true"
    ggzexecmoddir="\${prefix}/lib/ggz"
    ggzdatadir="\${prefix}/share/ggz"
    AC_SUBST(GGZ_CONFIG)
    AC_SUBST(ggzexecmoddir)
    AC_SUBST(ggzdatadir)
    AC_DEFINE_UNQUOTED(GAMEDIR, "${prefix}/lib/ggz", [Path where to install the games])
    AC_DEFINE_UNQUOTED(GGZDATADIR, "${prefix}/share/ggz", [Path where the games should look for their data files])
  else
    AC_MSG_RESULT([$have_ggz_config])
    if test "x$2" = "x"; then
      AC_MSG_ERROR([ggz-config not found. Please check your installation! ])
    fi

    # Perform actions given by argument 2.
    $2
  fi
else
  pathto_app=`echo $prefix/bin/ | tr -s "/"`
  pathto_ggz=`echo $ac_ggz_config/ | tr -s "/"`

  if test "x$pathto_app" != "x$pathto_ggz"; then
    AC_MSG_RESULT([$have_ggz_config (dismissed due to different prefix)])
    GGZ_CONFIG="true"
    ggzexecmoddir="\${prefix}/lib/ggz"
    ggzdatadir="\${prefix}/share/ggz"
    AC_SUBST(GGZ_CONFIG)
    AC_SUBST(ggzexecmoddir)
    AC_SUBST(ggzdatadir)
    AC_DEFINE_UNQUOTED(GAMEDIR, "${prefix}/lib/ggz", [Path where to install the games])
    AC_DEFINE_UNQUOTED(GGZDATADIR, "${prefix}/share/ggz", [Path where the games should look for their data files])
  else
    ac_cv_have_ggz_config="have_ggz_config=yes \
      ac_ggz_config=$ac_ggz_config"
    AC_MSG_RESULT([$ac_ggz_config/ggz-config])

    ggz_config="$ac_ggz_config"
    AC_SUBST(ggz_config)

    AC_ARG_ENABLE([noregistry],
      AC_HELP_STRING([--enable-noregistry], [Do not register game modules.]),
      [enable_noregistry=yes], [enable_noregistry=no])

    GGZ_CONFIG="${ggz_config}/ggz-config"
    if test "$enable_noregistry" = yes; then
      GGZ_CONFIG="$GGZ_CONFIG --noregistry=$enableval"
    fi
    AC_SUBST(GGZ_CONFIG)

    ggzmoduleconfdir=`$GGZ_CONFIG --configdir`
    AC_DEFINE_UNQUOTED(GGZMODULECONFDIR, "${ggzmoduleconfdir}", [Path where the game registry is located])
    ggzexecmoddir=`$GGZ_CONFIG --gamedir`
    AC_DEFINE_UNQUOTED(GAMEDIR, "${ggzexecmoddir}", [Path where to install the games])
    ggzdatadir=`$GGZ_CONFIG --datadir`
    AC_DEFINE_UNQUOTED(GGZDATADIR, "${ggzdatadir}", [Path where the games should look for their data files])
    packagesrcdir=`cd $srcdir && pwd`
    AC_DEFINE_UNQUOTED(PACKAGE_SOURCE_DIR, "${packagesrcdir}", [Path where the source is located])

    AC_SUBST(ggzmoduleconfdir)
    AC_SUBST(ggzexecmoddir)
    AC_SUBST(ggzdatadir)
    AC_SUBST(packagesrcdir)

    # Perform actions given by argument 1.
    $1
  fi
fi

])

dnl ------------------------------------------------------------------------
dnl Try to find the ggzmod headers and libraries.
dnl $(GGZMOD_LDFLAGS) will be -L ... (if needed)
dnl and $(GGZMOD_INCLUDES) will be -I ... (if needed)
dnl ------------------------------------------------------------------------
dnl
AC_DEFUN([AC_GGZ_GGZMOD],
[
AC_MSG_CHECKING([for GGZ library: ggzmod])

ac_ggzmod_includes=NO ac_ggzmod_libraries=NO
ggzmod_libraries=""
ggzmod_includes=""

AC_ARG_WITH(ggzmod-dir,
    AC_HELP_STRING([--with-ggzmod-dir=DIR],[ggzmod installation prefix]),
    [  ac_ggzmod_includes="$withval"/include
       ac_ggzmod_libraries="$withval"/lib
    ])
AC_ARG_WITH(ggzmod-includes,
    AC_HELP_STRING([--with-ggzmod-includes=DIR],
                   [where the ggzmod includes are]),
    [  ac_ggzmod_includes="$withval"
    ])
AC_ARG_WITH(ggzmod-libraries,
    AC_HELP_STRING([--with-ggzmod-libraries=DIR],
                   [where the ggzmod libs are]),
    [  ac_ggzmod_libraries="$withval"
    ])

AC_CACHE_VAL(ac_cv_have_ggzmod,
[
ggzmod_incdirs="$ac_ggzmod_includes $ac_ggz_stdinc"
AC_GGZ_REMOVEDUPS($ggzmod_incdirs, ggzmod_incdirs)
ggzmod_header=ggzmod.h

AC_GGZ_FIND_FILE($ggzmod_header, $ggzmod_incdirs, ggzmod_incdir)
ac_ggzmod_includes="$ggzmod_incdir"

ggzmod_libdirs="$ac_ggzmod_libraries $ac_ggz_stdlib"
AC_GGZ_REMOVEDUPS($ggzmod_libdirs, ggzmod_libdirs)

ggzmod_libdir=NO
for dir in $ggzmod_libdirs; do
  try="ls -1 $dir/libggzmod.la $dir/libggzmod.so"
  if test -n "`$try 2> /dev/null`"; then ggzmod_libdir=$dir; break; else echo "tried $dir" >&AC_FD_CC ; fi
done

ac_ggzmod_libraries="$ggzmod_libdir"

if test "$ac_ggzmod_includes" = NO || test "$ac_ggzmod_libraries" = NO; then
  ac_cv_have_ggzmod="have_ggzmod=no"
  ac_ggzmod_notfound=""
else
  have_ggzmod="yes"
fi
])

eval "$ac_cv_have_ggzmod"

if test "$have_ggzmod" != yes; then
  if test "x$2" = "xignore"; then
    AC_MSG_RESULT([$have_ggzmod (intentionally ignored)])
  else
    AC_MSG_RESULT([$have_ggzmod])
    if test "x$2" = "x"; then
      AC_GGZ_ERROR(ggzmod, $ggzmod_incdirs, $ggzmod_libdirs)
    fi

    # Perform actions given by argument 2.
    $2
  fi
else
  ac_cv_have_ggzmod="have_ggzmod=yes \
    ac_ggzmod_includes=$ac_ggzmod_includes ac_ggzmod_libraries=$ac_ggzmod_libraries"
  AC_MSG_RESULT([$have_ggzmod (libraries $ac_ggzmod_libraries, headers $ac_ggzmod_includes)])

  ggzmod_libraries="$ac_ggzmod_libraries"
  ggzmod_includes="$ac_ggzmod_includes"

  AC_SUBST(ggzmod_libraries)
  AC_SUBST(ggzmod_includes)

  GGZMOD_INCLUDES="-isystem $ggzmod_includes"
  GGZMOD_LDFLAGS="-L$ggzmod_libraries"

  AC_SUBST(GGZMOD_INCLUDES)
  AC_SUBST(GGZMOD_LDFLAGS)

  LIB_GGZMOD='-lggzmod'
  AC_SUBST(LIB_GGZMOD)

  # Perform actions given by argument 1.
  $1
fi

])

dnl ------------------------------------------------------------------------
dnl Try to find the ggzdmod headers and libraries.
dnl $(GGZDMOD_LDFLAGS) will be -L ... (if needed)
dnl and $(GGZDMOD_INCLUDES) will be -I ... (if needed)
dnl ------------------------------------------------------------------------
dnl
AC_DEFUN([AC_GGZ_GGZDMOD],
[
AC_MSG_CHECKING([for GGZ library: ggzdmod])

ac_ggzdmod_includes=NO ac_ggzdmod_libraries=NO
ggzdmod_libraries=""
ggzdmod_includes=""

AC_ARG_WITH(ggzdmod-dir,
    AC_HELP_STRING([--with-ggzdmod-dir=DIR], [ggzdmod installation prefix]),
    [  ac_ggzdmod_includes="$withval"/include
       ac_ggzdmod_libraries="$withval"/lib
    ])
AC_ARG_WITH(ggzdmod-includes,
    AC_HELP_STRING([--with-ggzdmod-includes=DIR], 
                   [where the ggzdmod includes are]),
    [  ac_ggzdmod_includes="$withval"
    ])
AC_ARG_WITH(ggzdmod-libraries,
    AC_HELP_STRING([--with-ggzdmod-libraries=DIR],
                   [where the ggzdmod libs are]),
    [  ac_ggzdmod_libraries="$withval"
    ])

AC_CACHE_VAL(ac_cv_have_ggzdmod,
[
ggzdmod_incdirs="$ac_ggzdmod_includes $ac_ggz_stdinc"
AC_GGZ_REMOVEDUPS($ggzdmod_incdirs, ggzdmod_incdirs)
ggzdmod_header=ggzdmod.h

AC_GGZ_FIND_FILE($ggzdmod_header, $ggzdmod_incdirs, ggzdmod_incdir)
ac_ggzdmod_includes="$ggzdmod_incdir"

ggzdmod_libdirs="$ac_ggzdmod_libraries $ac_ggz_stdlib"
AC_GGZ_REMOVEDUPS($ggzdmod_libdirs, ggzdmod_libdirs)

ggzdmod_libdir=NO
for dir in $ggzdmod_libdirs; do
  try="ls -1 $dir/libggzdmod.la $dir/libggzdmod.so"
  if test -n "`$try 2> /dev/null`"; then ggzdmod_libdir=$dir; break; else echo "tried $dir" >&AC_FD_CC ; fi
done

ac_ggzdmod_libraries="$ggzdmod_libdir"

if test "$ac_ggzdmod_includes" = NO || test "$ac_ggzdmod_libraries" = NO; then
  ac_cv_have_ggzdmod="have_ggzdmod=no"
  ac_ggzdmod_notfound=""
else
  have_ggzdmod="yes"
fi
])

eval "$ac_cv_have_ggzdmod"

if test "$have_ggzdmod" != yes; then
  if test "x$2" = "xignore"; then
    AC_MSG_RESULT([$have_ggzdmod (intentionally ignored)])
  else
    AC_MSG_RESULT([$have_ggzdmod])
    if test "x$2" = "x"; then
      AC_GGZ_ERROR(ggzdmod, $ggzdmod_incdirs, $ggzdmod_libdirs)
    fi

    # Perform actions given by argument 2.
    $2
  fi
else
  ac_cv_have_ggzdmod="have_ggzdmod=yes \
    ac_ggzdmod_includes=$ac_ggzdmod_includes ac_ggzdmod_libraries=$ac_ggzdmod_libraries"
  AC_MSG_RESULT([$have_ggzdmod (libraries $ac_ggzdmod_libraries, headers $ac_ggzdmod_includes)])

  ggzdmod_libraries="$ac_ggzdmod_libraries"
  ggzdmod_includes="$ac_ggzdmod_includes"

  AC_SUBST(ggzdmod_libraries)
  AC_SUBST(ggzdmod_includes)

  GGZDMOD_INCLUDES="-isystem $ggzdmod_includes"
  GGZDMOD_LDFLAGS="-L$ggzdmod_libraries"

  AC_SUBST(GGZDMOD_INCLUDES)
  AC_SUBST(GGZDMOD_LDFLAGS)

  LIB_GGZDMOD='-lggzdmod'
  AC_SUBST(LIB_GGZDMOD)

  # Perform actions given by argument 1.
  $1
fi

])

dnl ------------------------------------------------------------------------
dnl Try to find the ggz-gtk headers and libraries.
dnl $(GGZGTK_LDFLAGS) will be -L ... (if needed)
dnl and $(GGZGTK_INCLUDES) will be -I ... (if needed)
dnl ------------------------------------------------------------------------
dnl
AC_DEFUN([AC_GGZ_GTK],
[
AC_MSG_CHECKING([for GGZ library: ggz-gtk])

ac_ggz_gtk_includes=NO ac_ggz_gtk_libraries=NO
ggz_gtk_libraries=""
ggz_gtk_includes=""

AC_ARG_WITH(ggz-gtk-dir,
    AC_HELP_STRING([--with-ggz-gtk-dir=DIR], [ggz-gtk installation prefix]),
    [  ac_ggz_gtk_includes="$withval"/include
       ac_ggz_gtk_libraries="$withval"/lib
    ])
AC_ARG_WITH(ggz-gtk-includes,
    AC_HELP_STRING([--with-ggz-gtk-includes=DIR], 
                   [where the ggz-gtk includes are]),
    [  ac_ggz_gtk_includes="$withval"
    ])
AC_ARG_WITH(ggz-gtk-libraries,
    AC_HELP_STRING([--with-ggz-gtk-libraries=DIR],
                   [where the ggz-gtk libs are]),
    [  ac_ggz_gtk_libraries="$withval"
    ])

AC_CACHE_VAL(ac_cv_have_ggz_gtk,
[
ggz_gtk_incdirs="$ac_ggz_gtk_includes $ac_ggz_stdinc"
AC_GGZ_REMOVEDUPS($ggz_gtk_incdirs, ggz_gtk_incdirs)
ggz_gtk_header=ggz-gtk.h

AC_GGZ_FIND_FILE($ggz_gtk_header, $ggz_gtk_incdirs, ggz_gtk_incdir)
ac_ggz_gtk_includes="$ggz_gtk_incdir"

ggz_gtk_libdirs="$ac_ggz_gtk_libraries $ac_ggz_stdlib"
AC_GGZ_REMOVEDUPS($ggz_gtk_libdirs, ggz_gtk_libdirs)

ggz_gtk_libdir=NO
for dir in $ggz_gtk_libdirs; do
  try="ls -1 $dir/libggz-gtk.la $dir/libggz-gtk.so"
  if test -n "`$try 2> /dev/null`"; then ggz_gtk_libdir=$dir; break; else echo "tried $dir" >&AC_FD_CC ; fi
done

ac_ggz_gtk_libraries="$ggz_gtk_libdir"

if test "$ac_ggz_gtk_includes" = NO || test "$ac_ggz_gtk_libraries" = NO; then
  ac_cv_have_ggz_gtk="have_ggz_gtk=no"
  ac_ggz_gtk_notfound=""
else
  have_ggz_gtk="yes"
fi
])

eval "$ac_cv_have_ggz_gtk"

if test "$have_ggz_gtk" != yes; then
  if test "x$2" = "xignore"; then
    AC_MSG_RESULT([$have_ggz_gtk (intentionally ignored)])
  else
    AC_MSG_RESULT([$have_ggz_gtk])
    if test "x$2" = "x"; then
      AC_GGZ_ERROR(ggz-gtk, $ggz_gtk_incdirs, $ggz_gtk_libdirs)
    fi

    # Perform actions given by argument 2.
    $2
  fi
else
  ac_cv_have_ggz_gtk="have_ggz_gtk=yes \
    ac_ggz_gtk_includes=$ac_ggz_gtk_includes ac_ggz_gtk_libraries=$ac_ggz_gtk_libraries"
  AC_MSG_RESULT([$have_ggz_gtk (libraries $ac_ggz_gtk_libraries, headers $ac_ggz_gtk_includes)])

  ggz_gtk_libraries="$ac_ggz_gtk_libraries"
  ggz_gtk_includes="$ac_ggz_gtk_includes"

  AC_SUBST(ggz_gtk_libraries)
  AC_SUBST(ggz_gtk_includes)

  GGZ_GTK_INCLUDES="-isystem $ggz_gtk_includes"
  GGZ_GTK_LDFLAGS="-L$ggz_gtk_libraries"

  AC_SUBST(GGZ_GTK_INCLUDES)
  AC_SUBST(GGZ_GTK_LDFLAGS)

  LIB_GGZ_GTK='-lggz-gtk'
  AC_SUBST(LIB_GGZ_GTK)

  # Perform actions given by argument 1.
  $1
fi
])

dnl ------------------------------------------------------------------------
dnl Setup the game server configuration.
dnl Sets ggzdconfdir (ggzd configuration).
dnl Sets ggzddatadir (for game server data).
dnl ------------------------------------------------------------------------
dnl
AC_DEFUN([AC_GGZ_SERVER],
[
AC_MSG_CHECKING([for GGZ server: ggzd])
AC_ARG_WITH(ggzd-confdir,
    AC_HELP_STRING([--with-ggzd-confdir=DIR], [directory for room/game data]),
[ ac_ggzd_confdir="$withval"
])

AC_CACHE_VAL(ac_cv_have_ggzdconf,
[
	if test "x$1" = "xforce"; then
		if test "x$ac_ggzd_confdir" = "x"; then
			ggzdconfdirs="$ac_ggz_stdetc"
		else
			ggzdconfdirs="$ac_ggzd_confdir"
		fi
	else
		ggzdconfdirs="$ac_ggzd_confdir $ac_ggz_stdetc"
	fi

	ggzdconfdir=NONE
	for dir in $ggzdconfdirs; do
		if test -n "`ls -d $dir/rooms 2> /dev/null`"; then
			if test -n "`ls -d $dir/rooms 2> /dev/null`"; then
				ggzdconfdir=$dir; break;
			else
				echo "tried $dir" >&AC_FD_CC;
			fi
		else
			echo "tried $dir" >&AC_FD_CC;
		fi
	done

	if test "x$ggzdconfdir" = "xNONE"; then
		have_ggzdconf="no"
	else
		have_ggzdconf="yes"
	fi
])

eval "$ac_cv_have_ggzdconf"

if test "$have_ggzdconf" != yes; then
	if test "x$2" = "xignore"; then
	  AC_MSG_RESULT([$have_ggzdconf (intentionally ignored)])
	elif test "x$2" = "xforce"; then
	  if test "x$ac_ggzd_confdir" = "x"; then
	    ggzdconfdir="\${prefix}/etc/ggzd"
	  else
	    ggzdconfdir=$ac_ggzd_confdir
	  fi
	  AC_MSG_RESULT([$have_ggzdconf (but forced to ${ggzdconfdir})])
	else
	  AC_MSG_RESULT([$have_ggzdconf])
      if test "x$2" = "x"; then
	    AC_MSG_ERROR([GGZ server configuration not found. Please check your installation! ])
      fi

	  # Perform actions given by argument 2.
	  $2
	fi
else
	prefixed=0
	if test "x${prefix}" != "xNONE" && test "x${prefix}" != "x${ac_default_prefix}"; then
		prefixed=1
	fi
	if test "x$ggzdconfdir" != "x${prefix}/etc/ggzd" && test "x$prefixed" = "x1"; then
		AC_MSG_RESULT([$have_ggzdconf ($ggzdconfdir, but using ${prefix}/etc/ggzd nevertheless)])
		ggzdconfdir="\${prefix}/etc/ggzd"
	else
		AC_MSG_RESULT([$have_ggzdconf ($ggzdconfdir)])
	fi
fi

if test "$have_ggzdconf" = yes || test "x$2" = "xforce"; then
	AC_SUBST(ggzdconfdir)

	ggzddatadir=${prefix}/share/${PACKAGE}
	AC_DEFINE_UNQUOTED(GGZDDATADIR, "${ggzddatadir}", [Game server data directory])
	AC_SUBST(ggzddatadir)

	if test "x${libdir}" = 'x${exec_prefix}/lib'; then
	  if test "x${exec_prefix}" = "xNONE"; then
	    if test "x${prefix}" = "xNONE"; then
	      ggzdexecmoddir="\${ac_default_prefix}/lib/ggzd"
		  ggzdexecmodpath="${ac_default_prefix}/lib/ggzd"
	    else
	      ggzdexecmoddir="\${prefix}/lib/ggzd"
		  ggzdexecmodpath="${prefix}/lib/ggzd"
	    fi
	  else
	    ggzdexecmoddir="\${exec_prefix}/lib/ggzd"
		ggzdexecmodpath="${exec_prefix}/lib/ggzd"
	  fi
	else
	  ggzdexecmoddir="\${libdir}/ggzd"
	  ggzdexecmodpath="${libdir}/ggzd"
	fi
	AC_SUBST(ggzdexecmoddir)
	AC_SUBST(ggzdexecmodpath)

	# Perform actions given by argument 1.
	$1
fi

])

dnl ------------------------------------------------------------------------
dnl Find internationalization tools
dnl ------------------------------------------------------------------------
dnl
AC_DEFUN([AC_GGZ_INTL],
[
AC_PATH_PROG(GETTEXT, xgettext)
AC_PATH_PROG(MSGFMT, msgfmt)
AC_PATH_PROG(MSGMERGE, msgmerge)

intl=1
if test "x$GETTEXT" = "x"; then intl=0; fi
if test "x$MSGFMT" = "x"; then intl=0; fi
if test "x$MSGMERGE" = "x"; then intl=0; fi
AM_ICONV
LIBS="$LIBICONV $LIBS"
AC_CHECK_LIB(intl, gettext, [LIBS="-lintl $LIBS"])
AC_CHECK_FUNCS([gettext ngettext], [], [intl=0])
AC_CHECK_HEADERS([libintl.h locale.h])
if test "$intl" = 0; then
  if test "x$2" = "xignore"; then
    AC_MSG_WARN([Internationalization tools missing. (ignored)])
  else
    AC_MSG_RESULT([Internationalization tools missing.])
    if test "x$2" = "x"; then
      AC_MSG_ERROR([Internationalization tools missing.])
    fi

    # Perform actions given by argument 2.
    $2
  fi
else
  AC_MSG_RESULT([Internationalization tools found.])

  XGETTEXT=$GETTEXT
  GMSGFMT=$MSGFMT

  AC_SUBST(XGETTEXT)
  AC_SUBST(GETTEXT)
  AC_SUBST(GMSGFMT)
  AC_SUBST(MSGFMT)
  AC_SUBST(MSGMERGE)

  AC_DEFINE(ENABLE_NLS, 1, [Define if NLS is enabled])

  # Perform actions given by argument 1.
  $1
fi

])

# AC_GGZ_CHECK
#   Check for presence of GGZ client and server libraries.
#
#   Simply call this function in programs that use GGZ.  GGZ_SERVER and
#   GGZ_CLIENT will be #defined in config.h, and created as conditionals
#   in Makefile.am files.
#
#   The only argument accepted gives the frontend for client embedding:
#      "gtk" => means the libggz-gtk library will be checked
AC_DEFUN([AC_GGZ_CHECK],
[
  AC_GGZ_INIT
  AC_GGZ_LIBGGZ([try_ggz="yes"], [try_ggz="no"])

  ggz_server="no"
  ggz_client="no"
  AC_ARG_WITH(ggz-server,
              AC_HELP_STRING([--with-ggz-server], [Force GGZ server support]),
              [try_ggz_server=$withval])
  AC_ARG_WITH(ggz-client,
              AC_HELP_STRING([--with-ggz-client], [Force GGZ client support]),
              [try_ggz_client=$withval])

  if test "x$try_ggz_server" != "xno"; then
    if test "$try_ggz" = "yes"; then
      # Must pass something as the action-if-failed, or the macro will exit
      AC_GGZ_GGZDMOD([ggz_server="yes"], [ggz_server="no"])
    fi
    if test "$ggz_server" = "yes"; then
      AC_DEFINE(GGZ_SERVER, 1, [Server support for GGZ])
    else
      if test "$try_ggz_server" = "yes"; then
        AC_MSG_ERROR([Could not configure GGZ server support. See above messages.])
      fi
    fi
  fi

  if test "x$try_ggz_client" != "xno"; then
    if test "$try_ggz" = "yes"; then
      # Must pass something as the action-if-failed, or the macro will exit
      AC_GGZ_GGZMOD([AC_GGZ_CONFIG([ggz_client="yes"], [ggz_client="no"])],
                    [ggz_client="no"])
    fi
    if test "$ggz_client" = "yes"; then
      AC_DEFINE(GGZ_CLIENT, 1, [Client support for GGZ])
    else
      if test "$try_ggz_client" = "yes"; then
        AC_MSG_ERROR([Could not configure GGZ client support. See above messages.])
      fi
    fi
  fi

  ggz_gtk="no"
  if test "$ggz_client" = "yes"; then
    if test "x$1" = "xgtk"; then
      AC_GGZ_GTK([ggz_gtk="yes"])
      if test $ggz_gtk = "yes"; then
        AC_DEFINE(GGZ_GTK, 1, [Support for embedded GGZ through libggz-gtk])
      fi
    fi
  fi

  AM_CONDITIONAL(GGZ_CLIENT, test "$ggz_client" = "yes")
  AM_CONDITIONAL(GGZ_SERVER, test "$ggz_server" = "yes")
  AM_CONDITIONAL(GGZ_GTK, test "$ggz_gtk" = "yes")
])
#serial AM2
dnl From Bruno Haible.

AC_DEFUN([AM_ICONV],
[
  dnl Some systems have iconv in libc, some have it in libiconv (OSF/1 and
  dnl those with the standalone portable GNU libiconv installed).

  AC_ARG_WITH([libiconv-prefix],
[  --with-libiconv-prefix=DIR  search for libiconv in DIR/include and DIR/lib], [
    for dir in `echo "$withval" | tr : ' '`; do
      if test -d $dir/include; then CPPFLAGS="$CPPFLAGS -I$dir/include"; fi
      if test -d $dir/lib; then LDFLAGS="$LDFLAGS -L$dir/lib"; fi
    done
   ])

  AC_CACHE_CHECK(for iconv, am_cv_func_iconv, [
    am_cv_func_iconv="no, consider installing GNU libiconv"
    am_cv_lib_iconv=no
    AC_TRY_LINK([#include <stdlib.h>
#include <iconv.h>],
      [iconv_t cd = iconv_open("","");
       iconv(cd,NULL,NULL,NULL,NULL);
       iconv_close(cd);],
      am_cv_func_iconv=yes)
    if test "$am_cv_func_iconv" != yes; then
      am_save_LIBS="$LIBS"
      LIBS="$LIBS -liconv"
      AC_TRY_LINK([#include <stdlib.h>
#include <iconv.h>],
        [iconv_t cd = iconv_open("","");
         iconv(cd,NULL,NULL,NULL,NULL);
         iconv_close(cd);],
        am_cv_lib_iconv=yes
        am_cv_func_iconv=yes)
      LIBS="$am_save_LIBS"
    fi
  ])
  if test "$am_cv_func_iconv" = yes; then
    AC_DEFINE(HAVE_ICONV, 1, [Define if you have the iconv() function.])
    AC_MSG_CHECKING([for iconv declaration])
    AC_CACHE_VAL(am_cv_proto_iconv, [
      AC_TRY_COMPILE([
#include <stdlib.h>
#include <iconv.h>
extern
#ifdef __cplusplus
"C"
#endif
#if defined(__STDC__) || defined(__cplusplus)
size_t iconv (iconv_t cd, char * *inbuf, size_t *inbytesleft, char * *outbuf, size_t *outbytesleft);
#else
size_t iconv();
#endif
], [], am_cv_proto_iconv_arg1="", am_cv_proto_iconv_arg1="const")
      am_cv_proto_iconv="extern size_t iconv (iconv_t cd, $am_cv_proto_iconv_arg1 char * *inbuf, size_t *inbytesleft, char * *outbuf, size_t *outbytesleft);"])
    am_cv_proto_iconv=`echo "[$]am_cv_proto_iconv" | tr -s ' ' | sed -e 's/( /(/'`
    AC_MSG_RESULT([$]{ac_t:-
         }[$]am_cv_proto_iconv)
    AC_DEFINE_UNQUOTED(ICONV_CONST, $am_cv_proto_iconv_arg1,
      [Define as const if the declaration of iconv() needs const.])
  fi
  LIBICONV=
  if test "$am_cv_lib_iconv" = yes; then
    LIBICONV="-liconv"
  fi
  AC_SUBST(LIBICONV)
])
# $Id: mkdir.m4 5859 2004-02-09 01:38:22Z jdorje $
# FUNC_MKDIR_TAKES_ONE_ARG defines MKDIR_TAKES_ONE_ARG if, well, mkdir takes
# one arg (instead of 2 like it does on POSIX systems).
#
# Take from a phantom file contributed to GNU "patch" that I can't find
# anywhere except in mailing lists.  Attributed to Mumit Khan and Paul Eggert.
#
# Note that if you don't have the correct #includes in the test-compile code,
# the compiler will give a missing-prototype warning but will succeed.  Yuck!

AC_DEFUN([FUNC_MKDIR_TAKES_ONE_ARG],
  [AC_CHECK_FUNCS([mkdir])
   AC_CACHE_CHECK([whether mkdir takes only one argument],
     cv_mkdir_takes_one_arg,
     [cv_mkdir_takes_one_arg=no
      if test $ac_cv_func_mkdir = yes; then
 AC_TRY_COMPILE([
#include <dir.h>
   ],
   [mkdir (".");],
   cv_mkdir_takes_one_arg=yes,
   cv_mkdir_takes_one_arg=no
   )
      fi
     ]
   )
   if test $cv_mkdir_takes_one_arg = yes; then
     AC_DEFINE([MKDIR_TAKES_ONE_ARG], 1,
       [Define if mkdir takes only one argument.])
   fi
  ]
)
dnl ======================================
dnl GGZ Gaming Zone - Configuration Macros
dnl ======================================
dnl
dnl Copyright (C) 2005 Josef Spillner, josef@ggzgamingzone.org
dnl This file has been derived from libggz configuration macros.
dnl It is published under the conditions of the GNU General Public License.
dnl
dnl ======================================
dnl
dnl This file contains operating system specific checks such as for system
dnl libraries, header file flavours and non-POSIX/Unix handling.
dnl
dnl ======================================
dnl
dnl History:
dnl 2005-09-14: created from libggz's configure.ac file

dnl ------------------------------------------------------------------------
dnl Content of this file:
dnl ------------------------------------------------------------------------
dnl AC_GGZ_PLATFORM_BASE - minimal required call
dnl AC_GGZ_PLATFORM_POSIX - checks for various types and functions
dnl AC_GGZ_PLATFORM_POSIX_NET - network-related checks
dnl AC_GGZ_PLATFORM_POSIX_LIBC - system libraries in addition to libc
dnl AC_GGZ_PLATFORM_WIN32 - for cross-compilation on win32 systems
dnl AC_GGZ_PLATFORM - convenience macro to call all of the above
dnl   (except for _BASE which MUST be called before AC_PROG_LIBTOOL)
dnl

dnl ------------------------------------------------------------------------
dnl AC_GGZ_PLATFORM_BASE
dnl ------------------------------------------------------------------------
dnl
AC_DEFUN([AC_GGZ_PLATFORM_BASE],
[
AC_PROG_CC
AC_PROG_CXX
AC_PROG_INSTALL
AC_PROG_LN_S
AC_PROG_CPP
AC_LIBTOOL_WIN32_DLL
#AC_PROG_LIBTOOL # bug?
AC_PROG_RANLIB
AC_PROG_MAKE_SET
])

dnl ------------------------------------------------------------------------
dnl AC_GGZ_PLATFORM_WIN32
dnl ------------------------------------------------------------------------
dnl
AC_DEFUN([AC_GGZ_PLATFORM_WIN32],
[
  case $host_os in
    *mingw32* )
      MINGW32=yes
      ;;
    * )
      MINGW32=no
      ;;
  esac

  if test "$MINGW32" = "yes"; then
    LDFLAGS="$LDFLAGS -no-undefined"
    LDADD="$LDADD -lws2_32"
  fi
])

dnl ------------------------------------------------------------------------
dnl AC_GGZ_PLATFORM_POSIX
dnl ------------------------------------------------------------------------
dnl
AC_DEFUN([AC_GGZ_PLATFORM_POSIX],
[
# Check for header files
# ======================
#AC_CHECK_HEADERS([fcntl.h sys/param.h],
#  [],
#  [AC_MSG_ERROR([cannot find required header file])])
#AC_CHECK_HEADERS([arpa/inet.h netdb.h netinet/in.h sys/shm.h sys/socket.h])
#AC_CHECK_HEADERS([sys/uio.h sys/un.h sys/wait.h winsock2.h])

# Check for functions
# ===================
#AC_CHECK_FUNCS([getpagesize memmove memset mkdir stat strcasecmp strerror strstr strchr],
#  [],
#  [AC_MSG_ERROR([cannot find required function])])
#AC_CHECK_FUNCS([gethostbyname socket]) # These fail; dunno why
#AC_CHECK_FUNCS([alarm getuid inet_pton])

# Check for more functions
# ========================
#AC_FUNC_FORK
##AC_FUNC_MALLOC # use ggz_malloc() always
#AC_FUNC_MEMCMP
#AC_FUNC_REALLOC
#AC_FUNC_STAT
FUNC_MKDIR_TAKES_ONE_ARG # FIXME: external

# Check for typedefs, structures, and compiler characteristics
# ============================================================
#AC_C_CONST
#AC_TYPE_MODE_T
#AC_TYPE_PID_T
#AC_TYPE_SIGNAL
#AC_TYPE_SIZE_T
AC_C99_VARIABLE_ARRAYS # FIXME: external

# Check for debug types in syslog.h
# =================================
# If syslog is present, we take the log types from it.
# If not, define our own enumeration.
AC_CHECK_HEADER([syslog.h],
  [
    AC_DEFINE([GGZ_HAVE_SYSLOG_H], 1, [Check if syslog is present])
  ],
  [])
])

dnl ------------------------------------------------------------------------
dnl AC_GGZ_PLATFORM_POSIX_NET
dnl ------------------------------------------------------------------------
dnl
AC_DEFUN([AC_GGZ_PLATFORM_POSIX_NET],
[
# Check for sendmsg
# =================
# If sendmsg() is present, we can send file descriptors over a local socket
# via ggz_write_fd.  If not we disable this functionality.
#AC_MSG_CHECKING([for network feature: sendmsg])
AC_CHECK_FUNC([sendmsg],
  [
    AC_DEFINE([GGZ_HAVE_SENDMSG], 1, [Check if file descriptors can be sent])
  ],
  [])

# Check for PF_LOCAL/PF_UNIX
# ==========================
# There should be an autoconf macro to check this???
AC_MSG_CHECKING([for network feature: PF_LOCAL])
have_localsockets=0
AC_COMPILE_IFELSE(
  AC_LANG_PROGRAM([[#include <sys/socket.h>]], [[int a = PF_LOCAL;]]),
  [have_localsockets=1],
  [
    AC_COMPILE_IFELSE(
      AC_LANG_PROGRAM([[#include <sys/socket.h>]], [[int a = PF_UNIX;]]),
      [
        AC_DEFINE([PF_LOCAL], PF_UNIX, [PF_LOCAL is available via PF_UNIX])
        AC_DEFINE([AF_LOCAL], AF_UNIX, [AF_LOCAL is available via AF_UNIX])
        have_localsockets=1
      ],
      [])
  ])
if test "x$have_localsockets" = "x1"; then
  AC_DEFINE([GGZ_HAVE_PF_LOCAL], 1, [Check if local sockets are supported])
  AC_MSG_RESULT(yes)
else
  AC_MSG_RESULT(no)
fi

# Check for SUN_LEN
# =================
AC_MSG_CHECKING([for network feature: SUN_LEN])
AC_LINK_IFELSE(
  AC_LANG_PROGRAM(
    [[
      #include<sys/types.h>
      #include<sys/un.h>
    ]],
    [[struct sockaddr_un su; int i = SUN_LEN(&su);]]),
  [
    AC_MSG_RESULT(yes)
    AC_DEFINE([HAVE_SUN_LEN], 1, [Define if the SUN_LEN macro exists])
  ],
  [AC_MSG_RESULT(no)])

# Check for msghdr member msg_controllen and friends
# ==================================================
AC_MSG_CHECKING([for network feature: msg_controllen])
have_controllen=0
AC_COMPILE_IFELSE(
  AC_LANG_PROGRAM([[#include <sys/socket.h>]],
    [[struct msghdr m; m.msg_controllen = 0;]]),
  [have_controllen=1],
  [])

if test "x$have_controllen" = "x1"; then
  AC_DEFINE([HAVE_MSGHDR_MSG_CONTROL], 1, [Define if msghdr has a msg_controllen member])
  AC_MSG_RESULT(yes)

  AC_MSG_CHECKING([for network feature: CMSG_ALIGN])
  AC_LINK_IFELSE(
    AC_LANG_PROGRAM(
      [[
        #include<sys/types.h>
        #include<sys/socket.h>
      ]],
      [[int i = CMSG_ALIGN(0);]]),
    [
      AC_MSG_RESULT(yes)
      AC_DEFINE([HAVE_CMSG_ALIGN], 1, [Define if CMSG_ALIGN is available])
    ],
    [AC_MSG_RESULT(no)])

  AC_MSG_CHECKING([for network feature: CMSG_LEN])
  AC_LINK_IFELSE(
    AC_LANG_PROGRAM(
      [[
        #include<sys/types.h>
        #include<sys/socket.h>
      ]],
      [[int i = CMSG_LEN(0);]]),
    [
      AC_MSG_RESULT(yes)
      AC_DEFINE([HAVE_CMSG_LEN], 1, [Define if CMSG_LEN is available])
    ],
    [AC_MSG_RESULT(no)])

  AC_MSG_CHECKING([for network feature: CMSG_SPACE])
  AC_LINK_IFELSE(
    AC_LANG_PROGRAM(
      [[
        #include<sys/types.h>
        #include<sys/socket.h>
      ]],
      [[int i = CMSG_SPACE(0);]]),
    [
      AC_MSG_RESULT(yes)
      AC_DEFINE([HAVE_CMSG_SPACE], 1, [Define if CMSG_SPACE is available])
    ],
    [AC_MSG_RESULT(no)])
else
  AC_MSG_RESULT(no)
fi
])

dnl ------------------------------------------------------------------------
dnl AC_GGZ_PLATFORM_POSIX_LIBC
dnl ------------------------------------------------------------------------
dnl
AC_DEFUN([AC_GGZ_PLATFORM_POSIX_LIBC],
[
# Check for library functions
# ===========================
# On some systems additional libraries may be needed.  Simply include them
# if they're there.
AC_CHECK_LIB(socket, socket, [LDADD="$LDADD -lsocket"])
AC_CHECK_LIB(nsl, gethostbyname, [LDADD="$LDADD -lnsl"])
])

dnl ------------------------------------------------------------------------
dnl AC_GGZ_PLATFORM
dnl ------------------------------------------------------------------------
dnl
AC_DEFUN([AC_GGZ_PLATFORM],
[
#AC_GGZ_PLATFORM_BASE
AC_GGZ_PLATFORM_POSIX
AC_GGZ_PLATFORM_POSIX_NET
AC_GGZ_PLATFORM_POSIX_LIBC
AC_GGZ_PLATFORM_WIN32
])

dnl ======================================
dnl GGZ Gaming Zone - Configuration Macros
dnl ======================================
dnl
dnl Copyright (C) 2001 - 2005 Josef Spillner, josef@ggzgamingzone.org
dnl This file has heavily been inspired by KDE's acinclude :)
dnl It is published under the conditions of the GNU General Public License.
dnl
dnl ======================================
dnl
dnl This file contains all autoconf macros needed for any security
dnl implementation such as TLS/SSL.
dnl
dnl ======================================
dnl
dnl History:
dnl 2002-02-10: lookup TLS libraries; taken code from acinclude.ggz
dnl 2002-02-24: default to GnuTLS; create conditional compile vars
dnl 2005-09-14: several cleanups due to newer autotools

dnl ------------------------------------------------------------------------
dnl Content of this file:
dnl ------------------------------------------------------------------------
dnl AC_GGZ_TLS - find a TLS implementation (support for gnutls, openssl and
dnl              none)
dnl AC_PATH_SSL - OpenSSL implementation backend (code from kdelibs)
dnl AC_GGZ_GNUTLS - GNUTLS implementation backend
dnl

dnl ------------------------------------------------------------------------
dnl Find a single file
dnl ------------------------------------------------------------------------
dnl
AC_DEFUN([AC_GGZ_FIND_FILE],
[
$3=NO
for i in $2;
do
  for j in $1;
  do
    echo "configure: __oline__: $i/$j" >&AC_FD_CC
    if test -r "$i/$j"; then
      echo "taking that" >&AC_FD_CC
      $3=$i
      break 2
    fi
  done
done
])

dnl ------------------------------------------------------------------------
dnl Try to find the SSL headers and libraries.
dnl Exported are $(ssl_includes), $(ssl_libraries) and $(ssl_lib).
dnl ------------------------------------------------------------------------
dnl
AC_DEFUN([AC_PATH_SSL],
[
LIBSSL="-lssl -lcrypto"

ac_ssl_includes=NO ac_ssl_libraries=NO
ssl_libraries=""
ssl_includes=""
AC_ARG_WITH([ssl-dir],
  AC_HELP_STRING([--with-ssl-dir=DIR], [where the root of OpenSSL is installed]),
  [
    ac_ssl_includes="$withval"/include
    ac_ssl_libraries="$withval"/lib
  ],
  [])

AC_CACHE_VAL(ac_cv_have_ssl,
[#try to guess OpenSSL locations
  
  ssl_incdirs="/usr/include /usr/local/include /usr/ssl/include /usr/local/ssl/include $prefix/include $kde_extra_includes"
  ssl_incdirs="$ac_ssl_includes $ssl_incdirs"
  AC_GGZ_FIND_FILE(openssl/ssl.h, $ssl_incdirs, ssl_incdir)
  ac_ssl_includes="$ssl_incdir"

  ssl_libdirs="/usr/lib /usr/local/lib /usr/ssl/lib /usr/local/ssl/lib $prefix/lib $exec_prefix/lib $kde_extra_libs"
  if test ! "$ac_ssl_libraries" = "NO"; then
    ssl_libdirs="$ac_ssl_libraries $ssl_libdirs"
  fi

  test=NONE
  ssl_libdir=NONE
  for dir in $ssl_libdirs; do
    try="ls -1 $dir/libssl*"
    if test=`eval $try 2> /dev/null`; then ssl_libdir=$dir; break; else echo "tried $dir" >&AC_FD_CC ; fi
  done

  ac_ssl_libraries="$ssl_libdir"

  AC_LANG_SAVE
  AC_LANG_C

  ac_cflags_safe="$CFLAGS"
  ac_ldflags_safe="$LDFLAGS"
  ac_libs_safe="$LIBS"

  CFLAGS="$CFLAGS -I$ssl_incdir $all_includes"
  LDFLAGS="-L$ssl_libdir $all_libraries"
  LIBS="$LIBS $LIBSSL -lRSAglue -lrsaref"

  AC_TRY_LINK(,void RSAPrivateEncrypt(void);RSAPrivateEncrypt();,
  ac_ssl_rsaref="yes"
  ,
  ac_ssl_rsaref="no"
  )

  CFLAGS="$ac_cflags_safe"
  LDFLAGS="$ac_ldflags_safe"
  LIBS="$ac_libs_safe"

  AC_LANG_RESTORE

  if test "$ac_ssl_includes" = NO || test "$ac_ssl_libraries" = NO; then
    have_ssl=no
  else
    have_ssl=yes;
  fi

])

eval "$ac_cv_have_ssl"

if test "$have_ssl" = yes; then
  dnl Check for SSL version
  AC_CACHE_VAL(ac_cv_ssl_version,
  [
    AC_LANG_SAVE
    AC_LANG_C 

    cat >conftest.$ac_ext <<EOF
#include <openssl/opensslv.h>
#include <stdio.h>
    int main() {
 
#ifndef OPENSSL_VERSION_NUMBER
      printf("ssl_version=\\"error\\"\n");
#else
      if (OPENSSL_VERSION_NUMBER < 0x00906000)
        printf("ssl_version=\\"old\\"\n");
      else
        printf("ssl_version=\\"ok\\"\n");
#endif
     return (0);
    }
EOF

    ac_compile='${CC-gcc} $CFLAGS -I$ac_ssl_includes conftest.$ac_ext -o conftest'
    if AC_TRY_EVAL(ac_compile); then 

      if eval `./conftest 2>&5`; then
        if test $ssl_version = error; then
		  have_ssl=no
        else
          if test $ssl_version = old; then
            have_ssl=no
          fi
        fi
        ac_cv_ssl_version="ssl_version=$ssl_version"
      else
		have_ssl=no
      fi

    else
	  have_ssl=no
    fi 

    AC_LANG_RESTORE

  ])

  eval "$ac_cv_ssl_version"
fi

if test "$have_ssl" != yes; then
  LIBSSL="";
else
  AC_DEFINE(HAVE_SSL, 1, [If we are going to use OpenSSL])
  ac_cv_have_ssl="have_ssl=yes \
    ac_ssl_includes=$ac_ssl_includes ac_ssl_libraries=$ac_ssl_libraries ac_ssl_rsaref=$ac_ssl_rsaref"
  
  
  ssl_libraries="$ac_ssl_libraries"
  ssl_includes="$ac_ssl_includes"

  if test "$ac_ssl_rsaref" = yes; then
    LIBSSL="-lssl -lcrypto -lRSAglue -lrsaref" 
  fi

  if test $ssl_version = "old"; then
    AC_DEFINE(HAVE_OLD_SSL_API, 1, [Define if you have OpenSSL < 0.9.6])
  fi
fi

# got ssl_includes
# got ssl_libraries
ssl_lib=$LIBSSL

])

dnl ------------------------------------------------------------------------
dnl Try to find the GNUTLS headers and libraries.
dnl Exported are $(gnutls_includes), $(gnutls_libraries) and $(gnutls_lib).
dnl ------------------------------------------------------------------------
dnl
AC_DEFUN([AC_GGZ_GNUTLS],
[
ac_gnutls_includes=NO ac_gnutls_libraries=NO
gnutls_libraries=""
gnutls_includes=""
gnutls_lib=""

AC_ARG_WITH([gnutls-dir],
  AC_HELP_STRING([--with-gnutls-dir=DIR], [gnutls installation prefix]),
  [
    ac_gnutls_includes="$withval"/include/gnutls
    ac_gnutls_libraries="$withval"/lib
  ],
  [])
AC_ARG_WITH([gnutls-includes],
    AC_HELP_STRING([--with-gnutls-includes=DIR], [where the gnutls includes are.]),
    [
      ac_gnutls_includes="$withval"
    ],
    [])
AC_ARG_WITH([gnutls-libraries],
    AC_HELP_STRING([--with-gnutls-libraries=DIR], [where the gnutls libs are.]),
    [
      ac_gnutls_libraries="$withval"
    ],
    [])

AC_CACHE_VAL(ac_cv_have_gnutls,
[
if test "x${prefix}" = "xNONE"; then
   prefix_incdir="${ac_default_prefix}/include"
   prefix_libdir="${ac_default_prefix}/lib"
else
   prefix_incdir="${prefix}/include"
   prefix_libdir="${prefix}/lib"
fi
gnutls_incdirs="$ac_gnutls_includes $prefix_incdir/gnutls /usr/local/include/gnutls /usr/include/gnutls"
gnutls_header=gnutls.h

AC_GGZ_FIND_FILE($gnutls_header, $gnutls_incdirs, gnutls_incdir)
ac_gnutls_includes="$gnutls_incdir"

gnutls_libdirs="$ac_gnutls_libraries $prefix_libdir /usr/local/lib /usr/lib"

test=NONE
gnutls_libdir=NONE
for dir in $gnutls_libdirs; do
  try="ls -1 $dir/libgnutls.*"
  if test -n "`$try 2> /dev/null`"; then gnutls_libdir=$dir; break; else echo "tried $dir" >&AC_FD_CC ; fi
done

ac_gnutls_libraries="$gnutls_libdir"

if test "$ac_gnutls_includes" = NO || test "$ac_gnutls_libraries" = NO; then
  ac_cv_have_gnutls="have_gnutls=no"
else
  have_gnutls="yes"
fi
])

eval "$ac_cv_have_gnutls"

if test "$have_gnutls" != yes; then
  have_gnutls=no
else
  ac_cv_have_gnutls="have_gnutls=yes \
    ac_gnutls_includes=$ac_gnutls_includes ac_gnutls_libraries=$ac_gnutls_libraries"

  gnutls_libraries="$ac_gnutls_libraries"
  gnutls_includes="$ac_gnutls_includes"
  gnutls_lib="-lgnutls"
fi

])

dnl ------------------------------------------------------------------------
dnl Try to find a suitable TLS implementation.
dnl $(GGZTLS_LDFLAGS) will be -L ... (if needed)
dnl and $(GGZTLS_INCLUDES) will be -I ... (if needed)
dnl $(GGZTLS_SOURCES) is the dynamic list of source files, depending on the
dnl used implementation
dnl ------------------------------------------------------------------------
dnl
AC_DEFUN([AC_GGZ_TLS],
[
AC_MSG_CHECKING([for GGZ TLS implementation])

AC_ARG_WITH([tls],
  AC_HELP_STRING([--with-tls@<:@=ARG@:>@], [GnuTLS or OpenSSL - auto if no ARG]),
  [tls_type=$withval],
  [tls_type=no])

dnl None (defaults)
GGZTLS_INCLUDES=""
GGZTLS_LDFLAGS=""
LIB_GGZTLS=""
TLS_TYPE="no"

if test "$tls_type" = yes -o "$tls_type" = GnuTLS; then
  dnl GNUTLS check
  AC_GGZ_GNUTLS
  if test "$have_gnutls" = yes; then
    GGZTLS_INCLUDES="-I $gnutls_includes"
    GGZTLS_LDFLAGS="-L $gnutls_libraries"
    LIB_GGZTLS=$gnutls_lib
    AC_MSG_RESULT([using GnuTLS])
    AC_DEFINE_UNQUOTED([GGZ_TLS_GNUTLS], 1,
		       [Define if GNUTLS is to be used])
    TLS_TYPE="GnuTLS"
  fi
fi

if test \( "$tls_type" = yes -a "$have_gnutls" = no \) -o "$tls_type" = OpenSSL
then
  dnl OpenSSL check
  AC_PATH_SSL
  if test "$have_ssl" = yes; then
    GGZTLS_INCLUDES="-I $ssl_includes"
    GGZTLS_LDFLAGS="-L $ssl_libraries"
    LIB_GGZTLS=$ssl_lib
    AC_MSG_RESULT([using OpenSSL])
    AC_DEFINE_UNQUOTED([GGZ_TLS_OPENSSL], 1,
		      [Define if OpenSSL is to be used])
    TLS_TYPE="OpenSSL"
  fi
fi

if test "$TLS_TYPE" = no; then
  if test "$tls_type" = no; then
    AC_MSG_RESULT([no])
  else
    AC_MSG_WARN([No TLS implementation found - using 'no'!])
  fi
  AC_DEFINE_UNQUOTED([GGZ_TLS_NONE], 1,
		     [Define if no TLS is to be used])
fi


AC_SUBST(GGZTLS_INCLUDES)
AC_SUBST(GGZTLS_LDFLAGS)
AC_SUBST(LIB_GGZTLS)

])

