/*****************************************************************************************************************************************
 ctoolbar.h

 CClass                                         CClass
 |--CSerialized                                 |--CSerialized
    |--CMetaModule                                 |--CMetaModule
       |--CObject                                     |--CObjectListener
          |--CComponent                                  |
             |--CControl                                 |
                |--CWidget                               |--CWidgetListener
                   |--CContainer                            |
		      |--CLayout			    |
		      |  |--CToolBar			    |
		      |--CButton			    |--CButtonListener
			 |--CToolButton
			    |--CToolToggleButton
			    |  |--CToolRadioButton
			    |--CToolSeparatorButton

 Classical toolbar definition. A toolbar handles CToolButton instances. It is implemented as a layout i.e. an evoluted container in order
 to be able to specify dnd intelligent insertion points and not just default container append mode on the listener part.
 For CToolRadioButton instances, it is the CToolSeparatorButton definition that acts for implicit logical groups separation.
 In order to do less listener writing and because tool buttons often do what is already defined by a menu item job, the tool button could
 be associated to a menu item instance so that the click event is redirected to the menu item listener avoiding duplicating event handling.
*****************************************************************************************************************************************/

#ifndef __CTOOLBAR_H__
#define __CTOOLBAR_H__

#include "cbutton.h"
#include "clayout.h"
#include "cmenu.h"

//-----------------------------------------------------------------------------------------------------------------------------------------
// what's defined here
//-----------------------------------------------------------------------------------------------------------------------------------------
class CToolBar;
class CToolButton;
class CToolToggleButton;
class CToolRadioButton;
class CToolSeparatorButton;

//-----------------------------------------------------------------------------------------------------------------------------------------
// ctoolbar xml serialization constants
//-----------------------------------------------------------------------------------------------------------------------------------------
// <ctoolbar orientation="horizontal|vertical" tooltips-enable="true|false" style="icons|text|both|both-horiz" 
//	     icon-size="menu|small-toolbar|large-toolbar|button|dnd|dialog">
//    ...
// </ctoolbar>
//-----------------------------------------------------------------------------------------------------------------------------------------
static CString XML_TOOLBAR_ELEMENT			("ctoolbar");
static CString XML_TOOLBAR_ATTR_ORIENTATION		("orientation");
static CString XML_TOOLBAR_ATTR_STYLE			("style");
static CString XML_TOOLBAR_ATTR_ICON_SIZE		("icon-size");

//-----------------------------------------------------------------------------------------------------------------------------------------
// ctoolbutton xml serialization constants
//-----------------------------------------------------------------------------------------------------------------------------------------
// <ctoolbutton stock-id="gtk_string_id" tooltips-text="string" tooltips-private="string" menu-item="string">
//    ...
// </ctoolbutton>
//-----------------------------------------------------------------------------------------------------------------------------------------
static CString XML_TOOLBUTTON_ELEMENT			("ctoolbutton");
static CString XML_TOOLBUTTON_ATTR_STOCKID		("stock-id");
static CString XML_TOOLBUTTON_ATTR_TOOLTIPS_TEXT	("tooltips-text");
static CString XML_TOOLBUTTON_ATTR_TOOLTIPS_PRIVATE	("tooltips-private");
static CString XML_TOOLBUTTON_ATTR_MENUITEM		("menu-item");

//-----------------------------------------------------------------------------------------------------------------------------------------
// ctooltogglebutton xml serialization constants
//-----------------------------------------------------------------------------------------------------------------------------------------
// <ctooltogglebutton checked="bool">
//    ...
// </ctooltogglebutton>
//-----------------------------------------------------------------------------------------------------------------------------------------
static CString XML_TOOLTOGGLEBUTTON_ELEMENT		("ctooltogglebutton");
static CString XML_TOOLTOGGLEBUTTON_ATTR_CHECKED	("checked");

/*****************************************************************************************************************************************
 Tool buttons defintions
*****************************************************************************************************************************************/

//---------------------------------------------------------------------------------------------------------------------------------------
// CToolButton class
//---------------------------------------------------------------------------------------------------------------------------------------
class CToolButton : public CButton
{
	// instanciation section
	public :

		CToolButton			(CToolBar *inOwner=NULL, const CString &inStockId=CString(),
						 const CMenuItem *inMenuItem=NULL, const CButtonListener *inListener=NULL);
		virtual ~CToolButton		();

	// protected gtk widget handling
	protected :

		// gtk widget instanciation and initialization
		virtual GtkWidget *		PerformWidgetInstanciate	();
		virtual void			PerformWidgetInitialize		();

		// specific container handling
		virtual void			PerformContainerAdd		(CWidget *inChild);
		virtual void			PerformContainerRemove		(CWidget *inChild);
		
	// ccomponent redefinition
	public :

		// expected owner type : CToolBar
		virtual CMetaClasses		OwnerMustBe			() const;

	// ccontrol redefinition
	public :
		
		// draggable state affectation of the tool button
		virtual void			SetDraggable			(const Bool inDraggable);

	// cbutton redefinition
	public :

		// button caption
		virtual void			SetCaption			(const CString &inCaption);
		virtual CString			GetCaption			() const;

		// OnClick event sender
		virtual void			Click				();

	// ctoolbutton specifics
	public :

		// tool button tool tips
		virtual void			SetToolTips			(const CString &inText, const CString &inPrivate=CString());
		CString				GetToolTipsText			() const;
		CString				GetToolTipsPrivate		() const;

		// menu item tool button association : the OnClick button event is not performed on the button listener anymore but
		// redirected to the menu item listener if any as if it was the menu item that received the activate notification event
		virtual void			SetMenuItem			(const CMenuItem *inMenuItem);
		CMenuItem *			GetMenuItem			() const;

	// cserialized redefinition
	public :

		// specific ctoolbutton serialization
		virtual void			Serialize			(CXMLElementNode *&ioXMLElementNode, const int inMode)
										 THROWABLE;

	// protected attributes
	protected :

		// tool item stock id if any
		CString				m_StockId;

	// protected signal handlers
	protected :

		static void			OnClick				(GtkToolButton *, gpointer);

		// associated tooltips
		GtkTooltips *			m_GtkToolTips;
		CString				m_ToolTipsText;
		CString				m_ToolTipsPrivate;

		// potential associated menu item
		CMenuItem *			m_MenuItem;

		// serialization caption lock flag
		bool				m_CaptionLock;

		// dynamic metaclass association
		SECTION_DYNAMIC_METACLASS;
};

// metaclass and class tag declaration
DECLARE_DYNAMIC_METACLASS ('tlbn', CToolButton, CButton);

//---------------------------------------------------------------------------------------------------------------------------------------
// CToolToggleButton class
//---------------------------------------------------------------------------------------------------------------------------------------
class CToolToggleButton : public CToolButton
{
	// instanciation section
	public :

		CToolToggleButton		(CToolBar *inOwner=NULL, const CString &inStockId=CString(),
						 const CMenuItem *inMenuItem=NULL, const CButtonListener *inListener=NULL);
		virtual ~CToolToggleButton	();

	// protected gtk widget handling
	protected :

		// gtk widget instanciation and initialization
		virtual GtkWidget *		PerformWidgetInstanciate	();
		virtual void			PerformWidgetInitialize		();

	// cbutton redefinition
	public :

		// OnClick event sender, toggle state
		virtual void			Click				();


	// tool toggle button specifics
	public :

		// toggle state; true if pressed, no event notification
		virtual void			Check				(const bool inToggle=true);
		virtual bool			IsChecked			() const;

	// cserialized redefinition
	public :

		// specific ctoolbutton serialization
		virtual void			Serialize			(CXMLElementNode *&ioXMLElementNode, const int inMode)
										 THROWABLE;

	// protected section
	protected :

		static void			OnClick				(GtkToggleToolButton *, gpointer);
	
		// dynamic metaclass association
		SECTION_DYNAMIC_METACLASS;
};

// metaclass and class tag declaration
DECLARE_DYNAMIC_METACLASS ('tlbg', CToolToggleButton, CToolButton);

//---------------------------------------------------------------------------------------------------------------------------------------
// CToolRadioButton class
//---------------------------------------------------------------------------------------------------------------------------------------
class CToolRadioButton : public CToolToggleButton
{
	// instanciation section
	public :

		CToolRadioButton		(CToolBar *inOwner=NULL, const CString &inStockId=CString(),
						 const CMenuItem *inMenuItem=NULL, const CButtonListener *inListener=NULL);
		virtual ~CToolRadioButton	();

	// protected gtk widget handling
	protected :

		// gtk widget instanciation and initialization
		virtual GtkWidget *		PerformWidgetInstanciate	();

	// ccomponent redefinition
	public :

		// owner affectation
		virtual bool			SetOwner 			(CComponent *inOwner, const SInt16 inIndex=-1);

	// tool toggle button redefinition
	public :

		// toggle state; true if pressed, no event notification
		virtual void			Check				(const bool inToggle=true);

		// dynamic metaclass association
		SECTION_DYNAMIC_METACLASS;
};

// metaclass and class tag declaration
DECLARE_DYNAMIC_METACLASS ('tlbd', CToolRadioButton, CToolToggleButton);

//---------------------------------------------------------------------------------------------------------------------------------------
// CToolSeparatorButton class
//---------------------------------------------------------------------------------------------------------------------------------------
class CToolSeparatorButton : public CToolButton
{
	// instanciation section
	public :

		CToolSeparatorButton		(CToolBar *inOwner=NULL, const CButtonListener *inListener=NULL);
		virtual ~CToolSeparatorButton	();

	// protected gtk widget handling
	protected :

		// gtk widget instanciation and initialization
		virtual GtkWidget *		PerformWidgetInstanciate	();
		virtual void			PerformWidgetInitialize		();

	// serialization redefinition
	public :

		// specific ctoolbuttonseparator serialization, jump to ccontainer abstraction
		virtual void			Serialize			(CXMLElementNode *&ioXMLElementNode, const int inMode)
										 THROWABLE;

		// dynamic metaclass association
		SECTION_DYNAMIC_METACLASS;
};

// metaclass and class tag declaration
DECLARE_DYNAMIC_METACLASS ('tlbs', CToolSeparatorButton, CToolButton);

/*****************************************************************************************************************************************
 Toolbar definition
*****************************************************************************************************************************************/

//---------------------------------------------------------------------------------------------------------------------------------------
// CToolBar class
//---------------------------------------------------------------------------------------------------------------------------------------
class CToolBar : public CLayout
{
	// instanciation section
	public :

		CToolBar			(CContainer *inOwner=NULL, const CWidgetListener *inListener=NULL);
		virtual ~CToolBar		();

	// protected gtk widget handling
	protected :

		// gtk widget instanciation and initialization
		virtual GtkWidget *		PerformWidgetInstanciate	();
		virtual void			PerformWidgetInitialize		();

	// protected container handling
	protected :

		virtual void			PerformContainerAdd		(CWidget *inChild);
		virtual void			PerformContainerRemove		(CWidget *inChild);

	// ccomponent redefinition
	public :

		// default expected children type specifies __metaclasses(CToolButton)
		virtual CMetaClasses		ChildMustBe			() const;

	// clayout redefinition
	public :

		// get the drop index of a potential dragged toolbutton at the specified toolbar relatives coordinates
		virtual SInt16			GetDropIndexAtPoint		(const TPoint &inRelativePoint, 
										 const CControl *inCandidate) const;
	// ctoolbar specifics
	public :

		// toolbar orientation
		virtual void			SetOrientation			(const GtkOrientation inGtkOrientation);
		GtkOrientation			GetOrientation			() const;

		// toolbar style
		virtual void			SetStyle			(const GtkToolbarStyle inGtkToolbarStyle);
		GtkToolbarStyle			GetStyle			() const;

		// toolbar icon size
		virtual void			SetIconSize			(const GtkIconSize inGtkIconSize);
		GtkIconSize			GetIconSize			() const;

	// cserialized redefinition
	public :

		// ctoolbar xml serialization
		virtual void			Serialize			(CXMLElementNode *&ioXMLElementNode, const int inMode);

		// dynamic metaclass association
		SECTION_DYNAMIC_METACLASS;
};

// metaclass and class tag declaration
DECLARE_DYNAMIC_METACLASS ('tlbr', CToolBar, CLayout);

#endif
