/*****************************************************************************************************************************************
 gtkol-dump.c

 Simple gtkol xml dump sample.

 This sample constructs the whole short demonstration application the first time it is executed. When the demo closes, it dumps its state
 to an xml file that is then used to load the whole application instances when it is launched again.

 Specific listeners ared defined here from generic definitions in order to handle gtkol widget associated interested events. All those
 listeners are metaclasses described because the libgenerics has to know the listener classes identifiers in order to be able to
 instanciate them from serialization dynamic load process. To do so, those metaclasses must be dynamic ones (see cmetaclass.h).

 When the first execution has closed, you can play with the dumped xml file to modify some properties and see your modifications are 
 respected when launching the demo again. if so, please send me a queen pizza double cheese, if not send me a report...
*****************************************************************************************************************************************/

#include "capplication.h"

//----------------------------------------------------------------------------------------------------------------------------------------
// CMainFormListener
//----------------------------------------------------------------------------------------------------------------------------------------
class CMainFormListener : public CFormListener
{
	// called when the form is about to be closed
	virtual void OnQueryClose (CObject *inSender, Bool &ioDoClose)
	{
		// just ask for user confirmation
		switch (CMessageDialog (static_cast <CForm *> (inSender), CString("Confirmation"), 
			CString("You are about to close this xml dump sample, you can now play with the \"./gtkol-dump.xml\" file before")
			+ " launching again.\n\nAre you sure you want to quit ?", _DBYesNo_, 
			MESSAGEDIALOG_QUESTION, true).Run ())
		{
			case DIALOG_YES : ioDoClose = true;  break;
			default	 	: ioDoClose = false; break;
		}
	}

	// called when the form is closed
	virtual void OnClose (CObject *)
	{
		// open an xml document to dump the application state into, set the xml root name to "gtkol"
		CXMLDocument XMLDoc (CString("./gtkol-dump.xml"), XML_WRITE, CString("gtkol"));

		// dump the whole application state (the CApplication instance id is always 1L because it is the first instanciated gtkol
		// component, so get the application instance and redirect it into the xml document that will launch the serialization dump
		// process on the whole gtkol components hierarchy)
		XMLDoc << CComponent::GetComponent (1L);
	}

	SECTION_DYNAMIC_METACLASS;
};
DECLARE_DYNAMIC_METACLASS ('_mfm', CMainFormListener, CFormListener);
RESOLVE_DYNAMIC_METACLASS (CMainFormListener);

//----------------------------------------------------------------------------------------------------------------------------------------
// CFileExitMenuItemListener
//----------------------------------------------------------------------------------------------------------------------------------------
class CFileExitMenuItemListener : public CMenuItemListener
{
	// called when the menu is clicked 
	virtual void OnClick (CObject *inSender)
	{
		// in our sample, the exit menu item is handled in a file menu item that is handled in a menubar that is handled in a 
		// vboxlayout that is also handled in the main form : Form->Layout->MenuBar->File->Quit; we could call 
		// ((CForm*)(((CComponent*)inSender)->GetOwner()->GetOwner()->GetOwner()->GetOwner()))->Close() to get the main form this 
		// listener is supposed to close but we can get it indirectly by specifying the expected desired owner type as follow :
		static_cast <CForm *> (static_cast <CMenuItem *> (inSender) -> GetOwner (__metaclass(CForm))) -> Close ();
	}

	SECTION_DYNAMIC_METACLASS;
};
DECLARE_DYNAMIC_METACLASS ('_cmn', CFileExitMenuItemListener, CMenuItemListener);
RESOLVE_DYNAMIC_METACLASS (CFileExitMenuItemListener);

//----------------------------------------------------------------------------------------------------------------------------------------
// CMetaClassTreeViewListener
//----------------------------------------------------------------------------------------------------------------------------------------
class CMetaClassTreeViewListener : public CTreeViewListener
{
	// called when a drag start
	virtual void OnDragStart (CObject *&ioSender, TPoint &, CPixbuf *&)
	{
		// we want the scrollview owning the treeview to be dragged rather than the treeview itself, this function is called
		// when a treeview item is draged too, so check the sender type before...
		if (ioSender -> ClassIs (__metaclass(CTreeView)))
			ioSender = static_cast <CTreeView *> (ioSender) -> GetOwner (__metaclass(CScrollView));
	}

	// called when a drag ends for the inSender instance
	virtual void OnDragStop (CObject *inSender)
	{
		// expand the tree view item owner if the dragged instance was a tree view item...
		if (inSender -> ClassIs (__metaclass(CTreeViewItem)) && 
		    static_cast <CTreeViewItem *> (inSender) -> GetOwner() -> ClassIs (__metaclass(CTreeViewItem)))
			static_cast <CTreeViewItem *> (static_cast <CTreeViewItem *> (inSender) -> GetOwner()) -> Expand (true);
			
	}

	SECTION_DYNAMIC_METACLASS;
};
DECLARE_DYNAMIC_METACLASS ('_mtw', CMetaClassTreeViewListener, CTreeViewListener);
RESOLVE_DYNAMIC_METACLASS (CMetaClassTreeViewListener);

//----------------------------------------------------------------------------------------------------------------------------------------
// CMenuItemDeleteListener
//----------------------------------------------------------------------------------------------------------------------------------------
class CMenuItemDeleteListener : public CMenuItemListener
{
        // called when the menu item is clicked
        virtual void OnClick (CObject *inSender)
        {
                // get the menu item
                CMenuItem *inMenuItem = static_cast <CMenuItem *> (inSender);

                // get the menu item popup menu owner
                CMenuPopup *inMenuPopup = static_cast <CMenuPopup *> (inMenuItem -> GetOwner (__metaclass(CMenuPopup)));

                // get the control that sended the popup event
                CControl *inPopupSender = inMenuPopup -> GetPopupSender ();

                // delete the popup sender instance...
                delete inPopupSender;
        }

	SECTION_DYNAMIC_METACLASS;
};
DECLARE_DYNAMIC_METACLASS ('_cmd', CMenuItemDeleteListener, CMenuItemListener);
RESOLVE_DYNAMIC_METACLASS (CMenuItemDeleteListener);

//----------------------------------------------------------------------------------------------------------------------------------------
// popup menu listener
//----------------------------------------------------------------------------------------------------------------------------------------
class CDeletePopupMenuListener : public CMenuPopupListener
{
	// called when the inSender popup menu is about to be poped up
	virtual void OnQueryPopup (CObject *inSender, Bool &ioDoPopup)
	{
		// retreive our popup menu instance
		CMenuPopup *inMenuPopup = static_cast <CMenuPopup *> (inSender);

		// get the control that is about to send the popup event
                CControl *inPopupSender = inMenuPopup -> GetPopupSender ();

		// get the menu popup menu item first child (the "to be titled" one)
		CMenuItem *inToBeTitled = static_cast <CMenuItem *> (*inMenuPopup -> GetChildren ()[0]);

		// set the title
		inToBeTitled -> SetCaption ("[" + metaclass_cast(inPopupSender)->ClassName + "]");
	}

	SECTION_DYNAMIC_METACLASS;
};
DECLARE_DYNAMIC_METACLASS ('_dpm', CDeletePopupMenuListener, CMenuPopupListener);
RESOLVE_DYNAMIC_METACLASS (CDeletePopupMenuListener);

//----------------------------------------------------------------------------------------------------------------------------------------
// allocate a tree view hierarchy of known metaclasses
//----------------------------------------------------------------------------------------------------------------------------------------
static void AllocateMetaClassesTreeViewItems (CComponent *inOwner, CMetaClassNode &inNode)
{
        if (inOwner == NULL || !(inOwner -> ClassIs (__metaclass(CTreeView)) || inOwner -> ClassIs (__metaclass(CTreeViewItem)))) return;
        CTreeViewItem *newItem = NULL; if (inNode.GetValue() != NULL)
        {
                if (inOwner -> ClassIs (__metaclass(CTreeView)))
                        newItem = new CTreeViewItem ((CTreeView*)inOwner);
                else
                        newItem = new CTreeViewItem ((CTreeViewItem*)inOwner);
		newItem -> SetItemFieldValues (CItemFieldValues (7,
                                        new CItemFieldValueString   (inNode.GetValue() -> ClassName),
                                        new CItemFieldValueString   (::ClassTagToString(inNode.GetValue() -> ClassTag)),
                                        new CItemFieldValueString   (CString((UInt32)inNode.GetValue() -> ClassSize)),
                                        new CItemFieldValueBoolean  (inNode.GetValue() -> MetaClassType == METACLASS_GENERIC),
                                        new CItemFieldValueBoolean  (inNode.GetValue() -> MetaClassType == METACLASS_DYNAMIC),
                                        new CItemFieldValueBoolean  (inNode.GetValue() -> MetaClassType == METACLASS_CAPSULE),
					new CItemFieldValueProgress (1.0f, CString("% done"))));
		newItem -> SetDraggable (true);
		newItem -> SetDropSite  (true);
		CItemFieldValueProgress *inProgress = static_cast <CItemFieldValueProgress *> (*(newItem -> GetItemFieldValues())[6]);
		switch (inNode.GetValue() -> ClassTag)
		{
			case __classtag(CTreeView)	       : inProgress -> SetFieldValue(0.90f); break;
			case __classtag(CMenuItem)	       : inProgress -> SetFieldValue(0.95f); break;
			case __classtag(CMenuPopup) 	       : inProgress -> SetFieldValue(0.85f); break;
			case __classtag(CScrollView)	       : inProgress -> SetFieldValue(0.25f); break;
			case __classtag(CFileChooserDialog)    : inProgress -> SetFieldValue(0.85f); break;
			case __classtag(CItemFieldValueCombo)  : inProgress -> SetFieldValue(0.10f); break;
			case __classtag(CItemFieldValuePixbuf) : inProgress -> SetFieldValue(0.90f); break;
			case __classtag(CApplication) 	       : inProgress -> SetFieldValue(0.80f); break;
			case __classtag(CGraphics)	       : inProgress -> SetFieldValue(0.25f); break;
			case __classtag(CClass)		       :
			case __classtag(CException)	       :
			case __classtag(CXMLDocument)	       :
			case __classtag(CXMLElement)	       :
			case __classtag(CXMLAttribute)	       :
			case __classtag(CFile)		       :
			case __classtag(CSerialized)	       :
			case __classtag(CObject)	       :
			case __classtag(CObjectListener)       :
			case __classtag(CMetaModule)	       : inProgress -> SetVisible (false); break;
		}
        }
        for (size_t i=inNode.GetChildren().GetLength(), j=0; i>0; i--, j++)
                AllocateMetaClassesTreeViewItems (newItem != NULL ? newItem : inOwner, **inNode.GetChildren()[j]);
}

//----------------------------------------------------------------------------------------------------------------------------------------
// demonstration entry point
//----------------------------------------------------------------------------------------------------------------------------------------
int main (int argc, char **argv)
{
	// be proper, catch any potential exception that may be generated by the serialization processes
	try
	{
		// if the xml dump file does not exist, we have to instanciate the whole demonstration components
		if (!CFile::Exists (CString("./gtkol-dump.xml")))
		{
			// declare a gtkol application instance giving it the command line arguments; those arguments are used to 
			// initialize the gtk library (there won't be any initilization if you do not specify those arguments). The
			// serialization process of a CApplication instance keeps the command line to be able to launch again with
			// the same parameters the next time it is executed
			CApplication Application (argc, argv);

			// instanciate a main form with our specific listener and set its caption
			CForm *MainForm = new CForm (&Application, new CMainFormListener());
			MainForm -> SetDropSite (true);
			MainForm -> SetCaption (CString("Gtkol Dump"));
			MainForm -> SetBounds  (TBounds (TBounds ((CApplication::GetScreenSize().w-520)/2, 
								  (CApplication::GetScreenSize().h-670)/2, 520, 670)));
			// allocate a menu popup
			CMenuPopup *MenuPopup = new CMenuPopup (MainForm, new CDeletePopupMenuListener());
			(new CMenuItem (MenuPopup, CString("[to be titled]"))) -> Disable();
			new CMenuItemSeparator (MenuPopup);
			new CMenuItem (MenuPopup, CString("_Delete"), _HKNone_, new CMenuItemDeleteListener());

			// the form is a GtkBin widget i.e. it can contain only one child widget wich is a little bit restrictive; so
			// instanciate a vertical box layout so that we can put more than one child in our form and set some properties
			CVBoxLayout *VBoxLayout = new CVBoxLayout (MainForm);
			VBoxLayout -> SetHomogeneous (false);
			VBoxLayout -> SetSpacing (1);
			VBoxLayout -> SetBoxPack (BOXPACK_START, false, false, 0);
			VBoxLayout -> SetDropSite (true);

			// instanciate the menu components : VBoxLayout->MenuBar->File->Quit; the quit menu item is associated to an
			// accelerator (ctrl+q see cmenu.h) and our specific menu item listener
			CMenuBar  *MenuBar = new CMenuBar (VBoxLayout);
			CMenuItem *File    = new CMenuItem (MenuBar, CString("_File"));
			CMenuItem *Exit    = new CMenuItem (File, CString("_Quit"), _HKQuit_, new CFileExitMenuItemListener());


			CColorButton *cb = new CColorButton (VBoxLayout);
			CFontButton  *fb = new CFontButton  (VBoxLayout);
			fb -> SetUseFont (true);
			fb -> SetUseSize (true);


			// allocate a scroll view
			VBoxLayout -> SetBoxPack (BOXPACK_START, true, true, 0);
			CScrollView *ScrollView = new CScrollView (VBoxLayout);

			// allocate a tree view with a little complex model
			CTreeView *TreeView = new CTreeView (ScrollView, _IFVString_ + _IFVString_ + _IFVString_ + _IFVBoolean_ +
									 _IFVBoolean_ + _IFVBoolean_ + _IFVProgress_, 
							     new CMetaClassTreeViewListener());
		        TreeView -> SetDraggable (true);
		        TreeView -> SetDropSite  (true);
/*
			TreeView -> SetHeadersVisible (true);	
			TreeView -> SetRulesHint (true);
			CStrings titles;
			titles += CString("Class");
			titles += CString("Signature");
			titles += CString("Size");
			titles += CString("Generic");
			titles += CString("Dynamic");
			titles += CString("Capsule");
			titles += CString("Done");
			TreeView -> SetColumnsTitle (titles);
*/
		        // get the metaclasses tree
		        CMetaClassNode inMetaClasses (CMetaClass::GetMetaClassTree());

		        // set the tree items
		        ::AllocateMetaClassesTreeViewItems (TreeView, inMetaClasses);

			// expand the whole tree
			static_cast <CTreeViewItem *> (*TreeView -> GetChildren(__metaclass(CTreeViewItem))[0]) -> Expand (true);

			// show our job (the form is the only gtkol widget that is not shown by default)
			MainForm -> Show ();

			// launch the application event queue dispatching
			Application.Run ();
		}
		// the xml dump file exists, so load the whole application components hierarchy and states from the given template
		else
		{
			// open the dump file to read from and check the root node is a "gtkol" xml element; then get the xml root node
			CXMLDocument XMLDoc (CString("./gtkol-dump.xml"), XML_READ, CString("gtkol"));
			CXMLElementNode *inXMLRoot = XMLDoc.GetRootElement ();

			// instanciate the whole gtkol given hierarchy from the xml root element node
			CSerialized *inSerialized = CSerialized::Instanciate (inXMLRoot);

			// check we got a CApplication instance as expected and launch it !
			if (inSerialized -> ClassIs (__metaclass(CApplication))) static_cast <CApplication *> (inSerialized) -> Run ();

			// free the pointers
			delete inSerialized;
			delete inXMLRoot;
		}
	}
	// an exception occurred, just print it and return
	catch (CException *e)
	{
		fprintf (stderr, "GtkOL exception : %s\n", e->GetMessage().Get());
		return -1;
	}

	// ok
	return 0;
}

