/*******************************************************************************************************************************************
 ccomponent.c
*******************************************************************************************************************************************/

#include "ccomponent.h"

//------------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//------------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_GENERIC_METACLASS (CComponent);

//------------------------------------------------------------------------------------------------------------------------------------------
// static attributes resolution
//------------------------------------------------------------------------------------------------------------------------------------------
CComponentNode	 CComponent::m_AbsoluteNode;
UInt32		 CComponent::m_IdInc = 0L;
TBuffer <UInt32> CComponent::m_IdLost;

//------------------------------------------------------------------------------------------------------------------------------------------
// mutex static resolution and initialization
//------------------------------------------------------------------------------------------------------------------------------------------
GStaticRecMutex CComponent::m_GStaticRecMutex = G_STATIC_REC_MUTEX_INIT;

//------------------------------------------------------------------------------------------------------------------------------------------
// mutex lock
//------------------------------------------------------------------------------------------------------------------------------------------
void CComponent::Lock () 
{
	if (g_thread_supported()) ::g_static_rec_mutex_lock (&CComponent::m_GStaticRecMutex);
}

//------------------------------------------------------------------------------------------------------------------------------------------
// mutex try lock
//------------------------------------------------------------------------------------------------------------------------------------------
Bool CComponent::TryLock () 
{
	return g_thread_supported() ? ::g_static_rec_mutex_trylock (&CComponent::m_GStaticRecMutex) : true;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// mutex unlock
//------------------------------------------------------------------------------------------------------------------------------------------
void CComponent::Unlock () 
{
	if (g_thread_supported()) ::g_static_rec_mutex_unlock (&CComponent::m_GStaticRecMutex);
}

//------------------------------------------------------------------------------------------------------------------------------------------
// mutex lock full
//------------------------------------------------------------------------------------------------------------------------------------------
void CComponent::LockFull (const UInt32 inDepth) 
{
	if (g_thread_supported()) ::g_static_rec_mutex_lock_full (&CComponent::m_GStaticRecMutex, inDepth);
}

//------------------------------------------------------------------------------------------------------------------------------------------
// mutex unlock full
//------------------------------------------------------------------------------------------------------------------------------------------
UInt32 CComponent::UnlockFull ()
{
	return g_thread_supported() ? ::g_static_rec_mutex_unlock_full (&CComponent::m_GStaticRecMutex) : 0;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// constructor
//------------------------------------------------------------------------------------------------------------------------------------------
CComponent::CComponent	    (CComponent *inOwner, const CObjectListener *inListener)
	   :CObject	    (inListener),
	    m_ComponentNode (NULL),
	    m_Id	    (0L)
{
	// lock gtkol spine
	CComponent::Lock();

	// is there any id lost available to be used again ?
	if (CComponent::m_IdLost.GetLength() > 0)
	{
		const_cast <UInt32 &> (m_Id) = *m_IdLost[0];
		CComponent::m_IdLost -= m_Id;
	}
	// give this instance a new one
	else
		const_cast <UInt32 &> (m_Id) = ++CComponent::m_IdInc;

	// is this component's owner specified ? Is the specified owner's node allocated ?
	if (inOwner != NULL && inOwner -> m_ComponentNode != NULL)
	
		// create a child node pointing on this instance (true parameter specifies that when the corresponding node is deleted, be
		// sure its value i.e. this component instance is deleted too)
		m_ComponentNode = new CComponentNode (inOwner -> m_ComponentNode, this, true);

	// otherwise
	else

		// allocate an empty node to enable part components tree allocation before linking them together
		m_ComponentNode = new CComponentNode (NULL, this, true);

	// unlock gtkol spine
	CComponent::Unlock();
}

//------------------------------------------------------------------------------------------------------------------------------------------
// destructor
//------------------------------------------------------------------------------------------------------------------------------------------
CComponent::~CComponent ()
{
	// lock gtkol spine
	CComponent::Lock();

	// is the associated component node allocated ?
	if (m_ComponentNode != NULL)
	{
		// remove its value i.e. this instance as we do not want the node deletion to call this destructor again or it would result
		// in a beautifull segmentation fault and a marvelous core dump
		m_ComponentNode -> RemoveValue ();
	
		// delete the node (indirect calls to all subcomponents destructors)
		if (m_ComponentNode -> GetParent() == NULL && CComponent::m_ComponentNode != &CComponent::m_AbsoluteNode)

			// there might be a broken spine somewhere, so delete the node instance itself
			delete m_ComponentNode;

		// this instance is not a root of a broken spine or is the absolute node !
		else

			// perform a recursive tree deletion
			m_ComponentNode -> Delete ();
	}

	// add this component id to the lost ids list
	CComponent::m_IdLost += m_Id;

	// sort the list so the min id is used for the next component instanciation
	CComponent::m_IdLost.Sort();

	// unlock gtkol spine
	CComponent::Unlock();
}

//------------------------------------------------------------------------------------------------------------------------------------------
// component id access
//------------------------------------------------------------------------------------------------------------------------------------------
UInt32 CComponent::GetId () const
{
	return m_Id;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// component of specified id access
//------------------------------------------------------------------------------------------------------------------------------------------
CComponent * CComponent::GetComponent (const UInt32 inId)
{
	// get the whole list of components
	CComponents inComponents (CComponent::GetComponents ());

	// go through the list and search for the specified id
	for (size_t i=inComponents.GetLength(), j=0; i>0; i--, j++)
		if ((*inComponents[j])->m_Id == inId)
			return *inComponents[j];

	// id not found
	return NULL;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// components list access
//------------------------------------------------------------------------------------------------------------------------------------------
CComponents CComponent::GetComponents (const CMetaClass *inRequested, const CMetaClass *inRejected)
{
	// out components list
	CComponents outComponents;

	// get the whole components list
	CComponentNodes inComponentNodes (CComponent::m_AbsoluteNode.GetTree());

	// is there a rejected type specified ?
	if (inRejected)
	{
		// go through the list
		for (size_t i=inComponentNodes.GetLength(), j=0; i>0; i--, j++)
		{	
			CComponentNode *inComponentNode (*inComponentNodes[j]);
			if ((inComponentNode -> GetValue() != NULL && inComponentNode -> GetValue() -> ClassIs (inRequested)) &&
			   !(inComponentNode -> GetValue() != NULL && inComponentNode -> GetValue() -> ClassIs (inRejected)))
				outComponents += inComponentNode -> GetValue();
		}
	}
	// no rejected type specified
	else
	{
		// go through the list
		for (size_t i=inComponentNodes.GetLength(), j=0; i>0; i--, j++)
		{	
			CComponentNode *inComponentNode (*inComponentNodes[j]);
			if (inComponentNode -> GetValue() != NULL && inComponentNode -> GetValue() -> ClassIs (inRequested))
				outComponents += inComponentNode -> GetValue();
		}
	}

	// ok
	return outComponents;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// component children access
//------------------------------------------------------------------------------------------------------------------------------------------
CComponents CComponent::GetChildren (const CMetaClass *inRequested, const CMetaClass *inRejected) const
{
	// out components list
	CComponents outComponents;

	// component node check
	if (m_ComponentNode == NULL) return outComponents;

	// children list read in
	CComponentNodes inComponentChildren (m_ComponentNode -> GetChildren ());

	// is there a rejected type specified ?
	if (inRejected)
	{
		// go through the list
		for (size_t i=inComponentChildren.GetLength(), j=0; i>0; i--, j++)
		{
			CComponentNode *inComponentChild (*inComponentChildren[j]);
			if ((inComponentChild -> GetValue() != NULL && inComponentChild -> GetValue() -> ClassIs (inRequested)) &&
			   !(inComponentChild -> GetValue() != NULL && inComponentChild -> GetValue() -> ClassIs (inRejected)))
				outComponents += inComponentChild -> GetValue();
		}
	}
	// no rejected type specified
	else
	{
		// go through the list
		for (size_t i=inComponentChildren.GetLength(), j=0; i>0; i--, j++)
		{
			CComponentNode *inComponentChild (*inComponentChildren[j]);
			if (inComponentChild -> GetValue() != NULL && inComponentChild -> GetValue() -> ClassIs (inRequested))
				outComponents += inComponentChild -> GetValue();
		}
	}

	// ok
	return outComponents;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// component subcomponents access
//------------------------------------------------------------------------------------------------------------------------------------------
CComponents CComponent::GetSubComponents (const CMetaClass *inRequested, const CMetaClass *inRejected) const
{
	// out components list
	CComponents outComponents;

	// component node check
	if (m_ComponentNode == NULL) return outComponents;

	// subcomponents list read in
	CComponentNodes inSubComponentNodes (m_ComponentNode -> GetTree ());

	// is there a rejected type specified ?
	if (inRejected)
	{
		// go through the list
		for (size_t i=inSubComponentNodes.GetLength(), j=0; i>0; i--, j++)
		{
			CComponentNode *inSubComponentNode (*inSubComponentNodes[j]);
			if ((inSubComponentNode -> GetValue() != NULL && inSubComponentNode -> GetValue() -> ClassIs (inRequested)) &&
			   !(inSubComponentNode -> GetValue() != NULL && inSubComponentNode -> GetValue() -> ClassIs (inRejected)))
				outComponents += inSubComponentNode -> GetValue();
		}
	}
	// no rejected type specified
	else
	{
		// go through the list
		for (size_t i=inSubComponentNodes.GetLength(), j=0; i>0; i--, j++)
		{
			CComponentNode *inSubComponentNode (*inSubComponentNodes[j]);
			if (inSubComponentNode -> GetValue() != NULL && inSubComponentNode -> GetValue() -> ClassIs (inRequested))
				outComponents += inSubComponentNode -> GetValue();
		}
	}

	// ok
	return outComponents;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// component sibling access
//------------------------------------------------------------------------------------------------------------------------------------------
CComponents CComponent::GetSiblings (const CMetaClass *inRequested, const CMetaClass *inRejected) const
{
	// this component direct owner
	CComponent *inOwner (GetOwner());
	
	// do this instance have a owner ? if not, just give our pointer in the output sibling list
	if (inOwner == NULL) return CComponents (1, const_cast <CComponent *> (this));

	// get this component's entire level
	return inOwner -> GetChildren (inRequested, inRejected);
}

//------------------------------------------------------------------------------------------------------------------------------------------
// subcomponent hold on test
//------------------------------------------------------------------------------------------------------------------------------------------
Bool CComponent::HasSubComponent (const CComponent *inCandidate) const
{
	// get the component's subcomponents
	CComponents inSubComponents (GetSubComponents());

	// go through the list to find the specified candidate if any
	for (size_t i=inSubComponents.GetLength(), j=0; i>0; i--, j++)
		if ((*inSubComponents[j]) == const_cast <CComponent *> (inCandidate))
			return true;

	// not found
	return false;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// sibling test
//------------------------------------------------------------------------------------------------------------------------------------------
Bool CComponent::HasSibling (const CComponent *inComponent) const
{
	// get this instance siblings
	CComponents inSiblings (GetSiblings());

	// go through the given list
	for (size_t i=inSiblings.GetLength(); i>0; i--)
		if ((*inSiblings[i-1]) == const_cast <CComponent *> (inComponent))
			return true;

	// not found
	return false;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// in owner index calculation
//------------------------------------------------------------------------------------------------------------------------------------------
SInt16 CComponent::GetInOwnerIndex (const CComponent *inOwner, const CComponent *inChild, const CMetaClass *inRequested, 
				    const CMetaClass *inRejected)
{
	// pointers check in
	if (inOwner == NULL || inChild == NULL) return -1;

	// get the specified owner children components
	CComponents inChildren (inOwner -> GetChildren (inRequested, inRejected));

	// go through the children list
	for (size_t i=inChildren.GetLength(), j=0; i>0; i--, j++) 
		if ((*inChildren[j]) == const_cast <CComponent *> (inChild)) return j;

	// not found
	return -1;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// in owner index calculation
//------------------------------------------------------------------------------------------------------------------------------------------
SInt16 CComponent::GetInOwnerOccurence (const CComponent *inOwner, const CComponent *inChild, const CMetaClass *inRequested, 
					const CMetaClass *inRejected)
{
	// pointers check in
	if (inOwner == NULL || inChild == NULL) return -1;

	// get the specified owner subcomponents
	CComponents inSubComponents (inOwner -> GetSubComponents (inRequested, inRejected) - const_cast <CComponent *> (inOwner));

	// go through the subcomponents list
	for (size_t i=inSubComponents.GetLength(), j=0; i>0; i--, j++) 
		if ((*inSubComponents[j]) == const_cast <CComponent *> (inChild)) return j;

	// not found
	return -1;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// check if the specified component could be assigned as an owner of this instance
//------------------------------------------------------------------------------------------------------------------------------------------
Bool CComponent::CheckSetOwner (const CComponent *inCandidate) const
{
	// potential owner removal, no particular check
	if (inCandidate == NULL) return true;

	// basic pointer check
	if (inCandidate == this) return false;

	// do not inverse the rules...
	if (HasSubComponent (inCandidate)) return false;

	// get the candidate expected children types
	CMetaClasses inMetaClasses (inCandidate -> ChildMustBe());

	// go through the expected children types and check this instance is one of the expected types
	Bool doPurchase=false; 
	for (size_t i=inMetaClasses.GetLength(); i>0 && !doPurchase; i--) 
		if (ClassIs (*inMetaClasses[i-1])) doPurchase = true;
	if (!doPurchase) return false;

	// get the component instance expected owner types
	inMetaClasses = OwnerMustBe();

	// go through the expected owner types and check the candidate is one of the expected types
	for (size_t i=inMetaClasses.GetLength(); i>0 && !doPurchase; i--) 
		if (inCandidate -> ClassIs (*inMetaClasses[i-1])) doPurchase = true;

	// ok ?
	return doPurchase;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// component's owner affectation
//------------------------------------------------------------------------------------------------------------------------------------------
Bool CComponent::SetOwner (CComponent *inOwner, const SInt16 inIndex)
{
	// owner affectation check
	if (!CheckSetOwner (inOwner)) return false;

	// owner component associated node pointer check
	if (inOwner != NULL && inOwner -> m_ComponentNode == NULL) return false;

	// the specified index may be modified to accommodate the components hierarchy...
	SInt16 theIndex = inIndex;

	// if an insert request is specified i.e. if the inIndex parameter is not a negative value
	if (theIndex > 0)
	{
		// get the actual in owner children if any owner and any child...
		CComponents inChildren (inOwner != NULL ? inOwner -> GetChildren() : CComponents());

		// if out of range, transform as an append request i.e. a negative value specification
		if (theIndex >= inChildren.GetLength()) theIndex = -1;
	}

	// we have to change the node instance owner if the node has already been instanciated
	if (m_ComponentNode != NULL) 
	{
		// remove the current parent node
		m_ComponentNode -> RemoveParent ();

		// append requested for a negative value, insert for a positive one (above checked index)
		if (inOwner != NULL)
		{
			if (theIndex < 0) 
				m_ComponentNode -> SetParent (inOwner -> m_ComponentNode);
			else 
				inOwner -> m_ComponentNode -> InsertChild (m_ComponentNode, theIndex);
		}
	}
	// the instance has no owner yet, so allocate a new node on the specified owner node (should never happen since the ccomponent
	// allocates an empty tree node if so)
	else 
	{
		// append requested
		if (theIndex < 0) 
			m_ComponentNode = new CComponentNode (inOwner != NULL ? inOwner -> m_ComponentNode : NULL, this, true);

		// insert requested
		else
		{
			// allocate a new node with no owner
			m_ComponentNode = new CComponentNode (NULL, this, true);

			// insert it into the hierarchy at the specified index
			if (inOwner != NULL) inOwner -> m_ComponentNode -> InsertChild (m_ComponentNode, theIndex);
		}
	}

	// ok
	return true;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// component's direct or indirect owner access
//------------------------------------------------------------------------------------------------------------------------------------------
CComponent * CComponent::GetOwner (const CMetaClass *inRequested) const
{
	// current node search on
	CComponentNode *inNode = m_ComponentNode;

	// go through the owners
	while (inNode != NULL)
	{
		// if it not an absolute node
		if (inNode -> GetParent() != NULL)

			// be paranoïd, check the parent node has got an associated component instance, should always be true, anyway...
			if (inNode -> GetParent() -> GetValue() != NULL)

				// check the requested instance type
				if (inNode -> GetParent() -> GetValue() -> ClassIs (inRequested))

					// ok
					return inNode -> GetParent() -> GetValue();

		// next owner
		inNode = inNode -> GetParent();
	}

	// not found
	return NULL;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// component expected owner types
//------------------------------------------------------------------------------------------------------------------------------------------
CMetaClasses CComponent::OwnerMustBe () const
{
	return __metaclasses(CComponent);
}

//------------------------------------------------------------------------------------------------------------------------------------------
// component expected children types
//------------------------------------------------------------------------------------------------------------------------------------------
CMetaClasses CComponent::ChildMustBe () const
{
	return __metaclasses(CComponent);
}

//------------------------------------------------------------------------------------------------------------------------------------------
// component xml serialization
//------------------------------------------------------------------------------------------------------------------------------------------
void CComponent::Serialize (CXMLElementNode *&ioXMLElementNode, const int inMode) THROWABLE
{
	// generic serialization call
	CObject::Serialize (ioXMLElementNode, inMode);

	// serialization request analyse
	switch (inMode)
	{
		// component serialization dump
		case XML_WRITE :
		{
			// instanciate a new xml element
			CXMLElement *XMLElement = new CXMLElement (ioXMLElementNode, XML_COMPONENT_ELEMENT);

			// get the newly allocated xml node for this xml element and modify the in/out pointer so that the serialization 
			// of the overwritten definitions will continue under this node
			ioXMLElementNode = XMLElement -> GetXMLElementNode ();

			// get the component's children
			CComponents inChildren (GetChildren());

			// are there children under this component ?
			if (inChildren.GetLength())
			{
				// allocate the associated xml element
				XMLElement = new CXMLElement (ioXMLElementNode, XML_COMPONENT_CHILDREN_ELEMENT);

				// get the newly associated node
				CXMLElementNode *ioXMLNode = XMLElement -> GetXMLElementNode ();

				// serialize each of the children components
				for (size_t i=inChildren.GetLength(), j=0; i>0; i--, j++)
				{
					// each child registers its dump under the same xml local item
					CXMLElementNode *outXMLElementNode = ioXMLNode;

					// serialize the child component
					(*inChildren[j]) -> Serialize (outXMLElementNode, XML_WRITE);
				}
			}
		}	
		break;

		// component serialization load
		case XML_READ :
		{
			// get the current xml node interested child
			CXMLElementNode *inXMLNode = ::xml_node_get_child (ioXMLElementNode, XML_COMPONENT_ELEMENT);

			// check we got an expected element node
			if (inXMLNode == NULL)
				throw new CException (CString("CComponent::Serialize, specified xml node is not a \"") + 
							      XML_COMPONENT_ELEMENT + CString("\" element one."), __exception(XMLPARSE));

			// modify the input/output xml node so that serialization process continues under this node
			ioXMLElementNode = inXMLNode;

			// read in the component's owner-id attribute if any
			UInt32 inOwnerId = ::xml_node_get_attribute (inXMLNode, XML_COMPONENT_ATTR_OWNER_ID).GetValue().ToULong();
			
			// set this instance component's owner from the specified id
			if (inOwnerId > 0L) 
				SetOwner (CComponent::GetComponent (inOwnerId));

			// the component owner id is not specified, so it must be the absolute component node instance (Id 1 always), if
			// not so, default this owner instance to the absolute one
			else if (GetId() > 1L)
				SetOwner (CComponent::GetComponent (1L));

			// get the current xml node potential children element
			inXMLNode = ::xml_node_get_child (inXMLNode, XML_COMPONENT_CHILDREN_ELEMENT);

			// are there children specified for the current component ?
			if (inXMLNode != NULL)
			{
				// get the "children" element children nodes
				CXMLElementNodes inXMLNodes (::xml_node_get_children (inXMLNode));

				// foreach child of the "children" element, set the owner id xml attribute so that the recursive call
				// will know which point the new instance should be attached to
				for (size_t i=inXMLNodes.GetLength(), j=0; i>0; i--, j++)
				{
					// find the component xml node of the specified child
					inXMLNode = ::xml_node_search (*inXMLNodes[j], XML_COMPONENT_ELEMENT);

					// add this instance id as owner id attribute of the xml child node
					if (inXMLNode != NULL)
						inXMLNode -> GetValue() -> AddAttribute (XML_COMPONENT_ATTR_OWNER_ID, CString(GetId()));
				}

				// foreach child of the "children" element, launch the global serialization process
				for (size_t i=inXMLNodes.GetLength(), j=0; i>0; i--, j++)

					// launch the instanciation process
					CSerialized::Instanciate (*inXMLNodes[j]) THROWABLE;
			}
		}
		break;
	}
}

//------------------------------------------------------------------------------------------------------------------------------------------
// archive serialization not supported
//------------------------------------------------------------------------------------------------------------------------------------------
void CComponent::Serialize (CArchive &) THROWABLE
{
	// archive serialization not supported
	throw new CException (CString("CComponent::Serialize; archive mode serialization is not implemented in gtkol, consider using ") + 
			      "xml serialization instead.");
}




