/******************************************************************************************************************************************
 cspinbutton.h
 
 CClass					CClass
 |--CSerialized				|--CSerialized
    |--CMetaModule			   |--CMetaModule
       |--CObject			      |--CObjectListener
          |--CComponent			         |
	     |--CControl			 |
	        |--CWidget			 |--CWidgetListener
		   |--CEntry			    |--CEntryListener
		      |--CSpinButton		       |--CSpinButtonListener

 Convenience way of retrieve an integer or floating-point number from the user. The value is initially set to the minimum value 0 and a
 page increment of 10 * step is the default with step set to 0.1. The precision of the spin button is equivalent to the precision of step.
 Note that the way in which the precision is derived works best if step is a power of ten.
******************************************************************************************************************************************/

#ifndef __CSPINBUTTON_H__
#define __CSPINBUTTON_H__

#include "centry.h"

//-----------------------------------------------------------------------------------------------------------------------------------------
// cspinbutton xml serialization constants
//-----------------------------------------------------------------------------------------------------------------------------------------
// <cspinbutton value="Float64" min="Float64" max="Float64" digits="UInt8" step="Float64" page="Float64" update="always|if-valid" 
//		numeric="true|false" wrap="true|false" snap="true|false">
//    ...
// </cspinbutton>
//-----------------------------------------------------------------------------------------------------------------------------------------
static CString XML_SPINBUTTON_ELEMENT 		("cspinbutton");
static CString XML_SPINBUTTON_ATTR_VALUE	("value");
static CString XML_SPINBUTTON_ATTR_MIN		("min");
static CString XML_SPINBUTTON_ATTR_MAX		("max");
static CString XML_SPINBUTTON_ATTR_DIGITS	("digits");
static CString XML_SPINBUTTON_ATTR_STEP		("step");
static CString XML_SPINBUTTON_ATTR_PAGE		("page");
static CString XML_SPINBUTTON_ATTR_UPDATE	("update");
static CString XML_SPINBUTTON_ATTR_NUMERIC	("numeric");
static CString XML_SPINBUTTON_ATTR_WRAP		("wrap");
static CString XML_SPINBUTTON_ATTR_SNAP		("snap");

//------------------------------------------------------------------------------------------------------------------------------------------
// CSpinButtonListener class
//------------------------------------------------------------------------------------------------------------------------------------------
class CSpinButtonListener : public CEntryListener
{
	// instanciation section
	public :

		CSpinButtonListener		();
		virtual ~CSpinButtonListener	() =0;

	// specific api
	public :

		// called when the inSender spin button value has been updated
		virtual void			OnUpdate			(CObject *inSender)			{ }

		// metaclass association
		SECTION_GENERIC_METACLASS;
};

// metaclass and class tag declaration
DECLARE_GENERIC_METACLASS ('_sib', CSpinButtonListener, CEntryListener);

//------------------------------------------------------------------------------------------------------------------------------------------
// CSpinButton class
//------------------------------------------------------------------------------------------------------------------------------------------
class CSpinButton : public CEntry
{
	// instanciation section
	public :

		CSpinButton			(CContainer *inOwner=NULL, const CEntryListener *inListener=NULL);
		virtual ~CSpinButton		();

	// protected requests handling
	protected :

		// gtk widget instanciation and initialization
		virtual GtkWidget *		PerformWidgetInstanciate	();
		virtual void			PerformWidgetInitialize		();
	
	// protected cobject overloading
	protected :

		// expected listener type : CSpinButtonListener
		virtual const CMetaClass *	ListenerMustBe			() const;

	// spin button specifics
	public :

		// minimum and maximum allowable values for the spin button
		virtual void			SetRange			(const double inMin, const double inMax);
		double				GetMin				() const;
		double				GetMax				() const;

		// the precision to be displayed by the spin button, up to 20 digit precision is allowed
		virtual void			SetDigits			(const UInt8 inDigits);
		UInt8				GetDigits			() const;

		// step and page increment for the spin button, button 1 press, button 2 press
		virtual void			SetIncrements			(const double inStep, const double inPage);
		double				GetStep				() const;
		double				GetPage				() const;

		// determines whether the spin button is always updated or only when a valid value is set
		virtual void			SetUpdatePolicy			(const GtkSpinButtonUpdatePolicy);
		GtkSpinButtonUpdatePolicy	GetUpdatePolicy			() const;

		// determines if non-numeric text can be typed into the spin button
		virtual void			SetNumeric			(const bool inNumericOnly);
		bool			 	GetNumeric			() const;

		// determines if a spin button value wraps around to the opposite limit when the upper or lower limit of the range 
		// is exceeded
		virtual void			SetWrap				(const bool inWrap);
		bool				GetWrap				() const;

		// whether values are corrected to the nearest step increment when a spin button is activated after providing an invalid
		// value
		virtual void			SetSnapToTicks			(const bool inSnap);
		bool				GetSnapToTicks			() const;

		// value affectation and access
		virtual void			SetValue			(const double inValue);
		double				GetValue			() const;

		// increment or decrement a spin button's value in a specified direction by a specified amount
		virtual void			Spin				(const GtkSpinType inGtkSpinType, const double inInc);

		// force an update of the spin button
		virtual void			Update				();

	// CSerialized overloading
	public :

		// CEntry serialization definition
		virtual void			Serialize			(CXMLElementNode *&ioXMLElementNode, const int inMode)
										 THROWABLE;

	// protected section
	protected :

		static void			OnUpdate			(GtkSpinButton *, gpointer);

		// metaclass association
		SECTION_DYNAMIC_METACLASS;
};

// metaclass and class tag declaration
DECLARE_DYNAMIC_METACLASS ('spin', CSpinButton, CEntry);

#endif

