/* libguestfs - the guestfsd daemon
 * Copyright (C) 2009 Red Hat Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <config.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <inttypes.h>
#include <unistd.h>

#include "daemon.h"
#include "actions.h"

GUESTFSD_EXT_CMD(str_blockdev, blockdev);

/* These functions are all about using the blockdev command, so
 * we centralize it in one call.
 */
static int64_t
call_blockdev (const char *device, const char *switc, int extraarg, int prints)
{
  int r;
  int64_t rv;
  CLEANUP_FREE char *out = NULL, *err = NULL;
  const char *argv[] = {
    str_blockdev,
    switc,
    NULL,
    NULL,
    NULL
  };
  char buf[64];

  /* When you call close on any block device, udev kicks off a rule
   * which runs blkid to reexamine the device.  We need to wait for
   * this rule to finish running (from a previous operation) since it
   * holds the device open and can cause other operations to fail,
   * notably BLKRRPART.
   *
   * This is particularly a problem where we have just written to a
   * device (eg. zeroing it) and immediately call blockdev --rereadpt.
   *
   * Therefore, wait for udev to finish all outstanding events before
   * performing any blockdev command.
   */
  udev_settle ();

  if (extraarg > 0) {
    snprintf (buf, sizeof buf, "%d", extraarg);
    argv[2] = buf;
    argv[3] = device;
  } else
    argv[2] = device;

  r = commandv (&out, &err, argv);

  if (r == -1) {
    reply_with_error ("%s: %s", argv[0], err);
    return -1;
  }

  rv = 0;

  if (prints) {
    if (sscanf (out, "%" SCNi64, &rv) != 1) {
      reply_with_error ("%s: expected output, but got nothing", argv[0]);
      return -1;
    }
  }

  return rv;
}

int
do_blockdev_setro (const char *device)
{
  return (int) call_blockdev (device, "--setro", 0, 0);
}

int
do_blockdev_setrw (const char *device)
{
  return (int) call_blockdev (device, "--setrw", 0, 0);
}

int
do_blockdev_getro (const char *device)
{
  return (int) call_blockdev (device, "--getro", 0, 1);
}

int
do_blockdev_getss (const char *device)
{
  return (int) call_blockdev (device, "--getss", 0, 1);
}

int
do_blockdev_getbsz (const char *device)
{
  return (int) call_blockdev (device, "--getbsz", 0, 1);
}

int
do_blockdev_setbsz (const char *device, int blocksize)
{
  /* Do nothing.  See https://bugzilla.redhat.com/show_bug.cgi?id=1002825 */
  return 0;
}

int64_t
do_blockdev_getsz (const char *device)
{
  return call_blockdev (device, "--getsz", 0, 1);
}

int64_t
do_blockdev_getsize64 (const char *device)
{
  return call_blockdev (device, "--getsize64", 0, 1);
}

int
do_blockdev_flushbufs (const char *device)
{
  return call_blockdev (device, "--flushbufs", 0, 0);
}

int
do_blockdev_rereadpt (const char *device)
{
  return call_blockdev (device, "--rereadpt", 0, 0);
}
