// -*- C++ -*-
//
// The Hoard Multiprocessor Memory Allocator
// www.hoard.org
//
// Author: Emery Berger, http://www.cs.umass.edu/~emery
//
// Copyright (c) 1998-2003, The University of Texas at Austin.
//
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Library General Public License as
// published by the Free Software Foundation, http://www.fsf.org.
//
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
//////////////////////////////////////////////////////////////////////////////

/**
* @class superblock
* @brief The superblock class controls a number of blocks (allocatable units of memory).
* @author Emery Berger <http://www.cs.umass.edu/~emery>
*/

#ifndef _SUPERBLOCK_H_
#define _SUPERBLOCK_H_

#include "config.h"

#include <new.h>
//#include <stdio.h>

#include "arch-specific.h"
#include "block.h"

class hoardHeap; // forward declaration
class processHeap; // forward declaration

class superblock {

public:

  /// Construct a superblock for a given size class and set the heap owner.
  superblock (int numblocks,
	      int sizeclass,
	      hoardHeap * owner);

  ~superblock (void)
  {
	  hoardLockDestroy (_upLock);
  }

  /// Make (allocate or re-use) a superblock for a given size class.
  static superblock * makeSuperblock (int sizeclass, processHeap * pHeap);

  /// Find out who allocated this superblock.
  inline hoardHeap * getOwner (void);

  /// Set the superblock's owner.
  inline void setOwner (hoardHeap * o);

  /// Get a block from the superblock.
  inline block * getBlock (void);

  /// Put a block back in the superblock.
  inline void putBlock (block * b);

  /// Put *multiple* blocks into the superblock.
  // @note These blocks should all be linked in a list.
  //   hd = the head of the list.
  //   tl = the tail of the list.
  //   nblks = the number of blocks in the list.
  void putMultipleBlocks (block * hd, block * tl, int nblks);

  /// @return The number of available blocks.
  inline int getNumAvailable (void);

  /// @return The total number of blocks.
  inline int getNumBlocks (void) const;

  /// @return The size class for blocks in this superblock.
  inline int getBlockSizeClass (void) const;

  /// Insert this superblock before the next one.
  inline void insertBefore (superblock * nextSb);

  /// @return The next pointer (to the next superblock in the list).
  inline superblock * const getNext (void);

  /// @return The prev pointer (to the previous superblock in the list).
  inline superblock * const getPrev (void);

  /// @return The 'fullness' of this superblock.
  inline int getFullness (void);
  
#if HEAP_FRAG_STATS
  // Return the amount of waste in every allocated block.
  int getMaxInternalFragmentation (void);
#endif

  /// Remove this superblock from its linked list.
  inline void remove (void);

  /// @return True iff this superblock is valid.
  /// @note   Validity means it has the right magic number.
  inline int isValid (void) const;

  inline void upLock (void) {
    hoardLock (_upLock);
  }

  inline void upUnlock (void) {
    hoardUnlock (_upLock);
  }

  /// Compute the 'fullness' of this superblock.
  inline void computeFullness (void);

private:


  // Disable copying and assignment.

  superblock (const superblock&);
  const superblock& operator= (const superblock&);

  /// Used for sanity checking.
  enum { SUPERBLOCK_MAGIC = 0xCAFEBABE };

#if HEAP_DEBUG
  unsigned long _magic;
#endif

	/// The size class of blocks in the superblock.
  const int 	_sizeClass;

  	/// The number of blocks in the superblock.
  const int 	_numBlocks;

  	/// The number of blocks available.
  int		_numAvailable;

  /// How full is this superblock? (which SUPERBLOCK_FULLNESS group is it in)
  int		_fullness;

	/// A pointer to the first free block.
  block *	_freeList;
 
  /// The heap who owns this superblock.
  hoardHeap * 	_owner;

  /// The next superblock in the list.
  superblock * 	_next;

  /// The previous superblock in the list.
  superblock * 	_prev;

  bool dirtyFullness;

  /// Lock this when moving a superblock to the global (process) heap.
  hoardLockType _upLock;

  /// We insert a cache pad here to prevent false sharing with the
  /// first block (which immediately follows the superblock).

  double _pad[CACHE_LINE / sizeof(double)];
};


hoardHeap * superblock::getOwner (void)
{
  assert (isValid());
  hoardHeap * o = _owner;
  return o;
}


void superblock::setOwner (hoardHeap * o) 
{
  assert (isValid());
  _owner = o;
}


block * superblock::getBlock (void)
{
  assert (isValid());
  // Pop off a block from this superblock's freelist,
  // if there is one available.
  if (_freeList == NULL) {
    // The freelist is empty.
    assert (getNumAvailable() == 0);
    return NULL;
  }
  assert (getNumAvailable() > 0);
  block * b = _freeList;
  _freeList = _freeList->getNext();
  _numAvailable--;

  b->setNext(NULL);

  dirtyFullness = true;
  //  computeFullness();

  return b;
}


void superblock::putBlock (block * b)
{
  assert (isValid());
  // Push a block onto the superblock's freelist.
  assert (b->isValid());
  assert (b->getSuperblock() == this);
  assert (getNumAvailable() < getNumBlocks());
  b->setNext (_freeList);
  _freeList = b;
  _numAvailable++;

  dirtyFullness = true;
  //  computeFullness();
}

inline void superblock::putMultipleBlocks (block * hd, block * tl, int nblks)
{
  assert (isValid());
  // Push these blocks onto the superblock's freelist.
  assert (hd->isValid());
  assert (tl->isValid());
  assert (hd->getSuperblock() == this);
  assert (tl->getSuperblock() == this);
  assert (getNumAvailable() < getNumBlocks());
  tl->setNext (_freeList);
  _freeList = hd;
  _numAvailable += nblks;

  dirtyFullness = true;
  //  computeFullness();
}


int superblock::getNumAvailable (void)
{
  assert (isValid());
  return _numAvailable;
}


int superblock::getNumBlocks (void) const
{
  assert (isValid());
  return _numBlocks;
}


int superblock::getBlockSizeClass (void) const
{
  assert (isValid());
  return _sizeClass;
}


superblock * const superblock::getNext (void)
{
  assert (isValid());
  return _next; 
}

superblock * const superblock::getPrev (void)
{
  assert (isValid());
  return _prev; 
}


void superblock::insertBefore (superblock * nextSb) {
  assert (isValid());
  // Insert this superblock before the next one (nextSb).
  assert (nextSb != this);
  _next = nextSb;
  if (nextSb) {
    _prev = nextSb->_prev;
    nextSb->_prev = this;
  }
}


void superblock::remove (void) {
  // Remove this superblock from a doubly-linked list.
  if (_next) {
    _next->_prev = _prev;
  }
  if (_prev) {
    _prev->_next = _next;
  }
  _prev = NULL;
  _next = NULL;
}


int superblock::isValid (void) const
{
  assert (_numBlocks > 0);
  assert (_numAvailable <= _numBlocks);
  assert (_sizeClass >= 0);
  return 1;
}


void superblock::computeFullness (void)
{
  assert (isValid());
  _fullness = (((SUPERBLOCK_FULLNESS_GROUP - 1)
		* (getNumBlocks() - getNumAvailable())) / getNumBlocks());
}

int superblock::getFullness (void)
{
  assert (isValid());
  if (dirtyFullness) {
    computeFullness();
    dirtyFullness = false;
  }
  return _fullness;
}

#endif // _SUPERBLOCK_H_
