/* iksemel (XML parser for Jabber)
** Copyright (C) 2000-2003 Gurer Ozen <madcat@e-kolay.net>
** This code is free software; you can redistribute it and/or
** modify it under the terms of GNU Lesser General Public License.
*/

#include "common.h"
#include "iksemel.h"

struct iks_struct {
	struct iks_struct *next, *prev;
	struct iks_struct *children, *last_child;
	struct iks_struct *attribs, *last_attrib;
	struct iks_struct *parent;
	enum ikstype type;
	ikstack *s;
	char *name;
	char *cdata;
	size_t len;
};

/*****  Node Creating & Deleting  *****/

iks *
iks_new (const char *name)
{
	ikstack *s;
	iks *x;

	s = iks_stack_new (DEFAULT_IKS_CHUNK_SIZE);
	if (!s) return NULL;
	x = iks_new_within (name, s);
	if (!x) {
		iks_stack_delete (s);
		return NULL;
	}
	return x;
}

iks *
iks_new_within (const char *name, ikstack *s)
{
	iks *x;

	x = iks_stack_alloc (s, sizeof (iks));
	if (!x) return NULL;
	memset (x, 0, sizeof (iks));
	x->s = s;
	x->type = IKS_TAG;
	if(name) {
		x->name = iks_stack_strdup (s, name, 0);
		if (!x->name) return NULL;
	}
	return x;
}

iks *
iks_insert (iks *x, const char *name)
{
	iks *y;

	if (!x) return NULL;

	y = iks_new_within (name, x->s);
	if (!y) return NULL;
	y->parent = x;
	if (!x->children) x->children=y;
	if (x->last_child) {
		x->last_child->next = y;
		y->prev = x->last_child;
	}
	x->last_child = y;
	return y;
}

iks *
iks_insert_cdata (iks *x, const char *data, size_t len)
{
	iks *y;

	if(!x || !data) return NULL;
	if(len == 0) len = strlen (data);

	y = x->last_child;
	if (y && y->type == IKS_CDATA) {
		y->cdata = iks_stack_strecat (x->s, y->cdata, y->len, data, len);
		y->len += len;
	} else {
		y = iks_insert (x, NULL);
		if (!y) return NULL;
		y->type = IKS_CDATA;
		y->cdata = iks_stack_alloc (x->s, len + 1);
		if (!y->cdata) return NULL;
		memcpy (y->cdata, data, len);
		y->cdata[len] = '\0';
		y->len = len;
	}
	return y;
}

iks *
iks_insert_attrib (iks *x, const char *name, const char *value)
{
	iks *y;
	size_t len;

	if (!x) return NULL;

	y = x->attribs;
	while (y) {
		if (strcmp (name, y->name) == 0) break;
		y = y->next;
	}
	if (NULL == y) {
		y = iks_new_within (name, x->s);
		if (!y) return NULL;
		y->parent = x;
		if (!x->attribs) x->attribs = y;
		if (x->last_attrib) {
			x->last_attrib->next = y;
			y->prev = x->last_attrib;
		}
		x->last_attrib = y;
	}

	if (value) {
		len = strlen (value);
		y->type = IKS_ATTRIBUTE;
		y->cdata = iks_stack_alloc (x->s, len + 1);
		if (!y->cdata) return NULL;
		memcpy (y->cdata, value, len);
		*(y->cdata + len) = '\0';
		y->len = len;
	} else {
		if (y->next) y->next->prev = y->prev;
		if (y->prev) y->prev->next = y->next;
		if (x->attribs == y) x->attribs = y->next;
		if (x->last_attrib == y) x->last_attrib = y->prev;
	}

	return y;
}

iks *
iks_insert_node (iks *x, iks *y)
{
	y->parent = x;
	if (!x->children) x->children=y;
	if (x->last_child) {
		x->last_child->next = y;
		y->prev = x->last_child;
	}
	x->last_child = y;
	return y;
}

void
iks_hide (iks *x)
{
	iks *y;

	if (!x) return;

	if (x->prev) x->prev->next = x->next;
	if (x->next) x->next->prev = x->prev;
	y = x->parent;
	if (y) {
		if (y->children == x) y->children = x->next;
		if (y->last_child == x) y->last_child = x->prev;
	}
}

void
iks_delete (iks *x)
{
	if (x) iks_stack_delete (x->s);
}

/*****  Node Traversing  *****/

iks *
iks_next (iks *x)
{
	if (x) return x->next;
	return NULL;
}

iks *
iks_next_tag (iks *x)
{
	if (x) {
		while (1) {
			x = x->next;
			if (NULL == x) break;
			if (IKS_TAG == x->type) return x;
		}
	}
	return NULL;
}

iks *
iks_prev (iks *x)
{
	if (x) return x->prev;
	return NULL;
}

iks *
iks_prev_tag (iks *x)
{
	if (x) {
		while (1) {
			x = x->prev;
			if (NULL == x) break;
			if (IKS_TAG == x->type) return x;
		}
	}
	return NULL;
}

iks *
iks_parent (iks *x)
{
	if (x) return x->parent;
	return NULL;
}

iks *
iks_root (iks *x)
{
	if (x) {
		while (x->parent)
			x = x->parent;
	}
	return x;
}

iks *
iks_child (iks *x)
{
	if (x) return x->children;
	return NULL;
}

iks *
iks_first_tag (iks *x)
{
	if (x) {
		x = x->children;
		while (x) {
			if (IKS_TAG == x->type) return x;
			x = x->next;
		}
	}
	return NULL;
}

iks *
iks_attrib (iks *x)
{
	if (x) return x->attribs;
	return NULL;
}

iks *
iks_find (iks *x, const char *name)
{
	iks *y;

	if (!x) return NULL;
	y = x->children;
	while (y) {
		if (y->name && strcmp (y->name, name) == 0) return y;
		y = y->next;
	}
	return NULL;
}

char *
iks_find_cdata (iks *x, const char *name)
{
	iks *y;

	y = iks_find (x, name);
	if (!y) return NULL;
	y = y->children;
	if (!y) return NULL;
	return y->cdata;
}

char *
iks_find_attrib (iks *x, const char *name)
{
	iks *y;

	if (!x) return NULL;

	y = x->attribs;
	while (y) {
		if (y->name && strcmp (y->name, name) == 0) return y->cdata;
		y = y->next;
	}
	return NULL;
}

iks *
iks_find_with_attrib (iks *x, const char *tagname, const char *attrname, const char *value)
{
	iks *y;

	if (NULL == x) return NULL;

	if (tagname) {
		for (y = x->children; y; y = y->next) {
			if (IKS_TAG == y->type
				&& strcmp (y->name, tagname) == 0
				&& iks_strcmp (iks_find_attrib (y, attrname), value) == 0) {
					return y;
			}
		}
	} else {
		for (y = x->children; y; y = y->next) {
			if (IKS_TAG == y->type
				&& iks_strcmp (iks_find_attrib (y, attrname), value) == 0) {
					return y;
			}
		}
	}
	return NULL;
}

/*****  Node Information  *****/

ikstack *
iks_stack (iks *x)
{
	if (x) return x->s;
	return NULL;
}

enum ikstype
iks_type (iks *x)
{
	if (x) return x->type;
	return IKS_NONE;
}

char *
iks_name (iks *x)
{
	if (x) return x->name;
	return NULL;
}

char *
iks_cdata (iks *x)
{
	if (x) return x->cdata;
	return NULL;
}

size_t
iks_cdata_size (iks *x)
{
	if (x) return x->len;
	return 0;
}

int
iks_has_children (iks *x)
{
	if (x && x->children) return 1;
	return 0;
}

int
iks_has_attribs (iks *x)
{
	if (x && x->attribs) return 1;
	return 0;
}

/*****  Serializing  *****/

static size_t
escape_size (char *src, size_t len)
{
	size_t sz;
	char c;
	int i;

	sz = 0;
	for (i = 0; i < len; i++) {
		c = src[i];
		switch (c) {
			case '&': sz += 4; break;
			case '\'': sz += 5; break;
			case '"': sz += 5; break;
			case '<': sz += 3; break;
			case '>': sz += 3; break;
			default: sz++; break;
		}
	}
	return sz;
}

static char *
my_strcat (char *dest, char *src, size_t len)
{
	if (0 == len) len = strlen (src);
	memcpy (dest, src, len);
	return dest + len;
}

static char *
escape (char *dest, char *src, size_t len)
{
	char c;
	int i;
	int j = 0;

	for (i = 0; i < len; i++) {
		c = src[i];
		if ('&' == c || '<' == c || '>' == c || '\'' == c || '"' == c) {
			if (i - j > 0) dest = my_strcat (dest, src + j, i - j);
			j = i + 1;
			switch (c) {
			case '&': dest = my_strcat (dest, "&amp;", 5); break;
			case '\'': dest = my_strcat (dest, "&apos;", 6); break;
			case '"': dest = my_strcat (dest, "&quot;", 6); break;
			case '<': dest = my_strcat (dest, "&lt;", 4); break;
			case '>': dest = my_strcat (dest, "&gt;", 4); break;
			}
		}
	}
	if (i - j > 0) dest = my_strcat (dest, src + j, i - j);
	return dest;
}

char *
iks_string (ikstack *s, iks *x)
{
	size_t size;
	int level, dir;
	iks *y, *z;
	char *ret, *t;

	if (!x) return NULL;

	if (x->type == IKS_CDATA) {
		if (s) {
			iks_stack_strcat (s, x->cdata, x->len);
			return iks_stack_print (s);
		} else {
			ret = iks_malloc (x->len);
			memcpy (ret, x->cdata, x->len);
			return ret;
		}
	}

	size = 0;
	level = 0;
	dir = 0;
	y = x;
	while (1) {
		if (dir==0) {
			if (y->type == IKS_TAG) {
				size++;
				size += strlen (y->name);
				for (z = y->attribs; z; z = z->next) {
					size += 4 + strlen (z->name) + escape_size (z->cdata, strlen (z->cdata));
				}
				if (y->children) {
					size++;
					y = y->children;
					level++;
					continue;
				} else {
					size += 2;
				}
			} else {
				size += escape_size (y->cdata, y->len);
			}
		}
		z = y->next;
		if (z) {
			if (0 == level) {
				if (y->children) size += 3 + strlen (y->name);
				break;
			}
			y = z;
			dir = 0;
		} else {
			y = y->parent;
			level--;
			if (level >= 0) size += 3 + strlen (y->name);
			if (level < 1) break;
			dir = 1;
		}
	}

	if (s) ret = iks_stack_alloc (s, size + 1);
	else ret = iks_malloc (size + 1);

	if (!ret) return NULL;

	t = ret;
	level = 0;
	dir = 0;
	while (1) {
		if (dir==0) {
			if (x->type == IKS_TAG) {
				*t++ = '<';
				t = my_strcat (t, x->name, 0);
				y = x->attribs;
				while (y) {
					*t++ = ' ';
					t = my_strcat (t, y->name, 0);
					*t++ = '=';
					*t++ = '\'';
					t = escape (t, y->cdata, strlen (y->cdata));
					*t++ = '\'';
					y = y->next;
				}
				if (x->children) {
					*t++ = '>';
					x = x->children;
					level++;
					continue;
				} else {
					*t++ = '/';
					*t++ = '>';
				}
			} else {
				t = escape (t, x->cdata, x->len);
			}
		}
		y = x->next;
		if (y) {
			if (0 == level) {
				if (x->children) {
					*t++ = '<';
					*t++ = '/';
					t = my_strcat (t, x->name, 0);
					*t++ = '>';
				}
				break;
			}
			x = y;
			dir = 0;
		} else {
			x = x->parent;
			level--;
			if (level >= 0) {
					*t++ = '<';
					*t++ = '/';
					t = my_strcat (t, x->name, 0);
					*t++ = '>';
				}
			if (level < 1) break;
			dir = 1;
		}
	}
	*t = '\0';

	return ret;
}

/*****  Copying  *****/

iks *
iks_copy_within (iks *x, ikstack *s)
{
	int level=0, dir=0;
	iks *copy = NULL;
	iks *cur = NULL;
	iks *y;

	while (1) {
		if (dir == 0) {
			if (x->type == IKS_TAG) {
				if (copy == NULL) {
					copy = iks_new_within (x->name, s);
					cur = copy;
				} else {
					cur = iks_insert (cur, x->name);
				}
				for (y = x->attribs; y; y = y->next) {
					iks_insert_attrib (cur, y->name, y->cdata);
				}
				if (x->children) {
					x = x->children;
					level++;
					continue;
				} else {
					cur = cur->parent;
				}
			} else {
				iks_insert_cdata (cur, x->cdata, x->len);
			}
		}
		y = x->next;
		if (y) {
			if (0 == level) break;
			x = y;
			dir = 0;
		} else {
			if (level < 2) break;
			level--;
			x = x->parent;
			cur = cur->parent;
			dir = 1;
		}
	}
	return copy;
}

iks *
iks_copy (iks *x)
{
	return iks_copy_within (x, iks_stack_new (DEFAULT_IKS_CHUNK_SIZE));
}
