/* iksemel (XML parser for Jabber)
** Copyright (C) 2000-2003 Gurer Ozen <madcat@e-kolay.net>
** This code is free software; you can redistribute it and/or
** modify it under the terms of GNU Lesser General Public License.
*/

#include "common.h"
#include "iksemel.h"

struct align_test { char a; double b; };
#define DEFAULT_ALIGNMENT  ((size_t) ((char *) &((struct align_test *) 0)->b - (char *) 0))
#define ALIGN_MASK ( DEFAULT_ALIGNMENT - 1 )
#define MIN_CHUNK_SIZE ( DEFAULT_ALIGNMENT * 4 )
#define MIN_ALLOC_SIZE DEFAULT_ALIGNMENT
#define ALIGN(x) ( (x) + (DEFAULT_ALIGNMENT - ( (x) & ALIGN_MASK)) )

struct ikstack_struct {
	size_t size;
	size_t used;
	size_t pos;
	size_t last;
	struct ikstack_struct *next;
	char data[4];
};

static ikstack *
find_space (ikstack *s, size_t size)
{
	size_t i;

	i = s->size;
	while (1) {
		if (s->size - s->used >= size) return s;
		if (!s->next) {
			if (i > size) size = i;
			s->next = iks_stack_new (size);
			return s->next;
		}
		s = s->next;
	}
	return NULL;
}

ikstack *
iks_stack_new (size_t chunk_size)
{
	ikstack *s;

	if (chunk_size < MIN_CHUNK_SIZE) chunk_size = MIN_CHUNK_SIZE;
	chunk_size = ALIGN (chunk_size);

	s = iks_malloc (sizeof (ikstack) + chunk_size);
	if (!s) return NULL;
	s->size = chunk_size;
	s->used = 0;
	s->pos = -1;
	s->last = -1;
	s->next = NULL;
	return s;
}

void *
iks_stack_alloc (ikstack *s, size_t size)
{
	void *mem;

	if (size < MIN_ALLOC_SIZE) size = MIN_ALLOC_SIZE;
	size = ALIGN (size);

	s = find_space (s, size);
	if (!s) return NULL;
	mem = s->data + s->used;
	s->last = s->used;
	s->used += size;
	return mem;
}

void *
iks_stack_strdup (ikstack *s, const char *src, size_t len)
{
	char *dest;

	if (!src) return NULL;
	if (0 == len) len = strlen (src);
	dest = iks_stack_alloc (s, len + 1);
	if (!dest) return NULL;
	memcpy (dest, src, len);
	dest[len] = '\0';
	return dest;
}

int
iks_stack_strcat (ikstack *s, const char *src, size_t len)
{
	ikstack *tmp;
	size_t i;

	if (!src) return 1;
	if (0 == len) len = strlen (src);

	i = len;
	if (i > s->size) i += (len / 5);

	tmp = s;
	while (tmp) {
		if (tmp->pos != -1) break;
		tmp = tmp->next;
	}
	if (NULL == tmp) {
		tmp = find_space (s, i);
		if (NULL == tmp) return 1;
		tmp->pos = tmp->used;
		tmp->last = tmp->used;
	}

	if (tmp->size - tmp->used < len) {
		s = find_space (s, tmp->used + i);
		if (NULL == s) return 1;
		i = tmp->used - tmp->pos;
		memcpy (s->data + s->used, tmp->data + tmp->pos, i);
		tmp->used = tmp->pos;
		tmp->pos = -1;
		s->pos = s->used;
		s->last = s->used;
		s->used += i;
		tmp = s;
	}

	memcpy (tmp->data + tmp->used, src, len);
	tmp->used += len; 
	return 0;
}

int
iks_stack_strcatv (ikstack *s, ...)
{
	va_list ap;
	char *arg;
	int ret = 0;

	va_start (ap, s);
	while (1) {
		arg = va_arg (ap, char *);
		if (((void *) arg) == ((void *) s)) break;
		if (iks_stack_strcat (s, arg, 0)) {
			ret = 1;
			break;
		}
	}
	va_end (ap);

	return ret;
}

char *
iks_stack_print (ikstack *s)
{
	iks_stack_strcat (s, "\0", 1);
	while (s) {
		if (s->pos != -1) {
			char *mem = s->data + s->pos;
			s->pos = -1;
			return mem;
		}
		s = s->next;
	}
	return NULL;
}

char *
iks_stack_strecat (ikstack *s, char *old_str, size_t old_len, const char *new_str, size_t new_len)
{
	char *ret;
	ikstack *t;

	if (0 == old_len) old_len = strlen (old_str);
	if (0 == new_len) new_len = strlen (new_str);
	for (t = s; t; t = t->next) {
		if (t->data + t->last == old_str) break;
	}
	if (!t) {
		ret = iks_stack_alloc (s, old_len + new_len + 1);
		memcpy (ret, old_str, old_len);
		memcpy (ret + old_len, new_str, new_len);
		ret[old_len + new_len] = '\0';
		return ret;
	}

	if (t->size - t->used > new_len) {
		ret = t->data + t->last;
//		memcpy (t->data + t->used - 1, new_str, new_len);
		memcpy (ret + old_len, new_str, new_len);
		t->used += new_len;
		t->data[t->used] = '\0';
	} else {
		t = find_space (s, old_len + new_len + 1);
		if (NULL == t) return NULL;
		t->last = t->used;
		ret = t->data + t->used;
		memcpy (t->data + t->used, old_str, old_len);
		t->used += old_len;
		memcpy (t->data + t->used, new_str, new_len);
		t->used += new_len;
		t->data[t->used] = '\0';
		t->used++;
	}
	return ret;
}

void
iks_stack_stats (ikstack *s, int *nr_chunks, size_t *total_used)
{
	*nr_chunks = 0;
	*total_used = 0;
	while (s) {
		(*nr_chunks)++;
		(*total_used) += s->used;
		s = s->next;
	}
}

void
iks_stack_delete (ikstack *s)
{
	ikstack *tmp;

	while (s) {
		tmp = s->next;
		iks_free (s);
		s = tmp;
	}
}
