/*
 * Copyright 2005 [ini4j] Development Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ini4j;

import java.io.BufferedReader;
import java.io.InputStream;
import java.io.InputStreamReader;

/**
 * JDK JAR Services API alap service keres osztly.
 *
 * @author Szkiba Ivn
 * @version $Name: v0_2_6 $
 */
class ServiceFinder
{
    /**
     * Service osztly nevnek keresse
     *
     * a JDK JAR specifikciban definilt <B>Services API</B>-nak
     * megfelelen service osztly keress.</p><p>
     * Az implementl osztly nv keresse a <CODE>serviceId</CODE> nev
     * system property vizsglatval kezddik. Amennyiben nincs ilyen
     * property, gy a keress a
     * <CODE>/META-INF/services/<I>serviceId</I></CODE> nev file tartalmval
     * folytatdik. Amennyiben nincs ilyen nev file, gy a paramterknt tadott
     * <CODE>defaultService</CODE> lesz az osztly neve.</p><p>
     * @param serviceId keresett osztly/service neve
     * @param defaultService alaprtelmezett implementl osztly neve
     * @throws IllegalArgumentException keressi vagy pldnyostsi hiba esetn
     * @return a keresett osztly neve
     */
    protected static String findServiceClassName(String serviceId, String defaultService)
                                       throws IllegalArgumentException
    {
        if (defaultService == null)
        {
            throw new IllegalArgumentException("Provider for " + serviceId + " cannot be found");
        }

        ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
        String serviceClassName = null;

        // Use the system property first
        try
        {
            String systemProp = System.getProperty(serviceId);

            if (systemProp != null)
            {
                serviceClassName = systemProp;
            }
        }
        catch (SecurityException x)
        {
            ;
        }

        if (serviceClassName == null)
        {
            String servicePath = "META-INF/services/" + serviceId;

            // try to find services in CLASSPATH
            try
            {
                InputStream is = null;

                if (classLoader == null)
                {
                    is = ClassLoader.getSystemResourceAsStream(servicePath);
                }
                else
                {
                    is = classLoader.getResourceAsStream(servicePath);
                }

                if (is != null)
                {
                    BufferedReader rd = new BufferedReader(new InputStreamReader(is, "UTF-8"));

                    String line = rd.readLine();
                    rd.close();

                    if ((line != null) && !"".equals(line = line.trim()) )
                    {
                        serviceClassName = line.split("\\s|#")[0];
                    }
                }
            }
            catch (Exception x)
            {
                ;
            }
        }

        if (serviceClassName == null)
        {
            serviceClassName = defaultService;
        }

        return serviceClassName;
    }

    /**
     * Service osztly keress
     *
     * a JDK JAR specifikciban definilt <B>Services API</B>-nak
     * megfelelen service osztly keress.</p><p>
     * Az implementl osztly nv keresse a <CODE>serviceId</CODE> nev
     * system property vizsglatval kezddik. Amennyiben nincs ilyen
     * property, gy a keress a
     * <CODE>/META-INF/services/<I>serviceId</I></CODE> nev file tartalmval
     * folytatdik. Amennyiben nincs ilyen nev file, gy a paramterknt tadott
     * <CODE>defaultService</CODE> lesz az osztly neve.</p><p>
     * @param serviceId keresett osztly/service neve
     * @param defaultService alaprtelmezett implementl osztly neve
     * @throws IllegalArgumentException keressi vagy pldnyostsi hiba esetn
     * @return a keresett osztly objektum
     */
    protected static Class findServiceClass(String serviceId, String defaultService)
                                  throws IllegalArgumentException
    {
        ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
        String serviceClassName = findServiceClassName(serviceId, defaultService);

        try
        {
            return (classLoader == null) ? Class.forName(serviceClassName) : classLoader.loadClass(serviceClassName);
        }
        catch (ClassNotFoundException x)
        {
            throw (IllegalArgumentException) new IllegalArgumentException("Provider " + serviceClassName + " not found").initCause(x);
        }
    }

    /**
     * Service objektum keress s pldnyosts
     *
     * a JDK JAR specifikciban definilt <B>Services API</B>-nak
     * megfelelen service osztly keress, majd pedig pldny kpzs a context
     * ClassLoader segtsgvel.</p><p>
     * Az implementl osztly nv keresse a <CODE>serviceId</CODE> nev
     * system property vizsglatval kezddik. Amennyiben nincs ilyen
     * property, gy a keress a
     * <CODE>/META-INF/services/<I>serviceId</I></CODE> nev file tartalmval
     * folytatdik. Amennyiben nincs ilyen nev file, gy a paramterknt tadott
     * <CODE>defaultService</CODE> lesz az osztly neve.</p><p>
     * A fenti keresst kveten trtnik a pldny kpzs. A visszatrsi
     * rtk mindig egy valdi objektum, lvn minden hiba exception-t generl.
     * @param serviceId keresett osztly/service neve
     * @param defaultService alaprtelmezett implementl osztly neve
     * @throws IllegalArgumentException keressi vagy pldnyostsi hiba esetn
     * @return a keresett osztly implementl objektum
     */
    protected static Object findService(String serviceId, String defaultService)
                              throws IllegalArgumentException
    {
        try
        {
            return findServiceClass(serviceId, defaultService).newInstance();
        }
        catch (Exception x)
        {
            throw (IllegalArgumentException) new IllegalArgumentException("Provider " + serviceId + " could not be instantiated: " + x).initCause(x);
        }
    }
}
