
/*
  cc -g -c isofs_wrap.c
*/

/*
  libisofs related functions of libisoburn.

  Copyright 2007 - 2008  Vreixo Formoso Lopes <metalpain2002@yahoo.es>
                         Thomas Schmitt <scdbackup@gmx.net>
*/

#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#ifndef Xorriso_standalonE

#include <libburn/libburn.h>

#include <libisofs/libisofs.h>

#else /* ! Xorriso_standalonE */

#include "../libisofs/libisofs.h"
#include "../libburn/libburn.h"

#endif /* Xorriso_standalonE */

#include "isoburn.h"
#include "libisoburn.h"

#define BP(a,b) [(b) - (a) + 1]

struct ecma119_pri_vol_desc
{
	uint8_t vol_desc_type		BP(1, 1);
	uint8_t std_identifier		BP(2, 6);
	uint8_t vol_desc_version	BP(7, 7);
	uint8_t unused1			BP(8, 8);
	uint8_t system_id		BP(9, 40);
	uint8_t volume_id		BP(41, 72);
	uint8_t unused2			BP(73, 80);
	uint8_t vol_space_size		BP(81, 88);
	uint8_t unused3			BP(89, 120);
	uint8_t vol_set_size		BP(121, 124);
	uint8_t vol_seq_number		BP(125, 128);
	uint8_t block_size		BP(129, 132);
	uint8_t path_table_size		BP(133, 140);
	uint8_t l_path_table_pos	BP(141, 144);
	uint8_t opt_l_path_table_pos	BP(145, 148);
	uint8_t m_path_table_pos	BP(149, 152);
	uint8_t opt_m_path_table_pos	BP(153, 156);
	uint8_t root_dir_record		BP(157, 190);
	uint8_t	vol_set_id		BP(191, 318);
	uint8_t publisher_id		BP(319, 446);
	uint8_t data_prep_id		BP(447, 574);
	uint8_t application_id		BP(575, 702);
	uint8_t copyright_file_id	BP(703, 739);
	uint8_t abstract_file_id	BP(740, 776);
	uint8_t bibliographic_file_id	BP(777, 813);
	uint8_t vol_creation_time	BP(814, 830);
	uint8_t vol_modification_time	BP(831, 847);
	uint8_t vol_expiration_time	BP(848, 864);
	uint8_t vol_effective_time	BP(865, 881);
	uint8_t file_structure_version	BP(882, 882);
	uint8_t reserved1		BP(883, 883);
	uint8_t app_use			BP(884, 1395);
	uint8_t reserved2		BP(1396, 2048);
};

static
uint32_t iso_read_lsb(const uint8_t *buf, int bytes)
{
	int i;
	uint32_t ret = 0;

	for (i=0; i<bytes; i++) {
		ret += ((uint32_t) buf[i]) << (i*8);
	}
	return ret;
}


/* API function. See libisoburn.h
*/
IsoImage *isoburn_get_attached_image(struct burn_drive *d)
{
 int ret;
 struct isoburn *o= NULL;
 ret = isoburn_find_emulator(&o, d, 0);
 if (ret < 0)
   return NULL;
  
 if (o == NULL) {
   return NULL;
 }
 iso_image_ref(o->image);
 return o->image;
}


static void isoburn_idle_free_function(void *ignored)
{
 return;
}


/* API function. See libisoburn.h
*/
int isoburn_read_image(struct burn_drive *d,
                       struct isoburn_read_opts *read_opts, 
                       IsoImage **image)
{
 int ret, int_num, dummy;
 IsoReadOpts *ropts= NULL;
 IsoReadImageFeatures *features= NULL;
 uint32_t ms_block;
 char msg[160];
 enum burn_disc_status status= BURN_DISC_BLANK;
 IsoDataSource *ds= NULL;
 struct isoburn *o= NULL;

 if(d != NULL) {
   ret = isoburn_find_emulator(&o, d, 0);
   if (ret < 0 || o == NULL)
     return 0;
   status = isoburn_disc_get_status(d);
 }
 if(read_opts==NULL) {
   isoburn_msgs_submit(o, 0x00060000,
                       "Program error: isoburn_read_image: read_opts==NULL",
                       0, "FATAL", 0);
   return(-1);
 }
 if (d == NULL || status == BURN_DISC_BLANK || read_opts->pretend_blank) {
create_blank_image:;
   /*
    * Blank disc, we create a new image without files.
    */
   
   if (d == NULL) {
     /* New empty image without relation to a drive */
     if (image==NULL) {
       isoburn_msgs_submit(o, 0x00060000,
                           "Program error: isoburn_read_image: image==NULL",
                           0, "FATAL", 0);
       return -1;
     }
     /* create a new image */
     ret = iso_image_new("ISOIMAGE", image);
     if (ret < 0) {
       isoburn_report_iso_error(ret, "Cannot create image", 0, "FATAL", 0);
       return ret;
     }
   } else {
     /* Blank new image for the drive */
     iso_image_unref(o->image);
     ret = iso_image_new("ISOIMAGE", &o->image);
     if (ret < 0) {
       isoburn_report_iso_error(ret, "Cannot create image", 0, "FATAL", 0);
       return ret;
     }
     if (image) {
       *image = o->image;
       iso_image_ref(*image); /*protects object from premature free*/
     }
   }
   return 1;
 }
 
 if (status != BURN_DISC_APPENDABLE && status != BURN_DISC_FULL) {
   isoburn_msgs_submit(o, 0x00060000,
                    "Program error: isoburn_read_image: incorrect disc status",
                    0, "FATAL", 0);
   return -4;	
 }
 
 memset((char *) &ropts, 0, sizeof(ropts));

 ret = isoburn_disc_get_msc1(d, &int_num);
 if (ret <= 0)
   return -2;
 ms_block= int_num;
 ret = isoburn_read_iso_head(d, int_num, &dummy, NULL, 0);
 if (ret <= 0) {
   sprintf(msg, "No ISO 9660 image at LBA %d. Creating blank image.", int_num);
   isoburn_msgs_submit(o, 0x00060000, msg, 0, "WARNING", 0);
   goto create_blank_image;
 }

 /* create the data source */
 ret = iso_read_opts_new(&ropts, 0);
 if (ret < 0) {
   isoburn_report_iso_error(ret, "Cannot create write opts", 0, "FATAL", 0);
   return ret;
 }
 /* Important: do not return until iso_read_opts_free() */
 iso_read_opts_set_start_block(ropts, ms_block);
 iso_read_opts_set_no_rockridge(ropts, read_opts->norock);
 iso_read_opts_set_no_joliet(ropts, read_opts->nojoliet);
 iso_read_opts_set_no_iso1999(ropts, read_opts->noiso1999);
 iso_read_opts_set_preferjoliet(ropts, read_opts->preferjoliet);
 iso_read_opts_set_default_permissions(ropts,
                                       read_opts->mode, read_opts->dirmode);
 iso_read_opts_set_default_uid(ropts, read_opts->uid);
 iso_read_opts_set_default_gid(ropts, read_opts->gid);
 iso_read_opts_set_input_charset(ropts, read_opts->input_charset);

 /* <<< experimental API call of libisofs
 iso_read_opts_set_error_behavior(ropts, 1);
 */

 ds = isoburn_data_source_new(d);
 if(o->iso_data_source!=NULL)
   iso_data_source_unref(o->iso_data_source);
 o->iso_data_source= ds;
 iso_image_attach_data(o->image, o->read_pacifier_handle,
                       isoburn_idle_free_function);
 if(o->read_pacifier_handle==NULL)
   iso_tree_set_report_callback(o->image, NULL);
 else
   iso_tree_set_report_callback(o->image, o->read_pacifier);
 ret = iso_image_import(o->image, ds, ropts, &features);
 iso_tree_set_report_callback(o->image, NULL);
 iso_read_opts_free(ropts);

 if (ret < 0) {
   isoburn_report_iso_error(ret, "Cannot import image", 0, "FAILURE", 0);
   return ret;
 }
 /* Important: do not return until free(features) */
 if (image!=NULL) {
   *image = o->image;
   iso_image_ref(*image); /*protects object from premature free*/
 }
 read_opts->hasRR = iso_read_image_features_has_rockridge(features);
 read_opts->hasJoliet = iso_read_image_features_has_joliet(features);
 read_opts->hasIso1999 = iso_read_image_features_has_iso1999(features);
 read_opts->hasElTorito = iso_read_image_features_has_eltorito(features);
 read_opts->size = iso_read_image_features_get_size(features);
 iso_read_image_features_destroy(features);
 return 1;
}


/* API function. See libisoburn.h
*/
int isoburn_attach_image(struct burn_drive *d, IsoImage *image)
{
 int ret;
 struct isoburn *o;

 ret = isoburn_find_emulator(&o, d, 0);
 if (ret < 0 || o == NULL)
   return 0;
 if (image == NULL) {
   isoburn_msgs_submit(o, 0x00060000,
                       "Program error: isoburn_attach_image: image==NULL",
                       0, "FATAL", 0);
   return -1;
 }
 if(o->image != NULL)
   iso_image_unref(o->image);
 o->image = image;
 return(1);
}


/* API function. See libisoburn.h
*/
int isoburn_activate_session(struct burn_drive *drive)
{
 int ret;
 struct isoburn *o;

 ret = isoburn_find_emulator(&o, drive, 0);
 if (ret < 0)
   return -1;

 if (o->emulation_mode != 1)
   return 1; /* don't need to activate session */
 if (o->fabricated_msc2 >= 0)
   return 1; /* blind growing: do not alter anything outside the session */
 
 if (!(o->fabricated_disc_status == BURN_DISC_APPENDABLE ||
       (o->fabricated_disc_status == BURN_DISC_BLANK &&
        o->zero_nwa > 0)))
   return 1;
 
 ret = burn_random_access_write(drive, (off_t) 0, (char*)o->target_iso_head, 
                                Libisoburn_target_head_sizE, 1);

 return ret;
}


/** Initialize the emulation of multi-session on random access media.
    The need for emulation is confirmed already.
    @param o A freshly created isoburn object. isoburn_create_data_source() was
             already called, nevertheless.
    @return <=0 error , 1 = success
*/
int isoburn_start_emulation(struct isoburn *o, int flag)
{
 int ret, i;
 off_t data_count;
 struct burn_drive *drive;
 struct ecma119_pri_vol_desc *pvm;
 
 if(o==NULL) {
   isoburn_msgs_submit(NULL, 0x00060000,
                       "Program error: isoburn_start_emulation: o==NULL",
                       0, "FATAL", 0);
   return -1;
 }

 drive= o->drive;

 /* we can assume 0 as start block for image */
 /* TODO what about ms? where we validate valid iso image in ms disc? */
 ret = burn_read_data(drive, (off_t) 0, (char*)o->target_iso_head, 
                      (off_t) Libisoburn_target_head_sizE, &data_count, 2);

 /* an error means an empty disc */
 if (ret <= 0) {
   o->fabricated_disc_status= BURN_DISC_BLANK;
   return 1;
 }

 /* check first 64K. If 0's, the disc is treated as a blank disc, and thus
    overwritten without extra check. */
 i = Libisoburn_target_head_sizE;
 while (i && !o->target_iso_head[i-1]) 
   --i;

 if (!i) {
   o->fabricated_disc_status= BURN_DISC_BLANK;
   return 1;
 }

 pvm = (struct ecma119_pri_vol_desc *)(o->target_iso_head + 16 * 2048);

 if (!strncmp((char*)pvm->std_identifier, "CD001", 5)) {
   off_t size;

   /* sanity check */
   if (pvm->vol_desc_type[0] != 1 || pvm->vol_desc_version[0] != 1 
       || pvm->file_structure_version[0] != 1 ) {
     /* TODO for now I treat this as a full disc */
     o->fabricated_disc_status= BURN_DISC_FULL;
     return 1;
   }

   /* ok, PVM found, set size */
   size = (off_t) iso_read_lsb(pvm->vol_space_size, 4);
   size *= (off_t) 2048; /* block size in bytes */
   isoburn_set_start_byte(o, size, 0);
   o->fabricated_disc_status= BURN_DISC_APPENDABLE;
 } else if (!strncmp((char*)pvm->std_identifier, "CDXX1", 5)) {

   /* empty image */
   isoburn_set_start_byte(o, o->zero_nwa * 2048, 0);
   o->fabricated_disc_status= BURN_DISC_BLANK;
 } else {
   /* treat any disc in an unknown format as full */
   o->fabricated_disc_status= BURN_DISC_FULL;
 }
 return 1;
}


/** Alters and writes the first 64 kB of a "media" to invalidate
    an ISO image. (It shall stay restorable by skilled humans, though).
    The result shall especially keep libisoburn from accepting the media
    image as ISO filesystem.
    @param o A fully activated isoburn object. isoburn_start_emulation()
             was already called.
    @return <=0 error , 1 = success
*/
int isoburn_invalidate_iso(struct isoburn *o, int flag)
{
 /* 
  * replace CD001 with CDXX1 in PVM.
  * I think this is enought for invalidating an iso image
  */
 strncpy((char*)o->target_iso_head + 16 * 2048 + 1, "CDXX1", 5);
 return isoburn_activate_session(o->drive);
}


/* API @since 0.1.0 */
int isoburn_set_read_pacifier(struct burn_drive *drive,
                              int (*read_pacifier)(IsoImage*, IsoFileSource*),
                              void *read_handle)
{
 int ret;
 struct isoburn *o;

 ret = isoburn_find_emulator(&o, drive, 0);
 if(ret < 0 || o == NULL)
   return -1;
 o->read_pacifier_handle= read_handle;
 o->read_pacifier= read_pacifier;
 return(1);
}

