package jdepend.framework;

import java.io.*;

import junit.framework.*;

/**
 * The <code>ConstraintTest</code> is a <code>TestCase</code> 
 * to test package dependency constraints.
 * <p>
 * This test analyzes the JDepend class files.
 *
 * @author <b>Mike Clark</b> (mike@clarkware.com)
 * @author Clarkware Consulting, Inc.
 */
 
public class ConstraintTest extends TestCase {
	
	private JDepend jdepend;
	
	private String BASE_DIR;
	

	public ConstraintTest(String name) {
		super(name);
	}

	protected void setUp() {
	    
        BASE_DIR = System.getProperty("jdepend.home");
        if (BASE_DIR == null) {
            fail("Property 'jdepend.home' not defined");
        }
        
        PackageFilter filter = new PackageFilter();
        filter.addPackage("java.*");
        filter.addPackage("javax.*");
		jdepend = new JDepend(filter);
	}

	protected void tearDown() {
		jdepend = null;
	}
	
	public void testConstraintPass() {
	
		DependencyConstraint constraint = new DependencyConstraint();
		
		JavaPackage expectedA = constraint.addPackage("A");
		JavaPackage expectedB = constraint.addPackage("B");
		
		expectedA.dependsUpon(expectedB);
		
		JavaPackage actualA = new JavaPackage("A");
		JavaPackage actualB = new JavaPackage("B");

		actualA.dependsUpon(actualB);
		
		jdepend.addPackage(actualA);
		jdepend.addPackage(actualB);
						
		assertEquals(true, jdepend.dependencyMatch(constraint));	
	}
	
	public void testConstraintFail() {
	
		DependencyConstraint constraint = new DependencyConstraint();
		
		JavaPackage expectedA = constraint.addPackage("A");
		JavaPackage expectedB = constraint.addPackage("B");
		JavaPackage expectedC = constraint.addPackage("C");
								
		expectedA.dependsUpon(expectedB);

		JavaPackage actualA = new JavaPackage("A");
		JavaPackage actualB = new JavaPackage("B");
		JavaPackage actualC = new JavaPackage("C");
		
		actualA.dependsUpon(actualB);
		actualA.dependsUpon(actualC);
		
		jdepend.addPackage(actualA);
		jdepend.addPackage(actualB);
		jdepend.addPackage(actualC);
						
		assertEquals(false, jdepend.dependencyMatch(constraint));	
	}
	
	public void testAnalyzedConstraint() throws IOException {

		String filename = BASE_DIR + 
			File.separator + "build";
		
		jdepend.addDirectory(filename);
		
		jdepend.analyze();
		
		
		DependencyConstraint constraint = new DependencyConstraint();
		
		JavaPackage junitframework = constraint.addPackage("junit.framework");
		JavaPackage junitui = constraint.addPackage("junit.textui");
		JavaPackage framework = constraint.addPackage("jdepend.framework");
		JavaPackage text = constraint.addPackage("jdepend.textui");
		JavaPackage xml = constraint.addPackage("jdepend.xmlui");
		JavaPackage swing = constraint.addPackage("jdepend.swingui");
		
		framework.dependsUpon(junitframework);	
		framework.dependsUpon(junitui);	
		text.dependsUpon(framework);
		xml.dependsUpon(text);
		swing.dependsUpon(framework);
		
		assertEquals(true, jdepend.dependencyMatch(constraint));
	}

	public static void main(String args[]) {
		junit.textui.TestRunner.run(ConstraintTest.class);
	}
}
