/* CertificateRequest.java -- SSL CertificateRequest message.
   Copyright (C) 2003  Casey Marshall <rsdio@metastatic.org>

This program is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2 of the License, or (at your
option) any later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License along
with this program; if not, write to the

   Free Software Foundation, Inc.,
   59 Temple Place, Suite 330,
   Boston, MA  02111-1307
   USA  */


package org.metastatic.jessie.provider;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.EOFException;
import java.io.InputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.io.StringWriter;

import java.util.LinkedList;

import javax.security.auth.x500.X500Principal;

final class CertificateRequest implements Handshake.Body
{

  // Fields.
  // -------------------------------------------------------------------------

  private final ClientType[] types;
  private final X500Principal[] authorities;

  // Constructor.
  // -------------------------------------------------------------------------

  CertificateRequest(ClientType[] types,
                     X500Principal[] authorities)
  {
    if (types == null || authorities == null)
      {
        throw new NullPointerException();
      }
    this.types = types;
    this.authorities = authorities;
  }

  // Class methods.
  // -------------------------------------------------------------------------

  static CertificateRequest read(InputStream in) throws IOException
  {
    DataInputStream din = new DataInputStream(in);
    ClientType[] types = new ClientType[din.readUnsignedByte()];
    for (int i = 0; i < types.length; i++)
      {
        types[i] = ClientType.read(din);
      }

    LinkedList authorities = new LinkedList();
    byte[] buf = new byte[din.readUnsignedShort()];
    din.readFully(buf);
    ByteArrayInputStream bin = new ByteArrayInputStream(buf);
    while (bin.available() > 0)
      {
        buf = new byte[(bin.read() & 0xFF) << 8 | (bin.read() & 0xFF)];
        bin.read(buf);
        authorities.add(new X500Principal(buf));
      }
    return new CertificateRequest(types, (X500Principal[])
               authorities.toArray(new X500Principal[authorities.size()]));
  }

  // Instance methods.
  // -------------------------------------------------------------------------

  public void write(OutputStream out) throws IOException
  {
    ByteArrayOutputStream bout = new ByteArrayOutputStream();
    out.write(types.length);
    for (int i = 0; i < types.length; i++)
      {
        out.write(types[i].getValue());
      }

    for (int i = 0; i < authorities.length; i++)
      {
        byte[] buf = authorities[i].getEncoded();
        bout.write(buf.length >>> 8 & 0xFF);
        bout.write(buf.length & 0xFF);
        bout.write(buf, 0, buf.length);
      }
    out.write(bout.size() >>> 8 & 0xFF);
    out.write(bout.size() & 0xFF);
    bout.writeTo(out);
  }

  ClientType[] getTypes()
  {
    return types;
  }

  String[] getTypeStrings()
  {
    try
      {
        return (String[]) Util.transform(types, String.class, "toString", null);
      }
    catch (Exception x)
      {
        return null;
      }
  }

  X500Principal[] getAuthorities()
  {
    return authorities;
  }

  public String toString()
  {
    StringWriter str = new StringWriter();
    PrintWriter out = new PrintWriter(str);
    out.println("struct {");
    out.print("  types = ");
    for (int i = 0; i < types.length; i++)
      {
        out.print(types[i]);
        if (i != types.length - 1)
          out.print(", ");
      }
    out.println(";");
    out.println("  authorities =");
    for (int i = 0; i < authorities.length; i++)
      {
        out.print("    ");
        out.print(authorities[i].getName());
        if (i != types.length - 1)
          out.println(",");
      }
    out.println(";");
    out.println("} CertificateRequest;");
    return str.toString();
  }

  // Inner class.
  // -------------------------------------------------------------------------

  static final class ClientType implements Enumerated
  {

    // Constants and fields.
    // -----------------------------------------------------------------------

    static final ClientType
      RSA_SIGN     = new ClientType(1), DSS_SIGN     = new ClientType(2),
      RSA_FIXED_DH = new ClientType(3), DSS_FIXED_DH = new ClientType(4);

    private final int value;

    // Constructor.
    // -----------------------------------------------------------------------

    private ClientType(int value)
    {
      this.value = value;
    }

    // Class method.
    // -----------------------------------------------------------------------

    static ClientType read(InputStream in) throws IOException
    {
      int i = in.read();
      if (i == -1)
        {
          throw new EOFException("unexpected end of input stream");
        }
      switch (i & 0xFF)
        {
        case 1: return RSA_SIGN;
        case 2: return DSS_SIGN;
        case 3: return RSA_FIXED_DH;
        case 4: return DSS_FIXED_DH;
        default: return new ClientType(i);
        }
    }

    // Instance methods.
    // -----------------------------------------------------------------------

    public byte[] getEncoded()
    {
      return new byte[] { (byte) value };
    }

    public int getValue()
    {
      return value;
    }

    public String toString()
    {
      switch (value)
        {
        case 1: return "rsa_sign";
        case 2: return "dss_sign";
        case 3: return "rsa_fixed_dh";
        case 4: return "dss_fixed_dh";
        default: return "unknown(" + value + ")";
        }
    }
  }
}
