/* SSLSocketInputStream.java -- InputStream for SSL sockets.
   Copyright (C) 2003  Casey Marshall <rsdio@metastatic.org>

This file is a part of Jessie.

Jessie is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2 of the License, or (at your
option) any later version.

Jessie is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with Jessie; if not, write to the

   Free Software Foundation, Inc.,
   59 Temple Place, Suite 330,
   Boston, MA  02111-1307
   USA  */


package org.metastatic.jessie.provider;

import java.io.EOFException;
import java.io.FilterInputStream;
import java.io.InputStream;
import java.io.IOException;
import javax.net.ssl.SSLException;

class SSLSocketInputStream extends FilterInputStream
{

  // Fields.
  // -------------------------------------------------------------------------

  private final SSLSocket socket;
  private final boolean checkHandshake;

  // Constructors.
  // -------------------------------------------------------------------------

  SSLSocketInputStream(InputStream in, SSLSocket socket)
  {
    this(in, socket, true);
  }

  SSLSocketInputStream(InputStream in, SSLSocket socket, boolean checkHandshake)
  {
    super(in);
    this.socket = socket;
    this.checkHandshake = checkHandshake;
  }

  // Instance methods.
  // -------------------------------------------------------------------------

  public int available() throws IOException
  {
    if (checkHandshake)
      socket.checkHandshakeDone();
    if (checkAlert())
      return -1;
    int ret = super.available();
    checkAlert();
    return ret;
  }

  public int read() throws IOException
  {
    if (checkHandshake)
      socket.checkHandshakeDone();
    if (checkAlert() && in.available() <= 0)
      return -1;
    int ret = in.read();
    checkAlert();
    return ret;
  }

  public int read(byte[] buf) throws IOException
  {
    return read(buf, 0, buf.length);
  }

  public int read(byte[] buf, int off, int len) throws IOException
  {
    if (checkHandshake)
      socket.checkHandshakeDone();
    if (buf == null)
      throw new NullPointerException();
    if (off < 0 || len < 0 || off + len > buf.length)
      throw new ArrayIndexOutOfBoundsException();
    if (checkAlert() && in.available() <= 0)
      return -1;
    int ret = in.read(buf, off, len);
    checkAlert();
    return ret;
  }

  // Own methods.
  // -------------------------------------------------------------------------

  private boolean checkAlert() throws IOException
  {
    Alert alert = socket.checkAlert();
    if (alert == null) return false;
    if (alert.getLevel().equals(Alert.Level.FATAL))
      throw new SSLException(alert.getDescription().toString());
    if (alert.getDescription().equals(Alert.Description.CLOSE_NOTIFY))
      {
        try { return (in.available() <= 0); }
        catch (IOException ioe) { }
      }
    return false;
  }
}
