/* Text.java -- generic Plain/Compressed/Ciphertext.
   Copyright (C) 2003  Casey Marshall <rsdio@metastatic.org>

This file is a part of Jessie.

Jessie is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2 of the License, or (at your
option) any later version.

Jessie is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with Jessie; if not, write to the

   Free Software Foundation, Inc.,
   59 Temple Place, Suite 330,
   Boston, MA  02111-1307
   USA  */


package org.metastatic.jessie.provider;

import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.EOFException;
import java.io.InputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.io.StringWriter;

import javax.net.ssl.SSLProtocolException;

class Text implements Constructed
{

  // Fields.
  // -------------------------------------------------------------------------

  private ContentType type;
  private ProtocolVersion version;
  private int length;
  private byte[] fragment;

  // Constructors.
  // -------------------------------------------------------------------------

  Text(ContentType type, ProtocolVersion version, byte[] fragment)
  {
    this.type = type;
    this.version = version;
    length = fragment.length;
    this.fragment = fragment;
  }

  Text(int capacity)
  {
    this.fragment = new byte[capacity];
  }

  // Instance methods.
  // -------------------------------------------------------------------------

  /**
   * Reads (decodes) a text into this object.
   *
   * @param in The input stream.
   * @throws IOException If an I/O error occurs.
   * @throws OverflowException If text's fragment length exceeds the maximum
   *   fragment length for this text.
   */
  void read(InputStream in) throws IOException
  {
    type = ContentType.read(in);
    // If the high bit in this byte is 1, probably a V2 hello.
    if ((type.getValue() & 0x80) != 0 || (type.getValue() & 0x40) != 0)
      {
        in.read();
        if ((type.getValue() & 0x40) != 0)
          {
            in.read();
          }
        type = ContentType.read(in);
        if (type != ContentType.CLIENT_HELLO_V2)
          {
            throw new SSLProtocolException("unsupported V2 message");
          }
        type = ContentType.HANDSHAKE;
        // Record this message, and re-present it as a normal handshake
        // layer message. ClientHello will handle the real parsing.
        ByteArrayOutputStream buffer = new ByteArrayOutputStream(256);
        buffer.write(1); // The type we just read.
        in = new RecordingInputStream(in, buffer);
        version = ProtocolVersion.read(in);
        if (version.compareTo(ProtocolVersion.SSL_3) < 0)
          {
            throw new SSLProtocolException("unsupported client version");
          }
        int len = (in.read() & 0xFF) << 8 | (in.read() & 0xFF);
        len += (in.read() & 0xFF) << 8 | (in.read() & 0xFF);
        len += (in.read() & 0xFF) << 8 | (in.read() & 0xFF);
        int count = 0;
        while (count < len)
          {
            int l = (int) in.skip(len - count);
            if (l > 0)
              {
                count += l;
              }
          }
        length = buffer.size();
        System.arraycopy(buffer.toByteArray(), 0, fragment, 0, length);
        return;
      }
    version = ProtocolVersion.read(in);
    DataInputStream din = new DataInputStream(in);
    length = din.readUnsignedShort();
    if (length > fragment.length)
      {
        throw new OverflowException(String.valueOf(length));
      }
    din.readFully(fragment, 0, length);
  }

  /**
   * Writes (encodes) this text to the given output stream.
   *
   * @param out The output stream.
   * @throws IOException If an I/O error occurs.
   */
  public void write(OutputStream out) throws IOException
  {
    out.write(type.getValue());
    version.write(out);
    out.write((length >>> 8) & 0xFF);
    out.write(length & 0xFF);
    out.write(fragment, 0, length);
  }

  /**
   * Returns this text's content type.
   *
   * @return The content type field.
   */
  ContentType getType()
  {
    return type;
  }

  /**
   * Sets this text's content type.
   *
   * @param type The new content type.
   */
  void setType(ContentType type)
  {
    this.type = type;
  }

  /**
   * Returns this text's version field.
   *
   * @return The version field.
   */
  ProtocolVersion getVersion()
  {
    return version;
  }

  /**
   * Sets this text's version field.
   *
   * @param version The new version.
   */
  void setVersion(ProtocolVersion version)
  {
    this.version = version;
  }

  /**
   * Returns this text's fragment length.
   *
   * @return The fragment length.
   */
  int getLength()
  {
    return length;
  }

  /**
   * Sets this text's fragment length.
   *
   * @param length The new fragment length.
   */
  void setLength(int length)
  {
    this.length = length;
  }

  /**
   * Returns this text's fragment field. The number of valid bytes in the
   * returned array is the value returned by {@link #getLength()}.
   *
   * @return The fragment.
   */
  byte[] getFragment()
  {
    return fragment;
  }

  /**
   * Returns the capacity of this text's fragment, which is the maximum
   * fragment length.
   *
   * @return The capacity.
   */
  int getCapacity()
  {
    return fragment.length;
  }

  /**
   * Sets this text's capacity.
   *
   * @param capacity The new capacity.
   */
  void setCapacity(int capacity)
  {
    fragment = new byte[capacity];
  }

  public String toString()
  {
    StringWriter str = new StringWriter();
    PrintWriter out = new PrintWriter(str);
    out.println("struct {");
    out.println("  type = " + type + ";");
    out.println("  version = " + version + ";");
    out.println("  length = " + length + ";");
    out.println("  fragment =");
    out.print(Util.hexDump(fragment, 0, length, "  "));
    out.println("} Text;");
    return str.toString();
  }
}
