//////////////////////////////////////////////////////////////////////////////
//
//    KEXIF.CPP
//
//    Copyright (C) 2002-2004 Renchi Raju <renchi at pooh.tam.uiuc.edu>
//                            Gilles Caulier <caulier dot gilles at free.fr>
//                            Ralf Hoelzer <kde at ralfhoelzer.com>
//
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
//////////////////////////////////////////////////////////////////////////////

// Qt includes.

#include <qtabwidget.h>
#include <qlayout.h>
#include <qstring.h>
#include <qstringlist.h>
#include <qimage.h>
#include <qpixmap.h>
#include <qlabel.h>
#include <qgroupbox.h>
#include <qtextedit.h>
#include <qbuttongroup.h>
#include <qpushbutton.h>

// KDE includes.

#include <klocale.h>
#include <klistview.h>
#include <kurl.h>
#include <kmessagebox.h>
#include <kdebug.h>
#include <kaboutdata.h>
#include <khelpmenu.h>
#include <kiconloader.h>
#include <kpopupmenu.h>
#include <kapplication.h>

// Local includes.

#include "kexififd.h"
#include "kexifentry.h"
#include "kexifdata.h"
#include "kexiflistview.h"
#include "kexiflistviewitem.h"
#include "kexif.h"


KExif::KExif(QWidget *parent, const char *name)
     : KDialogBase( Plain, i18n("Exif Information"), Help|Close,
                    Close, parent, name, true, true )
{
    // About data and help button.
    
    KAboutData* about = new KAboutData("digikam",                       // All bugs must be reported to Digikam account in bko.
                                       I18N_NOOP("KExif"), 
                                       "0.1",  
                                       I18N_NOOP("An Exif data viewer"),
                                       KAboutData::License_GPL,
                                       "(c) 2003-2004, Digikam team", 
                                       0,
                                       "http://digikam.sourceforge.net");
    
    about->addAuthor("Renchi Raju", I18N_NOOP("Main developer"),
                     "renchi@pooh.tam.uiuc.edu");

    about->addAuthor("Gilles Caulier", I18N_NOOP("developer"),
                     "caulier dot gilles at free.fr");
    
    about->addAuthor("Ralf Hoelzer", I18N_NOOP("developer"),
                     "kde at ralfhoelzer.com");

    mhelpButton = actionButton( Help );
    KHelpMenu* helpMenu = new KHelpMenu(this, about, false);
    helpMenu->menu()->removeItemAt(0);
    helpMenu->menu()->insertItem(i18n("Digikam Handbook"), this, SLOT(slotHelp()), 0, -1, 0);
    mhelpButton->setPopup( helpMenu->menu() );

    // ------------------------------------------------------

    QGridLayout* layout = new QGridLayout(plainPage());
    layout->setSpacing(6);
    layout->setMargin(6);

    // ------------------------------------------------------

    mMainBox = new QGroupBox(1, Qt::Vertical, plainPage());
    
    mTabWidget = new QTabWidget(mMainBox);

    layout->addWidget(mMainBox, 0, 0);
    
    connect(mTabWidget, SIGNAL(currentChanged(QWidget*)),
            this, SLOT(slotTabPageChanged(QWidget*)));

    // ------------------------------------------------------

    QGroupBox *textBox = new QGroupBox(1, Qt::Vertical, plainPage());
    
    mTextEdit = new QTextEdit(textBox);
    mTextEdit->setReadOnly(true);
    mTextEdit->setSizePolicy(QSizePolicy(QSizePolicy::Expanding,
                                         QSizePolicy::Minimum));
    textBox->setSizePolicy(QSizePolicy(QSizePolicy::Expanding,
                                       QSizePolicy::Minimum));
    
    layout->addWidget(textBox, 1, 0);

    // ------------------------------------------------------
    
    mExifData = 0;
    resize(500, 500);
}

KExif::~KExif()
{
    if (mExifData)
        delete mExifData;
}

void KExif::slotHelp()
{
    KApplication::kApplication()->invokeHelp("kexif.anchor",
                                             "digikam");       // We using the Digikam handbook with kexif section.
}

bool KExif::loadFile(const QString& filename)
{
    if (filename.isNull())
        return false;
    
    mFileName = filename;

    mMainBox->setTitle(KURL(filename).filename());

    if (mExifData) 
    {
        delete mExifData;
        mExifData = 0;
    }

    mExifData = new KExifData;

    if (mExifData->readFromFile(filename))
    {
        buildView();
        return true;
    }

    return false;
}

bool KExif::loadData(const QString& filename, char *data, int size)
{
    if (data == 0 || size == 0)
    {
        kdWarning() << "KEXIF:: Invalid data" << endl;
        return false;
    }
    mMainBox->setTitle(filename);

    if (mExifData) {
        delete mExifData;
        mExifData = 0;
    }

    mExifData = new KExifData;

    if (mExifData->readFromData(data, size))
    {
        buildView();
        return true;
    }

    
    kdWarning() << "Failed to load exif from data" << endl;
    return false;
    
}

void KExif::buildView()
{
    QPtrList<KExifIfd> ifdList(mExifData->ifdList());

    for (KExifIfd* ifd = ifdList.first(); ifd; ifd = ifdList.next())
    {
        KExifListView* listview = new KExifListView(mTabWidget);
        listview->addItems(ifd->entryList());
        mTabWidget->addTab(listview, ifd->getName());

        connect(listview, SIGNAL(signal_itemDescription(const QString&)),
                this, SLOT(slotShowItemDescription(const QString&)));
    }

    QImage thumbnail(mExifData->getThumbnail());
        
    if (!thumbnail.isNull()) 
    {
        QWidget* widget = new QWidget(mTabWidget);
        mTabWidget->addTab(widget,i18n("Thumbnail"));
        QGridLayout* layout = new QGridLayout(widget);
        QLabel* label = new QLabel(widget);
        label->setFixedSize(thumbnail.size());
        label->setPixmap(QPixmap(thumbnail));
        layout->addWidget(label, 0, 0);
    }
}

void KExif::slotShowItemDescription(const QString& desc)
{
    if (desc.isEmpty())
        mTextEdit->setText(i18n("Select an item to see its description"));
    else
        mTextEdit->setText(desc);
}

void KExif::slotTabPageChanged(QWidget*)
{
    slotShowItemDescription(QString(""));
}


#include "kexif.moc"
