//
// trainer.cc
//
// Made by Guillaume Stordeur
// Login   <kami@GrayArea.Masaq>
//
// Started on  Fri Aug  2 05:53:59 2002 Guillaume Stordeur
// Last update Thu May  1 17:05:36 2003 Guillaume Stordeur
//

#include <stdlib.h>
#include <iostream>
#include <vector>
#include <getopt.h>
#include "timelaggednetwork.hh"
#include <exception/exception.hh>
#include <data/csvloader.hh>

enum options_e
  {
    opt_help = 'h',
    opt_version = 'v',
    opt_maxiter = 'm',
    opt_changeiter = 'c',
    opt_epsilon = 'e',
    opt_lrate = 'l',
    opt_fudge = 'f',
    opt_momentum = 'o',
    opt_batch = 'b',
    opt_algorithm = 'a',
    opt_nplus = 'n',
    opt_nminus = 'i',
    opt_deltamin = 'd',
    opt_deltamax = 't',
    opt_mu = 'u',
    opt_wdecay = 'y',
    opt_errorfunc = 'r',
    opt_pca = 'p'
  };

static struct option const long_options[] =
  {
    {"help",		0,		0,	opt_help},
    {"version",		0,		0,	opt_version},
    {"maxiter",		1,		0,	opt_maxiter},
    {"changeiter",	1,		0,	opt_changeiter},
    {"epsilon",		1,		0,	opt_epsilon},
    {"lrate",		1,		0,	opt_lrate},
    {"fudge",		1,		0,	opt_fudge},
    {"momentum",	1,		0,	opt_momentum},
    {"batch",		1,		0,	opt_batch},
    {"pca",		1,		0,	opt_pca},
    {"algorithm",	1,		0,	opt_algorithm},
    {"nplus",		1,		0,	opt_nplus},
    {"nminus",		1,		0,	opt_nminus},
    {"deltamin",	1,		0,	opt_deltamin},
    {"deltamax",	1,		0,	opt_deltamax},
    {"errorfunc",	1,		0,	opt_errorfunc},
    {"mu",		1,		0,	opt_mu},
    {"wdecay",		1,		0,	opt_wdecay},
    {NULL,		0,		NULL,	0}
  };

static const char short_options[] =  "hvp:m:c:e:l:f:o:b:a:n:i:d:t:r:y:";

void	help(char **argv)
{
  std::cout << "Usage: " << argv[0]
	    << " [options] network.nn data.csv separator outputColumn [outputColumn2, ...]" << std::endl;

  std::cout << "Options:" << std::endl
	    << "  -h, --help\t\tDisplay this help and exit" << std::endl
	    << "  -v, --version\t\tOutput version information and exit" << std::endl;

  std::cout << " Training halt criteria:" << std::endl
	    << "  -m, --maxiter\t\tStop training after this many iterations [default is 10000]" << std::endl
	    << "  -c, --changeiter\tStop training after this many iterations without error decrease [default is 0; disabled]" << std::endl
	    << "  -e, --epsilon\t\tStop training when total error is <= epsilon [default is 0.001]" << std::endl;

  std::cout << " Training algorithm global parameters:" << std::endl
	    << "  -p, --pca <algo>\tInit weights with a PCA algorithm : gha or ala" << std::endl
	    << "  -a, --algorithm\tLearning algorithm: NONE, BACKPROP, QUICKPROP or RPROP [default is BACKPROP]" << std::endl
	    << "  -b, --batch\t\tLearning is done via batch mode, 0 for stochastic [default is 1]" << std::endl
	    << "  -f, --fudge\t\tAdded to derivative of activation function [default is 0]" << std::endl
	    << "  -r, --errorfunc\tError function: DIFF, SCALED_DIFF, TRIMMED_DIFF, ATANH [default is DIFF]" << std::endl
	    << "  -y, --wdecay\t\tWeight decay used in batch mode [default is 0.0]" << std::endl;

  std::cout << " Training algorithm local parameters:" << std::endl
    	    << "  -l, --lrate\t\tLearning rate constant used in BACKPROP [default is 0.6]" << std::endl
	    << "  -o, --momentum\tMomentum constant used in BACKPROP [default is 0.7]" << std::endl
  	    << "  -n, --nplus\t\tDelta increase constant used in RPROP [default is 1.2]" << std::endl
	    << "  -i, --nminus\t\tDelta decrease constant used in RPROP [default is 0.5]" << std::endl
    	    << "  -d, --deltamin\tMinimum delta used in RPROP [default is 0.00001]" << std::endl
    	    << "  -t, --deltamax\tMaximum delta used in RPROP [default is 50]" << std::endl
      	    << "  -u, --mu\t\tMax factor used in QUICKPROP [default is 1.75]" << std::endl;

  exit(1);
}

void	ver(char **argv)
{
  std::cout << "Version: " << __NN_VERSION << std::endl;
  std::cout << "Authors: " << __NN_AUTHORS << std::endl;
  exit(1);
}

void	parse_args(float &lrate, float &momentum, float &nplus,
		   float &nminus, float &deltamin, float &deltamax,
		   float &epsilon, float &fudge, float &mu, float &wdecay,
		   NeuralNet::TrainingAlgorithm &algorithm, int &batch,
		   int &maxiter, int &changeiter, NeuralNet::ErrorFunctionType &errorfunc,
		   std::string& pca, int argc, char **argv)
{
  int idx, c;
  while ((c = getopt_long(argc, argv, short_options, long_options, &idx)) != -1)
    {
      switch (c)
	{
	case opt_deltamin:
	  deltamin = strtof(optarg, 0);
	  break;
	case opt_deltamax:
	  deltamax = strtof(optarg, 0);
	  break;
	case opt_nminus:
	  nminus = strtof(optarg, 0);
	  break;
	case opt_nplus:
	  nplus = strtof(optarg, 0);
	  break;
	case opt_wdecay:
	  wdecay = strtof(optarg, 0);
	  break;
	case opt_pca:
	  pca = std::string(optarg);
	  break;
	case opt_fudge:
	  fudge = strtof(optarg, 0);
	  break;
	case opt_mu:
	  mu = strtof(optarg, 0);
	  break;
	case opt_momentum:
	  momentum = strtof(optarg, 0);
	  break;
	case opt_lrate:
	  lrate = strtof(optarg, 0);
	  break;
	case opt_batch:
	  batch = strtol(optarg, 0, 10);
	  break;
	case opt_errorfunc:
	  if (!strcmp(optarg, "DIFF"))
	    errorfunc = NeuralNet::ERR_DIFF;
	  else if (!strcmp(optarg, "TRIMMED_DIFF"))
	    errorfunc = NeuralNet::ERR_TRIMMED_DIFF;
	  else if (!strcmp(optarg, "SCALED_DIFF"))
	    errorfunc = NeuralNet::ERR_SCALED_DIFF;
	  else if (!strcmp(optarg, "ATANH"))
	    errorfunc = NeuralNet::ERR_ATANH;
	  else
	    {
	      std::cerr << "Unsupported error function. Please use DIFF, TRIMMED_DIFF, or ATANH."
			<< std::endl;
	      exit(1);
	    }
	  break;
	case opt_algorithm:
	  if (!strcmp(optarg, "BACKPROP"))
	    algorithm = NeuralNet::ALG_BACKPROP;
	  else if (!strcmp(optarg, "RPROP"))
	    algorithm = NeuralNet::ALG_RPROP;
	  else if (!strcmp(optarg, "NONE"))
	    algorithm = NeuralNet::ALG_NONE;
	  else if (!strcmp(optarg, "QUICKPROP"))
	    algorithm = NeuralNet::ALG_QUICKPROP;
	  else
	    {
	      std::cerr << "Unsupported algorithm. Please use NONE, BACKPROP, QUICKPROP or RPROP."
			<< std::endl;
	      exit(1);
	    }
	  break;
	case opt_epsilon:
	  epsilon = strtof(optarg, 0);
	  break;
	case opt_changeiter:
	  changeiter = strtol(optarg, 0, 10);
	  break;
	case opt_maxiter:
	  maxiter = strtol(optarg, 0, 10);
	  break;
	case opt_help:
	  help(argv);
	  break;
	case opt_version:
	  ver(argv);
	  break;
	default:
	  //help(argv);
	  break;
	}
    }
}



//
// Main
//
int	main(int argc, char *argv[])
{
  float lrate = 0.6, momentum = 0.7, nplus = 1.2, nminus = 0.5, wdecay = 0.0,
    deltamin = 0.00001, deltamax = 50, epsilon = 0.001, fudge = 0, mu = 1.75;
  int batch = 1, maxiter = 10000, changeiter = 0;
  NeuralNet::ErrorFunctionType errorfunc = NeuralNet::ERR_DIFF;
  NeuralNet::TrainingAlgorithm algorithm = NeuralNet::ALG_BACKPROP;
  std::string csvfile, nnfile, pca;
  char separator;
  std::vector<std::string> outputCols;

  srand(time(NULL));

  try
    {
      //       // Parse command line
      parse_args(lrate, momentum, nplus, nminus, deltamin,
		 deltamax, epsilon, fudge, mu, wdecay, algorithm, batch, maxiter,
 		 changeiter, errorfunc, pca, argc, argv);

      //       // Remaining are "nnfile, csvfile, separator, outputCols"
      //       if (argc - optind < 4)
      // 	help(argv);
      //       else
      // 	{
      // 	  nnfile = std::string(argv[optind++]);
      // 	  csvfile = std::string(argv[optind++]);
      // 	  separator = argv[optind++][0];
      // 	  while (optind < argc)
      // 	    outputCols.push_back(argv[optind++]);
      // 	}

      //Init cvsloader and get cols
      Data::CsvLoader	loader;
      std::vector<Data::Column*>	cols;
      csvfile = "./test.csv";
      separator = ';';
      cols = loader.read_file(csvfile, separator);

      // Init neural net
      // Should not init over the file, or should be a new file
      // type...

      std::vector<unsigned> nb;
      nb.push_back(37); nb.push_back(3); nb.push_back(36);
      std::vector<unsigned> winsize;
      winsize.push_back(7);

      NeuralNet::TimeLaggedNeuralNet	nn(nb, winsize,
					   NeuralNet::ACT_SIGMOID,
					   NeuralNet::ACT_SIGMOID);

      nn.setLRate(lrate);
      nn.setMomentum(momentum);
      nn.setDeltaMin(deltamin);
      nn.setDeltaMax(deltamax);
      nn.setNPlus(nplus);
      nn.setNMinus(nminus);
      nn.setMaxIterations(maxiter);
      nn.setChangeIterations(changeiter);
      nn.setErrorEpsilon(epsilon);
      nn.setFudge(fudge);
      nn.setMu(mu);
      nn.setWDecay(wdecay);
      nn.setErrorFunction(errorfunc);

      // Should Check that the data correspond to our neural network


      // Make Training Pattern (inputs, results)
      std::vector<float> dres, input;
      std::vector<std::vector<float> > inputs, results;

      assert(cols[0]->get_type() == Data::FLOAT);
      for (unsigned int j = 0; j < cols[0]->get_size(); j++)
	if (cols[0]->get_type() == Data::FLOAT)
	  {
	    input.push_back((float)(dynamic_cast<Data::ColumnTyped<float>*> (cols[0]))->get_value(j));
	  }

      inputs.push_back(input);
      // Learn the Training Pattern
      if (algorithm != NeuralNet::ALG_NONE)
	nn.trainNetwork(inputs, batch, algorithm);

      // plot "output.gplot" with lines axes x1y1, "desired.gplot" with lines axes x1y2
      std::ofstream f("output.gplot");
      // to have the lag of 5 ...
      for (unsigned i = 0; i < winsize[0]; i++)
	f << 0 << std::endl;
      for (unsigned i = 0; i < input.size(); i++)
	f << nn.outputlag(input, i)[0] << std::endl;
    }
  catch (Exception::Exception e)
    {
      std::cerr << e;
      return 1;
    }
  return 0;
}
