(**
  Handles allocating of textblocks and text within textblocks.
**)

MODULE VO:EditBlock;

CONST
  blockLen * = 8192; (* Must not be greater than MAX(LONGINT) *)

TYPE
  Text*      = POINTER TO TextDesc;

  (**
    The text within a block.
  **)

  TextDesc*  = ARRAY blockLen OF CHAR;


  Block*     = POINTER TO BlockDesc;


  (**
    A block of texts. Runs point to an area within a block.
  **)

  BlockDesc* = RECORD
                 text* : TextDesc; (* The text        *)
                 size* : LONGINT;  (* Used space      *)
               END;

VAR
  block-      : Block;  (* current textstore-buffer *)


  (*****************************************
   * Buffer handling routines
   *****************************************)


  (**
    Allocates a new block.
  **)

  PROCEDURE GetNewBlock();

  BEGIN
    NEW(block);
    block.size:=0;
  END GetNewBlock;

  (**
    Allocates a new block with the given size. It is garanteed
    that block points to a block for text with at least the given size.

    NOTE
    The size given must no exceed the maximum size of a block.
  **)

  PROCEDURE GetBlockWithSize*(size : LONGINT);

  BEGIN
    ASSERT(size<=blockLen);

    IF block=NIL THEN
      GetNewBlock;
    ELSIF size>blockLen-block.size THEN
      GetNewBlock;
    END;
  END GetBlockWithSize;

  (**
    Stores the text of the given size into the current block.

    RESULT
    Returns the starting position of the copied text within the block.

    NOTE
    Size must not exceed the maximum size of a block.
  **)

  PROCEDURE StoreInBlock*(text : ARRAY OF CHAR; size : LONGINT):LONGINT;

  VAR x,y,result : LONGINT;

  BEGIN
    ASSERT(size<=blockLen);

    GetBlockWithSize(size);

    (* Text copy, could be optimized *)
    y:=block.size;
    result:=block.size;
    FOR x:=0 TO size-1 DO
      block.text[y]:=text[x];
      INC(y);
    END;

    INC(block.size,size);
    RETURN result;
  END StoreInBlock;

  (**
    Stores the given character count times in the current block.

    RESULT
    Returns the starting position of the copied text within the block.

    NOTE
    Size must not exceed the maximum size of a block.
  **)

  PROCEDURE StoreCharsInBlock*(char : CHAR; count : LONGINT):LONGINT;

  VAR x,y,result : LONGINT;

  BEGIN
    ASSERT(count<=blockLen);

    GetBlockWithSize(count);

    (* Text copy, could be optimized *)
    y:=block.size;
    result:=block.size;
    FOR x:=0 TO count-1 DO
      block.text[y]:=char;
      INC(y);
    END;

    INC(block.size,count);
    RETURN result;
  END StoreCharsInBlock;

  (**
    Stores the given character in the current block.

    RESULT
    Returns the starting position of the copied text within the block.

    NOTE
    Size must not exceed the maximum size of a block.
  **)

  PROCEDURE StoreCharInBlock*(char : CHAR):LONGINT;

  VAR result : LONGINT;

  BEGIN
    GetBlockWithSize(1);

    result:=block.size;
    block.text[block.size]:=char;

    INC(block.size);
    RETURN result;
  END StoreCharInBlock;

END VO:EditBlock.