MODULE VO:EditRun;

IMPORT B   := VO:EditBlock,

              Err,
       str := Strings;

CONST
  (* mark types *)

  mark*   = 0;
  cursor* = 1;
  error*  = 2;
  block*  = 3;

TYPE
  Run*         = POINTER TO RunDesc;

  (**
    A text consists of Runs. A Run represents a certain amount of
    characters within a text or special logical information.
    Because of the editor implementation Runs are never longer
    than a line. Run itself is only a abstract baseclass.
    There are a number of special Runs. Since the base allready
    knows of LineRuns the baseclass supports moving through text.
  **)

  RunDesc*     = RECORD
                   next*,              (* Pointer to the next Run *)
                   last*   : Run;      (* Pointer to the last Run *)
                 END;


  TextRun*     = POINTER TO TextRunDesc;

  (**
    An amount of text, not longer than one line.
  **)

  TextRunDesc* = RECORD (RunDesc)
                   block*  : B.Block;  (* Pointer to the text block           *)
                   pos*,               (* Position in the block 0..blockLen-1 *)
                   length* : LONGINT;  (* Length of the in block              *)
                 END;


  LineRun*     = POINTER TO LineRunDesc;


  LineEntry*   = POINTER TO LineEntryDesc;


  (**
    LineEntry give ou the possibility to store additional information
    into the line.

    Usefull for cursor positions, tokens found in the line etc...
  **)

  LineEntryDesc* = RECORD
                     next-,
                     last- : LineEntry;
                     line- : LineRun;
                     x*    : LONGINT;
                   END;

  (**
    A LineRun marks the start of a line.
  **)

  LineRunDesc*   = RECORD (RunDesc)
                     first*      : LineEntry;
                     parseState* : LONGINT;
                   END;

  Mark*     = POINTER TO MarkDesc;

  (**
    A Mark. One can insert a unlimited number of marks into the text.

    While the mark has an x- and y-position the text is responsible
    for updating them when the text gets changd.

    The member methods of the MarkRun will update them when they move the
    mark through the list of Runs.

    A mark can have an optional name.
  **)

  MarkDesc* = RECORD (LineEntryDesc)
                name-     : POINTER TO ARRAY OF CHAR; (* The name of the mark      *)
                y*        : LONGINT;
                type-     : LONGINT;
              END;


(* ------------------------------------------------------------------------ *)

  PROCEDURE (r : Run) Init*;

  BEGIN
   r.last:=NIL;
   r.next:=NIL;
  END Init;

  (**
    Returns the Run wich reflext the start of the last line.
  **)

  PROCEDURE (r : Run) LastLine*():LineRun;

  VAR
    run : Run;

  BEGIN
    run:=r;

    (* Search for start of current line *)

    WHILE ~(run IS LineRun) DO
      run:=run.last;
    END;

    (* Search for start of current line *)

    run:=run.last;
    WHILE (run#NIL) & ~(run IS LineRun) DO
      run:=run.last;
    END;

    IF run#NIL THEN
      RETURN run(LineRun);
    ELSE
      RETURN NIL;
    END;
  END LastLine;

  (**
    Returns the Run wich reflext the start of the next line.
  **)

  PROCEDURE (r : Run) NextLine*():LineRun;

  VAR
    run : Run;

  BEGIN
    run:=r.next;
    WHILE (run#NIL) & ~(run IS LineRun) DO
      run:=run.next;
    END;

    IF run#NIL THEN
      RETURN run(LineRun);
    ELSE
      RETURN NIL;
    END;
  END NextLine;

  (**
    Returns the start of the current line.
  **)

  PROCEDURE (r : Run) ThisLine*():LineRun;

  VAR
    run : Run;

  BEGIN
    run:=r;

    (* Search for start of current line *)

    WHILE ~(run IS LineRun) DO
      run:=run.last;
    END;

    IF run#NIL THEN
      RETURN run(LineRun);
    ELSE
      RETURN NIL;
    END;
  END ThisLine;

  (**
    Returns true, if the Run is part of the first line.
  **)

  PROCEDURE (r : Run) IsFirstLine*():BOOLEAN;

  BEGIN
    RETURN r.LastLine()=NIL;
  END IsFirstLine;

  (**
    Returns true, if the Run is part of the last line.
  **)

  PROCEDURE (r : Run) IsLastLine*():BOOLEAN;

  BEGIN
    RETURN r.NextLine()=NIL;
  END IsLastLine;

  (**
    Insert the given Run after the current one.
  **)

  PROCEDURE (r : Run) InsertAfter*(new : Run);

  BEGIN
    IF r.next#NIL THEN
      r.next.last:=new;
      new.next:=r.next;
    ELSE
      new.next:=NIL;
    END;
    r.next:=new;
    new.last:=r;
  END InsertAfter;

  (**
    Insert the given Run before the current one.
  **)

  PROCEDURE (r : Run) InsertBefore*(new : Run);

  BEGIN
    IF r.last#NIL THEN
      r.last.next:=new;
      new.last:=r;
    ELSE
      new.last:=NIL;
    END;
    r.last:=new;
    new.next:=r;
  END InsertBefore;

  (**
    Removes the Run from the list.
  **)

  PROCEDURE (r : Run) Remove*;

  BEGIN
    IF r.last#NIL THEN
      r.last.next:=r.next;
    END;

    IF r.next#NIL THEN
      r.next.last:=r.last;
    END;
  END Remove;

  (**
    Makes the given Run follower of the current run.
  **)

  PROCEDURE (r : Run) Join*(run : Run);

  BEGIN
    r.next:=run;
    IF run#NIL THEN
      run.last:=r;
    END;
  END Join;

  (**
    Create an exact copy of the given run. Run inheriting from Run must
    overload this method. Note however, that the method needs not to copy
    aditional member like info or first member. It can even reset them to NIL.
  **)

  PROCEDURE (r : Run) Copy*():Run;

  VAR
    run : Run;

  BEGIN
    NEW(run);
    run^:=r^;

    RETURN run;
  END Copy;

  (**
    A Run can be printed for debugging purposes.
    This is a abstract method, which should be overloaded for each special
    Run type.
  **)

  PROCEDURE (r : Run) Print*;

  BEGIN

  END Print;

  (* -------- Text runs ----------------------- *)

  PROCEDURE (t : TextRun) Init*;

  BEGIN
    t.Init^;

    t.block:=NIL;
    t.pos:=-1;
    t.length:=-1;
  END Init;

  (**
    Splits the given Run at the given position into two
    text Runs.

    RESULT
    It is possible that splitting a Run changes the run,
    (this can happen, when you split at position 0)
    so the (possible) new run is returned and you should
    never access the old one after splitting.

    NOTE
    The text will also be split when one of the resulting
    TextRuns does not contain any text. One of the runs
    will be empty then.
  **)

  PROCEDURE (t : TextRun) Split*(pos : LONGINT):TextRun;

  VAR
    new : TextRun;

  BEGIN
    DEC(pos,t.pos);
    IF pos=0 THEN
      NEW(new);
      new^:=t^;
      t.InsertAfter(new);
      t.pos:=-1;
      t.length:=0;
    ELSIF pos>t.pos+t.length-1 THEN
      NEW(new);
      new.Init;
      new.pos:=-1;
      new.length:=0;
      t.InsertAfter(new);
    ELSE
      NEW(new);
      new.Init;
      new.block:=t.block;
      new.pos:=t.pos+pos;
      new.length:=t.length-pos;

      DEC(t.length,new.length);

      t.InsertAfter(new);
    END;

    RETURN t;
  END Split;

  PROCEDURE (t : TextRun) LastChar(VAR pos : LONGINT):TextRun;

  VAR
    run : Run;

  BEGIN
    IF pos>t.pos THEN
      DEC(pos);
      RETURN t;
    ELSE
      run:=t.last;
      WHILE (run#NIL) & (~(run IS TextRun) OR (run(TextRun).length=0)) DO
        run:=run.last;
      END;
      IF (run#NIL) & (run IS TextRun) THEN
        pos:=run(TextRun).pos+run(TextRun).length-1;
        RETURN run(TextRun);
      ELSE
        RETURN NIL;
      END;
    END;
  END LastChar;


  PROCEDURE (t : TextRun) NextChar(VAR pos : LONGINT):TextRun;

  VAR
    run : Run;

  BEGIN
    IF pos<t.pos+t.length-1 THEN
      INC(pos);
      RETURN t;
    ELSE
      run:=t.next;
      WHILE (run#NIL) & (~(run IS TextRun) OR (run(TextRun).length=0)) DO
        run:=run.next;
      END;
      IF (run#NIL) & (run IS TextRun) THEN
        pos:=run(TextRun).pos;
        RETURN run(TextRun);
      ELSE
        RETURN NIL;
      END;
    END;
  END NextChar;

  PROCEDURE (t : TextRun) Match*(pos : LONGINT; text : ARRAY OF CHAR;
                                 VAR end : TextRun; VAR endPos : LONGINT):BOOLEAN;

  VAR
    z : LONGINT;

  BEGIN
    end:=t;
    z:=0;
    endPos:=pos;

    LOOP
      IF text[z]=0X THEN
        end:=end.LastChar(endPos);
        RETURN TRUE;
      END;

      IF text[z]#end.block.text[endPos] THEN
        RETURN FALSE;
      END;

      INC(z);
      end:=end.NextChar(endPos);
    END;
  END Match;

  (**
    Return a copy of the given TextRun.
  **)

  PROCEDURE (t : TextRun) Copy*():Run;

  VAR
    run : TextRun;

  BEGIN
    NEW(run);
    run^:=t^;

    RETURN run;
  END Copy;

  (**
    This function is only for debugging purposes.
  **)

  PROCEDURE (t : TextRun) Print*;

  VAR
    x : LONGINT;

  BEGIN
    FOR x:=t.pos TO t.pos+t.length-1 DO
      Err.Char(t.block.text[x]);
    END;
  END Print;

  PROCEDURE CreateTextRunChar*(char : CHAR):TextRun;

  VAR
    text : TextRun;

  BEGIN
    NEW(text);
    text.Init;
    text.pos:=B.StoreCharInBlock(char);
    text.block:=B.block;
    text.length:=1;

    RETURN text;
  END CreateTextRunChar;


  PROCEDURE CreateTextRunChars*(char : CHAR; count : LONGINT):TextRun;

  VAR
    text : TextRun;

  BEGIN
    ASSERT(count<=B.blockLen);

    NEW(text);
    text.Init;
    text.block:=B.block;
    text.pos:=B.StoreCharsInBlock(char,count);
    text.length:=count;

    RETURN text;
  END CreateTextRunChars;

  PROCEDURE CreateTextRunString*(string : ARRAY OF CHAR; length : LONGINT):TextRun;

  VAR
    text : TextRun;

  BEGIN
    ASSERT(length<=B.blockLen);

    NEW(text);
    text.Init;
    text.block:=B.block;
    text.pos:=B.StoreInBlock(string,length);
    text.length:=length;

    RETURN text;
  END CreateTextRunString;


  (* -------- LineEntry ----------------------- *)

  PROCEDURE (l : LineEntry) Init*;

  BEGIN
    l.last:=NIL;
    l.next:=NIL;
    l.line:=NIL;
    l.x:=-1;
  END Init;

  PROCEDURE (l : LineEntry) Remove*;

  BEGIN
    IF l.line.first=l THEN
      l.line.first:=l.next;
      IF l.line.first#NIL THEN
        l.line.first.last:=NIL;
      END;
    ELSE
      l.last.next:=l.next;
    END;
    IF l.next#NIL THEN
      l.next.last:=l.last;
    END;
    l.next:=NIL;
    l.last:=NIL;
  END Remove;

  PROCEDURE (l : LineEntry) Print*;

  BEGIN
  END Print;

  (* -------- Line runs ----------------------- *)

  PROCEDURE (l : LineRun) Init*;

  BEGIN
    l.Init^;

    l.first:=NIL;
  END Init;

  PROCEDURE (l : LineRun) PrintLine*;

  VAR
    run   : Run;
    entry : LineEntry;

  BEGIN
    IF l.last#NIL THEN
      IF l.last.next#l THEN
        Err.String("!last line inconsistent!");
      END;
    END;

    IF l.next#NIL THEN
      IF l.next.last#l THEN
        Err.String("!next line inconsistent!");
      END;
    END;

    run:=l.next;
    WHILE (run#NIL) & ~(run IS LineRun) DO
      run.Print;
      Err.Char("");
      run:=run.next;
    END;
    Err.Char("");

    IF l.first#NIL THEN
      IF l.first.last#NIL THEN
        Err.String("!first entry inconsistent!");
      END;
    END;

    entry:=l.first;
    WHILE entry#NIL DO

      IF entry.last#NIL THEN
        IF entry.last.next#entry THEN
          Err.String("!last entry inconsistent!");
        END;
      END;

      IF entry.next#NIL THEN
        IF entry.next.last#entry THEN
          Err.String("!next entry inconsistent!");
        END;
      END;

      IF entry.x<=0 THEN
        Err.String("!entry pos inconsistent!");
      END;

      entry.Print;
      entry:=entry.next;
    END;
    Err.Ln;
  END PrintLine;

  (**
    Inserts a new line entry to the list of existing line entries for this line.
    The new entries will be inserted sorted by its x position.
  **)

  PROCEDURE (l : LineRun) InsertEntry*(entry : LineEntry);

  VAR
    help : LineEntry;

  BEGIN
    IF l.first=NIL THEN
      l.first:=entry;
      entry.last:=NIL;
      entry.next:=NIL;
    ELSIF entry.x<=l.first.x THEN
      entry.next:=l.first;
      l.first.last:=entry;
      entry.last:=NIL;
      l.first:=entry;
    ELSE
      help:=l.first;
      WHILE (help.next#NIL) & (entry.x>help.next.x) DO
        help:=help.next;
      END;
      entry.next:=help.next;
      IF entry.next#NIL THEN
        entry.next.last:=entry;
      END;
      entry.last:=help;
      help.next:=entry;
    END;

    entry.line:=l;
  END InsertEntry;

  (**
    Returns a pointer to the first line entry starting at the given position.
  **)

  PROCEDURE (l : LineRun) GetEntryAt*(x : LONGINT):LineEntry;

  VAR
    entry : LineEntry;

  BEGIN
    entry:=l.first;
    WHILE entry#NIL DO
      IF entry.x=x THEN
        RETURN entry;
      END;
      entry:=entry.next;
    END;
    RETURN NIL;
  END GetEntryAt;

  (**
    Returns the mark at the given position of the line.
  **)

  PROCEDURE (l : LineRun) GetMarkAt*(x : LONGINT):Mark;

  VAR
    entry : LineEntry;

  BEGIN
    entry:=l.first;
    WHILE (entry#NIL) & (entry.x#x) DO
      entry:=entry.next;
    END;

    IF entry#NIL THEN
      WHILE (entry#NIL) & (entry.x=x) DO
        IF entry IS Mark THEN
          RETURN entry(Mark);
        END;
        entry:=entry.next;
      END;
    END;
    RETURN NIL;
  END GetMarkAt;

  (**
    Returns TRUE if a marks is set at the given position of the line.
  **)

  PROCEDURE (l : LineRun) IsMarkAt*(x : LONGINT):BOOLEAN;

  BEGIN
    RETURN l.GetMarkAt(x)#NIL;
  END IsMarkAt;

  (**
    Returns the Run and the position within this Run
    which represents the given x position within the line.

    NOTE
    Pos may point beyond the textrange of the returned run,
    when the x position is greater than the length of the line.

    run may be NIL, when there is no textrun in the current line.
  **)

  PROCEDURE (l : LineRun) GetPos*(x : LONGINT; VAR run : TextRun; VAR pos : LONGINT);

  VAR
    r    : Run;
    last : TextRun;
    y    : LONGINT;

  BEGIN
    y:=1;
    r:=l.next;
    last:=NIL;

    LOOP
      IF r=NIL THEN         (* past end of text *)
        IF last#NIL THEN
          run:=last;
          pos:=x-y-1;
        ELSE
          run:=NIL;
          pos:=-1;
        END;
        RETURN;
      END;

      WITH
        r: LineRun DO      (* past end of line *)
        IF last#NIL THEN
          run:=last;
          pos:=last.pos+x-y;
        ELSE
          run:=NIL;
          pos:=-1;
        END;
        RETURN;
      | r : TextRun DO
        IF x<=y+r.length THEN
          run:=r;
          pos:=r.pos+x-y;
          RETURN;
        ELSE
          INC(y,r.length);
        END;
        last:=r;
      ELSE
      END;
      r:=r.next;
    END;
  END GetPos;

  (**
    Returns the textual length of the line.
  **)

  PROCEDURE (l : LineRun) Length*():LONGINT;

  VAR
    run    : Run;
    length : LONGINT;

  BEGIN
    length:=0;
    run:=l.next;

    WHILE (run#NIL) & ~(run IS LineRun) DO
      IF run IS TextRun THEN
        INC(length,run(TextRun).length);
      END;
      run:=run.next;
    END;

    RETURN length;
  END Length;

  PROCEDURE (l : LineRun) Split*(x : LONGINT);

  VAR
    text  : TextRun;
    line  : LineRun;
    entry : LineEntry;
    pos   : LONGINT;

  BEGIN
    NEW(line);
    line.Init;

    IF l.first#NIL THEN
      IF l.first.x>=x THEN
        line.first:=l.first;
        l.first:=NIL;
      ELSE
        entry:=l.first;
        WHILE (entry.next#NIL) & (entry.next.x<x) DO
          entry:=entry.next;
        END;
        IF entry.next#NIL THEN
          line.first:=entry.next;
          line.first.last:=NIL;
          entry.next:=NIL;
        END;
      END;

      entry:=line.first;
      WHILE entry#NIL DO
        DEC(entry.x,x-1);
        entry.line:=line;
        entry:=entry.next;
      END;
    END;

    IF (l.Length()>0) & (x>1) THEN
      l.GetPos(x,text,pos);
      text:=text.Split(pos);
      text.InsertAfter(line);
    ELSE
      l.InsertAfter(line);
    END;
  END Split;

  (**
    Removes the LineRun. as the result the previous line and the current
    line will be joined.
  **)

  PROCEDURE (l : LineRun) Remove*;

  VAR
    lastLine   : LineRun;
    entry,
    help       : LineEntry;
    lastLength : LONGINT;

  BEGIN
    ASSERT(l.last#NIL); (* We must not remove the first line in the text *)

    lastLine:=l.LastLine();
    lastLength:=lastLine.Length();

    (* delete from chain *)
    l.last.next:=l.next;
    IF l.next#NIL THEN
      l.next.last:=l.last;
    END;

    (* Add entries of current line to last line *)
    entry:=l.first;
    WHILE entry#NIL DO
      INC(entry.x,lastLength);
      IF entry IS Mark THEN
        DEC(entry(Mark).y);
      END;
      help:=entry.next;
      lastLine.InsertEntry(entry);
      entry:=help;
    END;

    l.Remove^;
  END Remove;

  (**
    Tries to join splited TextRuns.

    TextRuns can be joined when they point to text in the same block
    and the last char of the first Run is immediately followed by the
    first char of the second run.
  **)

  PROCEDURE (l : LineRun) Nice*;

  VAR
    run : Run;
    b   : TextRun;

  BEGIN
    run:=l.next;

    WHILE (run#NIL) & ~(run IS LineRun) DO
      WITH
        run : TextRun DO
          IF run.length<=0 THEN
            run.Remove;
          ELSIF (run.next#NIL) & (run.next IS TextRun) THEN
            b:=run.next(TextRun);
            IF (run.block=b.block) & (run.pos+run.length=b.pos) THEN
              INC(run.length,b.length);
              b.Remove;
            END;
          END;
      ELSE
      END;
      run:=run.next;
    END;
  END Nice;

  PROCEDURE (l : LineRun) Copy*():Run;

  VAR
    run : LineRun;

  BEGIN
    NEW(run);
    run^:=l^;
    run.first:=NIL;

    RETURN run;
  END Copy;

  PROCEDURE (l : LineRun) Print*;

  BEGIN
    IF l.last#NIL THEN
      Err.String("");
      Err.Ln;
    END;
  END Print;

  (**
    Move line entry to another line.
  **)

  PROCEDURE (e : LineEntry) RepositionEntry*(newLine : LineRun; newX : LONGINT);

  BEGIN
    e.Remove;
    e.x:=newX;
    newLine.InsertEntry(e);
  END RepositionEntry;

  (* -------- Mark ----------------------- *)

  PROCEDURE (m : Mark) Init*;

  BEGIN
    m.Init^;

    NEW(m.name,1);
    m.name[0]:=0X;
    m.y:=-1;
    m.type:=mark;
  END Init;

  (**
    Set the name of the mark.
  **)

  PROCEDURE (m : Mark) SetName*(name : ARRAY OF CHAR);

  BEGIN
    NEW(m.name,str.Length(name)+1);
    COPY(name,m.name^);
  END SetName;

  PROCEDURE (m : Mark) Move*(x,y : LONGINT):BOOLEAN;

  VAR
    new : LineRun;
    pos : LONGINT;

  BEGIN
    IF x<1 THEN
      RETURN FALSE;
    END;

    IF y=m.y THEN
      m.x:=x;
      m.Remove;
      m.line.InsertEntry(m);
      RETURN TRUE;
    ELSE
      new:=m.line;
      pos:=m.y;
      IF y>m.y THEN
       WHILE (new#NIL) & (pos<y) DO
         new:=new.NextLine();
         INC(pos);
       END;
      ELSE (* y<m.y *)
       WHILE (new#NIL) & (y<pos) DO
         new:=new.LastLine();
         DEC(pos);
       END;
      END;

      IF new#NIL THEN
        m.Remove;
        m.y:=y;
        m.x:=x;
        new.InsertEntry(m);
        RETURN TRUE;
      ELSE
        RETURN FALSE;
      END;
    END;
  END Move;

  PROCEDURE (m : Mark) Back*():BOOLEAN;

  VAR
    line : LineRun;

  BEGIN
    IF m.x=1 THEN
      IF m.line.IsFirstLine() THEN
        RETURN FALSE;
      ELSE
        line:=m.line.LastLine();
        RETURN m.Move(line.Length()+1,m.y-1);
      END;
    ELSE
      RETURN m.Move(m.x-1,m.y);
    END;
  END Back;

  PROCEDURE (m : Mark) Foreward*():BOOLEAN;

  BEGIN
    IF m.x>m.line.Length()+1 THEN
      IF m.line.IsLastLine() THEN
        RETURN FALSE;
      ELSE
        RETURN m.Move(1,m.y+1);
      END;
    ELSE
      RETURN m.Move(m.x+1,m.y);
    END;
  END Foreward;

  PROCEDURE (m : Mark) SetMark*(x,y : LONGINT):Mark;

  VAR
    new  : LineRun;
    mark : Mark;
    pos  : LONGINT;

  BEGIN
    IF x<1 THEN
      RETURN NIL;
    END;

    IF y=m.y THEN
      NEW(mark);
      mark.Init;
      mark.x:=x;
      mark.y:=y;
      m.line.InsertEntry(mark);
      RETURN mark;
    ELSE
      new:=m.line;
      pos:=m.y;
      IF y>m.y THEN
       WHILE (new#NIL) & (pos<y) DO
         new:=new.NextLine();
         INC(pos);
       END;
      ELSE (* y<m.y *)
       WHILE (new#NIL) & (y<pos) DO
         new:=new.LastLine();
         DEC(pos);
       END;
      END;

      IF new#NIL THEN
        NEW(mark);
        mark.Init;
        mark.x:=x;
        mark.y:=y;
        new.InsertEntry(mark);
        RETURN mark;
      ELSE
        RETURN NIL;
      END;
    END;
  END SetMark;

  PROCEDURE (m : Mark) SetType*(type : LONGINT);

  BEGIN
    m.type:=type;
  END SetType;

  PROCEDURE (m : Mark) Set*(x,y : LONGINT);

  BEGIN
    m.x:=x;
    m.y:=y;
  END Set;

  PROCEDURE (m : Mark) Print*;

  BEGIN
    Err.String("|");
    IF m.name#NIL THEN
      Err.String(m.name^);
      Err.String("/");
      Err.LongInt(m.x,0);
      Err.String("/");
      Err.LongInt(m.y,0);
      Err.String("|");
    END;
  END Print;

  PROCEDURE (m : Mark) Reposition*(x,y : LONGINT);

  BEGIN
    m.x:=x;
    m.y:=y;
    m.Remove;
    m.line.InsertEntry(m);
  END Reposition;

  PROCEDURE (m : Mark) Equal*(b : Mark):BOOLEAN;

  BEGIN
    RETURN (m.y=b.y) & (m.x=b.x);
  END Equal;

  PROCEDURE (m : Mark) Before*(b : Mark):BOOLEAN;

  BEGIN
    RETURN (m.y<b.y) OR ((m.y=b.y) & (m.x<b.x));
  END Before;

  PROCEDURE (m : Mark) BeforeEqual*(b : Mark):BOOLEAN;

  BEGIN
    RETURN (m.y<b.y) OR ((m.y=b.y) & (m.x<=b.x));
  END BeforeEqual;

  PROCEDURE (m : Mark) After*(b : Mark):BOOLEAN;

  BEGIN
    RETURN (m.y>b.y) OR ((m.y=b.y) & (m.x>b.x));
  END After;

  PROCEDURE (m : Mark) AfterEqual*(b : Mark):BOOLEAN;

  BEGIN
    RETURN (m.y>b.y) OR ((m.y=b.y) & (m.x>=b.x));
  END AfterEqual;

  PROCEDURE (m : Mark) SameLine*(b : Mark):BOOLEAN;

  BEGIN
    RETURN m.y=b.y;
  END SameLine;

END VO:EditRun.