(**
   A simple boolean/radio button with a selected and an unselected state.
**)

MODULE VO:State;

(*
    A simple state gadget.
    Copyright (C) 20001  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT D  := VO:Base:Display,
       E  := VO:Base:Event,
       O  := VO:Base:Object,

       VM := VO:Model:Value,

       G  := VO:Object,
       V  := VO:VecImage;

TYPE
  State*     = POINTER TO StateDesc;

 (**
   Implementation of a bool gadget.
 **)

  StateDesc* = RECORD (G.GadgetDesc)
                  imageType : LONGINT;
                  image     : V.VecImage;
                  model-    : VM.ValueModel;
                  index,
                  tmpState  : LONGINT;
                END;

  PROCEDURE (b : State) Init*;

  BEGIN
    b.Init^;

    b.SetFlags({G.canFocus});

    b.model:=NIL;

    b.imageType:=V.led;
    b.index:=1;
    
    b.image:=V.CreateVecImage();
    b.image.SetParent(b);
    b.image.SetFlags({G.horizontalFlex,G.verticalFlex});
  END Init;

  PROCEDURE (b : State) IsIn(x,y : LONGINT):BOOLEAN;

  BEGIN
    RETURN b.PointIsIn(x,y) OR ((b.label#NIL) & (b.label.PointIsIn(x,y)));
  END IsIn;


  (**
    Assign a bool model to the bool-gadget. The bool gadget
    will up to now always display the value of the bool model.
  **)

  PROCEDURE (b : State) SetModel*(model : O.Model);

  BEGIN
    IF b.model#NIL THEN
      b.UnattachModel(b.model);
    END;
    IF (model#NIL) & (model IS VM.ValueModel) THEN
      b.model:=model(VM.ValueModel);
      b.AttachModel(b.model);
    ELSE
      b.model:=NIL;
    END;
  END SetModel;

  PROCEDURE (b : State) SetImage*(image : LONGINT);

  BEGIN
    b.imageType:=image;
  END SetImage;

  PROCEDURE (b : State) SetIndex*(index : LONGINT);

  BEGIN
    b.index:=index;
  END SetIndex;

  (**
    This function is used to check if an argument to SetModel
    was successfully accepted.
   **)

  PROCEDURE (b : State) ModelAccepted * (m : O.Model):BOOLEAN;

  BEGIN
    RETURN m=b.model
  END ModelAccepted;


  PROCEDURE (b : State) CalcSize*;

  BEGIN
    (*
      If the label has a custom way for displaying a focus frame
      we delegate the focus displaying to the label.
    *)
    IF (b.label#NIL) & ~b.label.StdFocus() & b.MayFocus() THEN
      b.RemoveFlags({G.stdFocus});
      b.label.SetFlags({G.mayFocus});
    END;

    b.image.Set(b.imageType);
    b.image.CalcSize;
    b.width:=b.image.width;
    b.height:=b.image.height;
    b.minWidth:=b.image.minWidth;
    b.minHeight:=b.image.minHeight;

    b.CalcSize^;
  END CalcSize;

  PROCEDURE (b : State) OnClick*;

  BEGIN
    IF b.model.type=VM.boolean THEN
      IF b.tmpState=0 THEN
        b.model.SetBoolean(TRUE);
      ELSE
        b.model.SetBoolean(FALSE);
      END;
    ELSIF b.model.type=VM.longint THEN
      b.model.SetLongint(b.index);
    END;
  END OnClick;

  PROCEDURE (b : State) HandleMouseEvent*(event : E.MouseEvent;
                                         VAR grab : G.Object):BOOLEAN;

  BEGIN
    IF ~b.visible OR b.disabled OR (b.model=NIL) OR b.model.IsNull() THEN
      RETURN FALSE;
    END;

    WITH event : E.ButtonEvent DO
      IF (event.type=E.mouseDown)
      & b.IsIn(event.x,event.y) & (event.button=E.button1) THEN
        b.tmpState:=b.model.GetLongint();
        b.OnClick;

        grab:=b;
        RETURN TRUE;
      ELSIF (event.type=E.mouseUp) & (event.button=E.button1) THEN
        IF b.IsIn(event.x,event.y) THEN
          b.OnClick;
        END;

        grab:=NIL;
        RETURN TRUE;
      END;
    | event : E.MotionEvent DO
      IF grab#NIL THEN
        IF b.IsIn(event.x,event.y) THEN
          b.OnClick;
        ELSE
          b.model.SetLongint(b.tmpState);
        END;
        RETURN TRUE;
      END;
    ELSE
    END;
    RETURN FALSE;
  END HandleMouseEvent;

  PROCEDURE (b : State) HandleKeyEvent*(event : E.KeyEvent):BOOLEAN;

  BEGIN
    IF event.type=E.keyDown THEN
      IF event.key=E.space THEN
        b.tmpState:=b.model.GetLongint();
        b.OnClick;
        RETURN TRUE;
      END;
    END;
    RETURN FALSE;
  END HandleKeyEvent;

  PROCEDURE (b : State) Layout*;

  BEGIN
    b.image.Resize(b.width,b.height);
    b.image.Move(b.x+(b.width-b.image.width) DIV 2,
                 b.y+(b.height-b.image.height) DIV 2);

    b.Layout^;
  END Layout;

  PROCEDURE (b : State) Draw*(x,y,w,h : LONGINT);

  VAR
    draw : D.DrawInfo;

  BEGIN
    b.Draw^(x,y,w,h);

    IF ~b.Intersect(x,y,w,h) THEN
      RETURN;
    END;

    draw:=b.GetDrawInfo();


    IF (b.model#NIL) & ~b.model.IsNull() & (b.model.GetLongint()=b.index) THEN
      draw.mode:={D.selected};
    ELSE
      draw.mode:={};
    END;

    IF b.disabled & (G.canDisable IN b.image.flags) THEN
      INCL(draw.mode,D.disabled);
    END;

    b.image.Draw(x,y,w,h);

    draw.mode:={};

    IF b.disabled & ~(G.canDisable IN b.image.flags) THEN
      b.DrawDisabled;
    END;
  END Draw;

  (**
    Draw the keyboard focus.
  **)

  PROCEDURE (b : State) DrawFocus*;

  BEGIN
    (* If our image can draw a keyboard focus, delegate it *)
    IF (b.label#NIL) & ~b.label.StdFocus() THEN
      b.label.DrawFocus;
    ELSE
      (* Delegate drawing to the baseclass *)
      b.DrawFocus^;
    END;
  END DrawFocus;

  (**
    Hide the keyboard focus.
  **)

  PROCEDURE (b : State) HideFocus*;

  BEGIN
    (* If our image can draw a keyboard focus, delegate it *)
    IF (b.label#NIL) & ~b.label.StdFocus() THEN
      b.label.HideFocus;
    ELSE
      (* Delegate drawing to the baseclass *)
      b.HideFocus^;
    END;
  END HideFocus;

  PROCEDURE (b : State) Resync*(model : O.Model; msg : O.ResyncMsg);

  BEGIN
    IF b.visible THEN
      b.Redraw;
    END;
  END Resync;

  PROCEDURE (b : State) Hide*;

  BEGIN
    IF b.visible THEN
      b.image.Hide;
      b.Hide^;
    END;
  END Hide;

  PROCEDURE CreateState*(): State;

  VAR
    state : State;

  BEGIN
    NEW(state);
    state.Init;

    RETURN state;
  END CreateState;

END VO:State.