(* 	$Id: File.Mod,v 1.4 2000/07/31 14:25:23 mva Exp $	 *)
MODULE URI:Scheme:File;
(*  Implementation of the "file" URI scheme.
    Copyright (C) 2000  Michael van Acken

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with OOC. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)


IMPORT
  Msg, Strings, Files, OS:ProcessParameters, 
  URI0 := URI, URI:Error, URI:CharClass, URI:String, 
  URI:Scheme:Hierarchical, URI:Authority:ServerBased, URI:Parser;


TYPE
  URI* = POINTER TO URIDesc;
  URIDesc = RECORD
  (**This class implements the @samp{file:} URI scheme.  It uses
     @otype{ServerBased.Authority} for its authority component, and does
     not support a query part.  *)
    (Hierarchical.GenericDesc)
  END;


PROCEDURE Init* (file: URI; schemeId: URI0.StringPtr;
                 authority: URI0.Authority; query: URI0.Query);
  BEGIN
    Hierarchical.Init (file, schemeId, authority, query)
  END Init;

PROCEDURE New* (schemeId: URI0.StringPtr;
                authority: URI0.Authority; query: URI0.Query): URI;
  VAR
    file: URI;
  BEGIN
    NEW (file);
    Init (file, schemeId, authority, query);
    RETURN file
  END New;

PROCEDURE (file: URI) NewAuthority* (): URI0.Authority;
  BEGIN
    RETURN ServerBased.New (NIL, String.Copy (""), -1, -1)
  END NewAuthority;

PROCEDURE (file: URI) NewQuery* (): URI0.Query;
  BEGIN
    RETURN NIL
  END NewQuery;

PROCEDURE (file: URI) Clone* (): URI;
  VAR
    copy: URI;
  BEGIN
    NEW (copy);
    file. Copy (copy);
    RETURN copy
  END Clone;

PROCEDURE (file: URI) GetPath* (VAR filePath: ARRAY OF CHAR);
(**Returns the file path associated with the URI @oparam{file}.  On a Unix 
   system, this is identical to the URI's path component.  *)
  VAR
    segm: Hierarchical.Segment;
  BEGIN
    COPY ("", filePath);
    segm := file. pathList;
    WHILE (segm # NIL) DO
      IF (segm # file. pathList) OR file. absolutePath THEN
        Strings.Append ("/", filePath)
      END;
      Strings.Append (segm. string^, filePath);
      segm := segm. next
    END
  END GetPath;

PROCEDURE (file: URI) GetChannel* (mode: URI0.ChannelMode; VAR res: Error.Msg): Files.File;
  VAR
    filePath: ARRAY 2*1024 OF CHAR;
  BEGIN
    file. GetPath (filePath);
    CASE mode OF
    | URI0.channelNew:
      RETURN Files.New (filePath, {Files.read, Files.write}, res)
    | URI0.channelTmp:
      RETURN Files.Tmp (filePath, {Files.read, Files.write}, res)
    | URI0.channelOld:
      RETURN Files.Old (filePath, {Files.read}, res)
    END
  END GetChannel;

PROCEDURE NewPrototype*(): URI;
  BEGIN
    RETURN New (String.Copy ("file"), NIL, NIL)
  END NewPrototype;


PROCEDURE GetCwd*(): URI;
(**Creates an absolute file URI for the current working directory.  If the
   current working directory cannot be determined, the value @code{NIL} is
   returned.  *)
  VAR
    uri: URI0.URI;
    path, uriString: ARRAY 4*1024 OF CHAR;
    res: Msg.Msg;
  BEGIN
    ProcessParameters.getcwd (path, res);
    IF (res = NIL) THEN
      uriString := "file:";
      String.AppendEscaped (path, CharClass.unreservedPChar+"/", uriString);
      IF (uriString[Strings.Length (uriString)-1] # "/") THEN
        Strings.Append ("/", uriString)
      END;
      uri := Parser.NewURI (uriString, NIL, res);
      ASSERT (uri # NIL);
      RETURN uri(URI)
    END;
    RETURN NIL
  END GetCwd;

PROCEDURE ToURI* (filePath: ARRAY OF CHAR): URI;
(**Constructs a @samp{file:} URI for a given file path.  Characters that
   are not valid within an URI are escaped.  *)
  VAR
    uri: URI0.URI;
    uriString: ARRAY 4*1024 OF CHAR;
    res: Msg.Msg;
  BEGIN
    IF (filePath[0] = "/") THEN
      uriString := "file:";
      String.AppendEscaped (filePath, CharClass.unreservedPChar+"/", uriString);
      uri := Parser.NewURI (uriString, NIL, res)
    ELSE
      uri := Parser.NewURI (filePath, GetCwd(), res)
    END;
    IF (uri = NIL) THEN
      RETURN NIL
    ELSE
      RETURN uri(URI)
    END
  END ToURI;

BEGIN
  URI0.RegisterScheme (NewPrototype())
END URI:Scheme:File.
