/*################################################################################
# Linux Management Providers (LMP), Provider Common Library
# Copyright (C) 2007 Frederic Desmons, ETRI <desmons@etri.re.kr, desmons_frederic@yahoo.fr>
# 
# This program is being developed under the "OpenDRIM" project.
# The "OpenDRIM" project web page: http://opendrim.sourceforge.net
# The "OpenDRIM" project mailing list: opendrim@googlegroups.com
# 
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; version 2
# of the License.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#################################################################################

#################################################################################
# To contributors, please leave your contact information in this section
# AND comment your changes in the source code.
# 
# Modified by <Author(s)>, <Affiliation>, <Year>
################################################################################*/

#include "Indication.h"

void CI_enableFilter(bool& enabled, _workerFlag& workerFlag, _workerSleep& workerSleep) {
	if (!enabled) {
		pthread_mutex_lock(&workerFlag.mutex);
		// Worker sleeps because the indication is disabled.
		// Wake up!
		if ((workerFlag.flag & WORKER_STOP) == WORKER_STOP) {
			workerFlag.flag = workerFlag.flag & !WORKER_STOP | WORKER_START;
			pthread_mutex_lock(&workerSleep.mutex);
			pthread_cond_signal(&workerSleep.cond);
			pthread_mutex_unlock(&workerSleep.mutex);
		}
		pthread_mutex_unlock(&workerFlag.mutex);
	}
	enabled=true;
}

// Indication utility, disable an indication filter
void CI_disableFilter(bool& enabled, _workerFlag& workerFlag) {
	if (enabled) {
		pthread_mutex_lock(&workerFlag.mutex);
		// Worker is awake, we're gonna put it to sleep
		if ((workerFlag.flag & WORKER_START) == WORKER_START) {
			workerFlag.flag = workerFlag.flag & !WORKER_START | WORKER_STOP;
		}
		pthread_mutex_unlock(&workerFlag.mutex);
	}
	enabled=false;
}

int CI_addFilter(const CMPISelectExp* filter, _indicationFilter& indicationFilter, bool enabled, _workerFlag& workerFlag, _workerSleep& workerSleep, string& errorMessage) {
	if (filter == NULL){
		errorMessage+="Filter argument is null!";
		return FAILED;
	}
	pthread_mutex_lock(&indicationFilter.mutex);
	if (indicationFilter.isSet) {
		errorMessage+="Duplicated indication filter found!";
		return FAILED;
	}
	indicationFilter.isSet=true;
	CMPIStatus rc;
    if (filter!=NULL) {
		indicationFilter.filter=CMClone(filter, &rc);
		if (rc.rc==CMPI_RC_ERR_NOT_SUPPORTED) {
			errorMessage+="Cloning of filters is not supported!";
			indicationFilter.filter=(CMPISelectExp*) filter;
		}
		else if (rc.rc==CMPI_RC_ERR_FAILED) {
			errorMessage+="Cloning of filter failed!";
			return FAILED;
		}
    }
    pthread_mutex_unlock(&indicationFilter.mutex);
    // The filter was not set, assume the worker was sleeping, so we awake it.
    // In the case the worker is stopped because the indication is disabled, we skip this
    pthread_mutex_lock(&workerFlag.mutex);
    if (enabled) {
    	pthread_mutex_lock(&workerSleep.mutex);
		pthread_cond_signal(&workerSleep.cond);
		pthread_mutex_unlock(&workerSleep.mutex);
    }
    pthread_mutex_unlock(&workerFlag.mutex);
	return OK;
}

int CI_removeFilter(const CMPISelectExp* filter, _indicationFilter& indicationFilter, string& errorMessage) {
	if (filter == NULL){
		errorMessage+="Filter argument is null!";
		return FAILED;
	}
	pthread_mutex_lock(&indicationFilter.mutex);
	if (!indicationFilter.isSet) {
		errorMessage+="Filter was not registered!";
		return FAILED;
	}
	// Just mark the filter as unset
	indicationFilter.isSet=false;
	pthread_mutex_unlock(&indicationFilter.mutex);
	return OK;
}

int CI_initWorker(const CMPIBroker* broker, const CMPIContext* ctx, bool& inited, _workerFlag& workerFlag, void* (*worker)(void*), pthread_t& workerID, string& errorMessage) {
	if (broker == NULL) {
		errorMessage+="CMPI broker is NULL!";
    	return FAILED;
	}
  	CMPIContext* context = CBPrepareAttachThread (broker, ctx);
	if (context == NULL) {
		errorMessage+="CMPI context is NULL!";
		return FAILED;
	}
	if (inited)
		return OK;
	pthread_mutex_lock(&workerFlag.mutex);
	if ((workerFlag.flag & BROKER_REGISTERED) == BROKER_REGISTERED) {
		errorMessage+="Attempt to register the broker twice!";
		pthread_mutex_unlock(&workerFlag.mutex);
		return FAILED;
    }
	workerFlag.flag = BROKER_REGISTERED | WORKER_STOP;
	pthread_mutex_unlock(&workerFlag.mutex);
	void** threadData=(void**) malloc(sizeof(CMPIBroker*)+sizeof(CMPIContext*));
	threadData[0]=(CMPIBroker*) broker;
	threadData[1]=(CMPIContext*) context;
	workerID=0;
	pthread_create(&workerID, NULL, worker, threadData);
	inited=true;
	return OK;	
}

int CI_finalizeWorker(_workerFlag& workerFlag, _workerSleep& workerSleep, pthread_t workerID, bool& inited, _indicationFilter& indicationFilter, string& errorMessage) {
	pthread_mutex_lock(&workerFlag.mutex);
	if ((workerFlag.flag & BROKER_NOT_REGISTERED) == BROKER_NOT_REGISTERED) {
		errorMessage+="Broker was not registered!";
		pthread_mutex_unlock(&workerFlag.mutex);
		return FAILED;
	}
	pthread_mutex_lock(&indicationFilter.mutex);
	if (indicationFilter.isSet) {
		errorMessage+="At least on filter is still registered, will not shutdown!";
		pthread_mutex_unlock(&indicationFilter.mutex);
		pthread_mutex_unlock(&workerFlag.mutex);
		return DO_NOT_UNLOAD;
	}
	workerFlag.flag = BROKER_NOT_REGISTERED | WORKER_END_LIFE;
	pthread_mutex_unlock(&indicationFilter.mutex);
	pthread_mutex_unlock(&workerFlag.mutex);
	pthread_cond_signal(&workerSleep.cond);
	// Join the worker
	pthread_join(workerID, NULL);
	inited=false;
	return OK;
}
