/*################################################################################
# Linux Management Providers (LMP), Provider Common Library
# Copyright (C) 2008 Guillaume BOTTEX, ETRI <guillaumebottex@etri.re.kr, guillaumebottex@gmail.com>
#
# This program is being developed under the "OpenDRIM" project.
# The "OpenDRIM" project web page: http://opendrim.sourceforge.net
# The "OpenDRIM" project mailing list: opendrim@googlegroups.com
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; version 2
# of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#################################################################################

#################################################################################
# To contributors, please leave your contact information in this section
# AND comment your changes in the source code.
#
# 2008 Guillaume BOTTEX, ETRI <guillaumebottex@etri.re.kr, guillaumebottex@gmail.com>
################################################################################*/

#include "SMBIOS.h"

string SMBIOS_cacheInformationToString(const _cache_information& cache_information, const string& identation) {
	_E_;
	string cache_information_string;
	cache_information_string += identation + "handle: " + CF_intToStr(cache_information.handle) + "\n";
	cache_information_string += identation + "operational_mode: " + CF_intToStr(cache_information.operational_mode) + "\n";
	cache_information_string += identation + "enabled: " + CF_boolToStr(cache_information.enabled) + "\n";
	cache_information_string += identation + "cache_level: " + CF_intToStr(cache_information.cache_level) + "\n";
	cache_information_string += identation + "granularity: " + CF_intToStr(cache_information.granularity) + "\n";
	cache_information_string += identation + "installed_size: " + CF_intToStr(cache_information.installed_size) + "\n";
	cache_information_string += identation + "system_cache_type: " + CF_intToStr(cache_information.system_cache_type) + "\n";
	cache_information_string += identation + "associativity: " + CF_intToStr(cache_information.associativity) + "\n";
	_L_;
	return cache_information_string;
}

string SMBIOS_processorInformationToString(const _processor_information& processor_information, const string& identation) {
	_E_;
	string processor_information_string;
	processor_information_string += identation + "processor_type: " + CF_intToStr(processor_information.processor_type) + "\n";
	processor_information_string += identation + "processor_family: " + CF_intToStr(processor_information.processor_family) + "\n";
	processor_information_string += identation + "has_voltage: " + CF_boolToStr(processor_information.has_voltage) + "\n";
	processor_information_string += identation + "voltage: " + CF_intToStr(processor_information.voltage) + "\n";
	processor_information_string += identation + "external_clock: " + CF_intToStr(processor_information.external_clock) + "\n";
	processor_information_string += identation + "max_speed: " + CF_intToStr(processor_information.max_speed) + "\n";
	processor_information_string += identation + "status: " + CF_intToStr(processor_information.status) + "\n";
	processor_information_string += identation + "processor_upgrade: " + CF_intToStr(processor_information.processor_upgrade) + "\n";
	processor_information_string += identation + "core_enabled: " + CF_intToStr(processor_information.core_enabled) + "\n";
	processor_information_string += identation + "processor_characteristics: ";
	for (unsigned int i = 0; i < processor_information.processor_characteristics.size(); i++)
		processor_information_string += CF_intToStr(processor_information.processor_characteristics[i])+ ";";
	processor_information_string += "\n";
	processor_information_string += identation + "has_l1_cache: " + CF_boolToStr(processor_information.has_l1_cache) + "\n";
	processor_information_string += identation + "l1_cache_handle: " + CF_intToStr(processor_information.l1_cache_handle) + "\n";
	if (processor_information.has_l1_cache)
		processor_information_string += SMBIOS_cacheInformationToString(processor_information.l1_cache, identation + "   ");
	processor_information_string += identation + "has_l2_cache: " + CF_boolToStr(processor_information.has_l2_cache) + "\n";
	processor_information_string += identation + "l2_cache_handle: " + CF_intToStr(processor_information.l2_cache_handle) + "\n";
	if (processor_information.has_l2_cache)
		processor_information_string += SMBIOS_cacheInformationToString(processor_information.l2_cache, identation + "   ");
	processor_information_string += identation + "has_l3_cache: " + CF_boolToStr(processor_information.has_l3_cache) + "\n";
	processor_information_string += identation + "l3_cache_handle: " + CF_intToStr(processor_information.l3_cache_handle) + "\n";
	if (processor_information.has_l3_cache)
		processor_information_string += SMBIOS_cacheInformationToString(processor_information.l3_cache, identation + "   ");
	_L_;
	return processor_information_string;
}

void SMBIOS_cacheInformationToDefault(_cache_information& cache_information) {
	_E_;
	cache_information.handle = 0;
	cache_information.operational_mode = 0; // Unknown
	cache_information.enabled = false;
	cache_information.cache_level = 0; // Unknown
	cache_information.granularity = 0;
	cache_information.installed_size = 0;
	cache_information.system_cache_type = 0x2; // Unknown
	cache_information.associativity = 0x2; // Unknown
	_L_;
}

void SMBIOS_processorInformationToDefault(_processor_information& processor_information) {
	_E_;
	processor_information.processor_characteristics.clear();
	processor_information.has_l1_cache = false;
	processor_information.has_l2_cache = false;
	processor_information.has_l3_cache = false;
	processor_information.core_enabled = 0;
	_L_;
}

int SMBIOS_findEntryPoint(int& dev_mem, void** ptrptr, _smbios_anchor** smbios_anchor_ptrptr, string& errorMessage) {
	_E_;
	// try to open the system memory (read-only)
	dev_mem = open("/dev/mem", O_RDONLY);

	// we cannot open the file, exit
	if (dev_mem < 0) {
		errorMessage = "Cannot open file: /dev/mem";
		return FAILED;
    }

    // look for smbios anchor
    // the anchor is located between 0x000f0000 and 0x000fffff in system memory
    *ptrptr = mmap(NULL, _SMBIOS_ANCHOR_SEARCH_RANGE, PROT_READ, MAP_SHARED, dev_mem, _SMBIOS_ANCHOR_SEARCH_OFFSET);

    if (*ptrptr ==  MAP_FAILED) {
    	errorMessage = "Cannot map system BIOS memory";
    	close(dev_mem);
		return FAILED;
    }

    for ((*smbios_anchor_ptrptr) = (_smbios_anchor*) *ptrptr; (unsigned long) (*smbios_anchor_ptrptr) < ((unsigned long) *ptrptr + (unsigned long) _SMBIOS_ANCHOR_SEARCH_RANGE);) {
    	// look for the sequence "_SM_"
    	if (memcmp("_SM_", (*smbios_anchor_ptrptr)->anchor_string, 4) == 0) {
    		return OK;
    	}
    	// skip 16 bytes (anchor string is aligned on 16 bytes paragraph boundaries)
    	*smbios_anchor_ptrptr+=4;
    }

    errorMessage = "Failed to locate SMBIOS information in mermory";
	_L_;
	return FAILED;
}

char* SMBIOS_nextSMBIOSStructureHeader(char* ptr, _string_map *string_map_ptr) {
	_E_;
	char* base = ptr + ((_smbios_structure_header*) ptr)->length;
	memset(string_map_ptr, 0, sizeof(_string_map));
	string_map_ptr->string[0]="unknown";
	if (base[0]==0 && base[1]==0)
		return base + 2;
	for (unsigned int i = 1; *base != 0 && i < 256; i++) {
		string_map_ptr->string[i] = base;
		base += strlen(base) + 1;
	}
	_L_;
	return base + 1;
}

int SMBIOS_getProcessorsInformation(vector<_processor_information>& processors_information, string& errorMessage) {
	_E_;
	void* ptr = NULL;
	int dev_mem;
	_smbios_anchor* smbios_anchor = NULL;

	// look for the SMBIOS entry point
	CF_assert(SMBIOS_findEntryPoint(dev_mem, &ptr, &smbios_anchor, errorMessage));

	_smbios_entry_point* smbios_entry_point_ptr = (_smbios_entry_point*) smbios_anchor;

	unsigned smbios_off;

	// align to memory pages
	size_t page_size = getpagesize();
	if (page_size != 0 && smbios_entry_point_ptr->structure_table_address%page_size != 0)
		smbios_off = smbios_entry_point_ptr->structure_table_address/page_size * page_size;
	else
		smbios_off = smbios_entry_point_ptr->structure_table_address;

	// map the SMBIOS memory (page aligned)
	void* smbios_ptr = mmap(NULL, smbios_entry_point_ptr->structure_table_address + smbios_entry_point_ptr->structure_table_length - smbios_off, PROT_READ, MAP_SHARED, dev_mem, smbios_off);

	if (smbios_ptr ==  MAP_FAILED) {
    	errorMessage = "Cannot map SMBIOS memory";
    	close(dev_mem);
		return FAILED;
    }

    // text strings
    _string_map string_map;

    // pointer to the first SMBIOS structure
    char* char_ptr = (char*) (smbios_entry_point_ptr->structure_table_address - smbios_off + (unsigned long) smbios_ptr);

    // recover the processor and cache information first
    unsigned short number_of_smbios_structure = smbios_entry_point_ptr->number_of_smbios_structure;
    vector<_cache_information> caches_information;
    while (number_of_smbios_structure > 0 && char_ptr) {
    	char* next_ptr = SMBIOS_nextSMBIOSStructureHeader(char_ptr, &string_map);
    	// processor
    	if (((_smbios_structure_header*) char_ptr)->type == _PROCESSOR_INFORMATION) {
    		_processor_information processor_information;
    		SMBIOS_processorInformationToDefault(processor_information);
    		// Processor Type
    		processor_information.processor_type = (unsigned char) char_ptr[0x5];
    		// Processor Family
    		processor_information.processor_family = (unsigned char) char_ptr[0x6];
    		// Voltage
    		unsigned char temp = char_ptr[0x11];
    		if (temp & 0x80) {
    			// real time voltage
    			processor_information.voltage = (temp & 0x7f)*100;
    			processor_information.has_voltage = true;
    		}
    		else {
				// we are not interested in the voltage capabilities...
				processor_information.has_voltage = false;
	   		}
	   		// External Clock
    		processor_information.external_clock = ((unsigned char) char_ptr[0x12]) + (((unsigned char) char_ptr[0x13]) << 8);
    		// Max Speed
    		processor_information.max_speed = ((unsigned char) char_ptr[0x14]) + (((unsigned char) char_ptr[0x15]) << 8);
    		// Status
    		temp = char_ptr[0x18];
    		processor_information.status = (temp & 0x7);
    		// Processor Upgrade
    		processor_information.processor_upgrade = (unsigned char) char_ptr[0x19];
    		// Core Enabled
    		if (((_smbios_structure_header*) char_ptr)->length >= 0x25)
    			processor_information.core_enabled = (unsigned char)  char_ptr[0x24];
    		else
    			processor_information.core_enabled = 0;
    		// Processor Charateristics
    		if (((_smbios_structure_header*) char_ptr)->length >= 0x28) {
    			temp = char_ptr[0x27];
    			if (temp & 0x2)
    				processor_information.processor_characteristics.push_back(1);
    			if (temp & 0x4)
    				processor_information.processor_characteristics.push_back(2);
    			if (temp & 0x8)
    				processor_information.processor_characteristics.push_back(3);
    			if (temp & 0x10)
    				processor_information.processor_characteristics.push_back(4);
    			if (temp & 0x20)
    				processor_information.processor_characteristics.push_back(5);
    			if (temp & 0x40)
    				processor_information.processor_characteristics.push_back(6);
    			if (temp & 0x80)
    				processor_information.processor_characteristics.push_back(7);
    			temp = char_ptr[0x26];
    			if (temp & 0x1)
    				processor_information.processor_characteristics.push_back(8);
    			if (temp & 0x2)
    				processor_information.processor_characteristics.push_back(9);
    			if (temp & 0x4)
    				processor_information.processor_characteristics.push_back(10);
    			if (temp & 0x8)
    				processor_information.processor_characteristics.push_back(11);
    			if (temp & 0x10)
    				processor_information.processor_characteristics.push_back(12);
    			if (temp & 0x20)
    				processor_information.processor_characteristics.push_back(13);
    			if (temp & 0x40)
    				processor_information.processor_characteristics.push_back(14);
    			if (temp & 0x80)
    				processor_information.processor_characteristics.push_back(15);
    		}
    		// L1 Cache Handle
    		if (((_smbios_structure_header*) char_ptr)->length >= 0x1c) {
    			processor_information.l1_cache_handle = ((unsigned char) char_ptr[0x1a]) + (((unsigned char) char_ptr[0x1b]) << 8);
    			processor_information.has_l1_cache = true;
    			// no cache or no information about the cache memory
    			if (processor_information.l1_cache_handle == 0x0ffff)
    				processor_information.has_l1_cache = false;
    		}
    		// L2 Cache Handle
    		if (((_smbios_structure_header*) char_ptr)->length >= 0x1e) {
    			processor_information.l2_cache_handle = ((unsigned char) char_ptr[0x1c]) + (((unsigned char) char_ptr[0x1d]) << 8);
    			processor_information.has_l2_cache = true;
    			// no cache or no information about the cache memory
    			if (processor_information.l2_cache_handle == 0x0ffff)
    				processor_information.has_l2_cache = false;
    		}
    		// L3 Cache Handle
    		if (((_smbios_structure_header*) char_ptr)->length >= 0x20) {
    			processor_information.l3_cache_handle = ((unsigned char) char_ptr[0x1e]) + (((unsigned char) char_ptr[0x1f]) << 8);
    			processor_information.has_l3_cache = true;
    			// no cache or no information about the cache memory
    			if (processor_information.l3_cache_handle == 0x0ffff)
    				processor_information.has_l3_cache = false;
    		}
    		processors_information.push_back(processor_information);
    	}
    	// cache
    	if (((_smbios_structure_header*) char_ptr)->type == _CACHE_INFORMATION) {
    		_cache_information cache_information;
    		SMBIOS_cacheInformationToDefault(cache_information);
    		// Handle
    		cache_information.handle = ((unsigned char) char_ptr[0x2]) + (((unsigned char) char_ptr[0x3]) << 8);
    		// Cache Configuration
    		unsigned char temp = char_ptr[0x5];
    		if ((temp & 0x3) == 0)
    			cache_information.operational_mode = 3;
    		if ((temp & 0x3) == 1)
    			cache_information.operational_mode = 2;
    		if ((temp & 0x3) == 2)
    			cache_information.operational_mode = 4;
    		if ((temp & 0x3) == 3)
    			cache_information.operational_mode = 0;
    		cache_information.enabled = false;
    		temp = char_ptr[0x6];
    		if (temp & 0x80)
    			cache_information.enabled = true;
    		cache_information.cache_level = (temp & 0x7);
    		cache_information.cache_level+=3;
    		// Installed Size
    		temp = char_ptr[0x9];
    		if (temp & 0x80)
    			cache_information.granularity = 64*1024;
    		else
    			cache_information.granularity = 1024;
    		cache_information.installed_size = (temp & 0x78) + (((unsigned char) char_ptr[0xa]) << 8);
  			// System Cache Type
  			if (((_smbios_structure_header*) char_ptr)->length >= 0x12)
  				cache_information.system_cache_type = (unsigned char) char_ptr[0x11];
  			// Associativity
  			if (((_smbios_structure_header*) char_ptr)->length >= 0x13)
  				cache_information.associativity = (unsigned char) char_ptr[0x12];
    		caches_information.push_back(cache_information);
    	}
    	number_of_smbios_structure--;
    	char_ptr = next_ptr;
    }

    // unmap the memory
	if (smbios_ptr)
		munmap(smbios_ptr, smbios_entry_point_ptr->structure_table_address + smbios_entry_point_ptr->structure_table_length - smbios_off);
	if (ptr)
		munmap(ptr, _SMBIOS_ANCHOR_SEARCH_RANGE);
	// close the memory file
	if (dev_mem>0)
		close(dev_mem);

	// map caches to processors...
	for (unsigned int i = 0; i < processors_information.size(); i++) {
		bool got_l1 = !(processors_information[i].has_l1_cache);
		bool got_l2 = !(processors_information[i].has_l2_cache);
		bool got_l3 = !(processors_information[i].has_l3_cache);
		for (unsigned int j = 0; j < caches_information.size(); j++) {
			if (!got_l1 && caches_information[j].handle == processors_information[i].l1_cache_handle) {
				processors_information[i].l1_cache = caches_information[j];
				// this means the cache doesn't exist
				if (caches_information[j].installed_size == 0)
					processors_information[i].has_l1_cache = false;
				got_l1 = true;
			}
			if (!got_l2 && caches_information[j].handle == processors_information[i].l2_cache_handle) {
				processors_information[i].l2_cache = caches_information[j];
				// this means the cache doesn't exist
				if (caches_information[j].installed_size == 0)
					processors_information[i].has_l2_cache = false;
				got_l2 = true;
			}
			if (!got_l3 && caches_information[j].handle == processors_information[i].l3_cache_handle) {
				processors_information[i].l3_cache = caches_information[j];
				// this means the cache doesn't exist
				if (caches_information[j].installed_size == 0)
					processors_information[i].has_l3_cache = false;
				got_l3 = true;
			}
			if (got_l1 && got_l2 && got_l3)
				break;
		}
		// we didn't find the information about the cache so we'll set it to default
		if (!got_l1)
			SMBIOS_cacheInformationToDefault(processors_information[i].l1_cache);
		if (!got_l2)
			SMBIOS_cacheInformationToDefault(processors_information[i].l2_cache);
		if (!got_l3)
			SMBIOS_cacheInformationToDefault(processors_information[i].l3_cache);
	}

	_L_;
	return OK;
}

void *SMBIOS_getRawData(size_t base, size_t len, string& errorMessage)
{
	int fd; 		// File descriptor
	void *p;		// Pointer to return
	void *mmp;	// Memory Map pointer
	size_t mmoffset;

	p=NULL;

	if((fd=open(DEV_MEM, O_RDONLY))==-1)
	{
		errorMessage = DEV_MEM;
		errorMessage+= ": ";
		errorMessage+= strerror(errno);
		return NULL;
	}

	if((p=malloc(len))==NULL)
	{
		errorMessage = "malloc: ";
		errorMessage+= strerror(errno);
		return NULL;
	}

#ifdef _SC_PAGESIZE
	mmoffset=base%sysconf(_SC_PAGESIZE);
#else
	mmoffset=base%getpagesize();
#endif /* _SC_PAGESIZE */

	mmp=mmap(NULL, mmoffset+len, PROT_READ, MAP_SHARED, fd, base-mmoffset);
	if(mmp==MAP_FAILED)
	{
		errorMessage = "mmap: ";
		errorMessage+= strerror(errno);
		close(fd);
		return NULL;
	}

	memcpy(p, (u8 *)mmp+mmoffset, len);

	if(munmap(mmp,mmoffset+len)==-1)
	{
		errorMessage = "unmap: ";
		errorMessage+= strerror(errno);
		close(fd);
		return NULL;
	}

	if(close(fd)==-1)
	{
		errorMessage = DEV_MEM;
		errorMessage+= ": ";
		errorMessage+= strerror(errno);
	}

	return p;
}

_smbios_entry_point *SMBIOS_getEntryPoint(void *start)
{
	_smbios_anchor *anchor;

	for(anchor = (_smbios_anchor*) start;
			(unsigned long) anchor < ((unsigned long) start + (unsigned long) _SMBIOS_ANCHOR_SEARCH_RANGE);
			anchor+=4) // skip 16 bytes (anchor string is aligned on 16 bytes paragraph boundaries)
	{
		// look for the sequence "_SM_"
		if (memcmp("_SM_", anchor->anchor_string, 4) == 0)
			return (_smbios_entry_point*)anchor;
	}

	return NULL;
}

int SMBIOS_getStructure(vector<void*>& output,char *tableAddress, u16 structureCount, u8 Type)
{
	u16 i;
	u8 lasttype;
	i = 0;

	output.clear();
	while( i < structureCount)
	{
		lasttype = ((_smbios_structure_header *)tableAddress)->type;

		if( lasttype == Type )
			output.push_back((void*)tableAddress);

		char* base = tableAddress + ((_smbios_structure_header*) tableAddress)->length;
		if (base[0]==0 && base[1]==0)
			tableAddress = base + 2;
		else
		{
			for (unsigned int j = 1; *base != 0 && j < 256; j++)
				base += strlen(base) + 1;
			tableAddress = base + 1;
		}
		i++;
	}

	return OK;
}

const char *SMBIOS_getDmiString(_smbios_structure_header *dm, BYTE s)
{
	//char *bp=(char *)dm->data;
	char *bp;
	size_t i, len;

	if(s==0)
		return "Not Specified";

	bp=(char *)dm;
	bp+=dm->length;
	while(s>1 && *bp)
	{
		bp+=strlen(bp);
		bp++;
		s--;
	}


	if(!*bp)
		return "BAD INDEX";

	// ASCII filtering
	len=strlen(bp);
	for(i=0; i<len; i++)
		if(bp[i]<32 || bp[i]==127)
			bp[i]='.';

	return bp;
}

int SMBIOS_getSystemInformation(system_information& system_info,vector<string>& dmi_strings,string& errorMessage)
{

	vector<void*> infos;
	void *start_ptr;
	char *tableAddress;
	_smbios_entry_point *anchor;
	vector<int> indexes;
	int index_max;

	if((start_ptr=SMBIOS_getRawData(_SMBIOS_ANCHOR_SEARCH_OFFSET,_SMBIOS_ANCHOR_SEARCH_RANGE,errorMessage))==NULL)
		return FAILED;

	anchor=SMBIOS_getEntryPoint(start_ptr);

	if((tableAddress=(char*)SMBIOS_getRawData(anchor->structure_table_address,anchor->structure_table_length,errorMessage))==NULL)
	{
		free(start_ptr);
		return FAILED;
	}

	SMBIOS_getStructure(infos,tableAddress,anchor->number_of_smbios_structure,_SYSTEM_INFORMATION);

	memcpy(&system_info,infos[0],sizeof(system_information));

	if(system_info.length>=0x08)
	{
		indexes.push_back((int)system_info.manufacturer);
		indexes.push_back((int)system_info.product_name);
		indexes.push_back((int)system_info.version);
		indexes.push_back((int)system_info.serial_number);

		if(system_info.length>=0x19)
		{
			if(system_info.length>=0x1B)
			{
				indexes.push_back((int)system_info.sku_number);
				indexes.push_back((int)system_info.family);
			}
		}

	}

	index_max=*max_element(indexes.begin(), indexes.end());

	for(int i=0;i<=index_max;i++)
		dmi_strings.push_back(SMBIOS_getDmiString((_smbios_structure_header*)infos[0],i));

	free(tableAddress);
	free(start_ptr);

	return OK;
}

