/*################################################################################
# Linux Management Providers (LMP), Provider Common Library
# Copyright (C) 2008 Guillaume BOTTEX, ETRI <guillaumebottex@etri.re.kr, guillaumebottex@gmail.com>
#
# This program is being developed under the "OpenDRIM" project.
# The "OpenDRIM" project web page: http://opendrim.sourceforge.net
# The "OpenDRIM" project mailing list: opendrim@googlegroups.com
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; version 2
# of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#################################################################################

#################################################################################
# To contributors, please leave your contact information in this section
# AND comment your changes in the source code.
#
# 2008 Guillaume BOTTEX, ETRI <guillaumebottex@etri.re.kr, guillaumebottex@gmail.com>
################################################################################*/

#ifndef SMBIOS_H_
#define SMBIOS_H_

#include <algorithm>
#include <sys/mman.h>

#include "Common.h"

#define _SMBIOS_ANCHOR_SEARCH_OFFSET	0xF0000
#define _SMBIOS_ANCHOR_SEARCH_RANGE		0xFFFF

// Define SMBIOS structure types
#define _SYSTEM_INFORMATION			1
#define _PROCESSOR_INFORMATION			4
#define _CACHE_INFORMATION 				7
#define _MEMORY_DEVICE 					17
#define _MEMORY_ERROR_INFORMATION		18
#define _MEMORY_ARRAY_MAPPED_ADDRESS	19
#define _MEMORY_DEVICE_MAPPED_ADDRESS	20

#define DEV_MEM "/dev/mem"

typedef unsigned char u8;
typedef unsigned short u16;
typedef unsigned int u32;
typedef unsigned long long u64;

typedef signed short s16;

typedef u8 BYTE;
typedef u16 WORD;
typedef u32 DWORD;
typedef u64 QWORD;

typedef struct {
	char anchor_string[4]; // _SM_
} _smbios_anchor;

typedef struct {
	BYTE anchor_string[4]; // _SM_
	BYTE entry_point_structure_checksum;
	BYTE entry_point_length;
	BYTE smbios_major_version;
	BYTE smbios_minor_version;
	WORD maximum_structure_size;
	BYTE entry_point_revision;
	BYTE formatted_area[5];
	BYTE intermediate_anchor_string[5]; //_DMI_
	BYTE intermediate_checksum;
	WORD structure_table_length;
	DWORD structure_table_address;
	WORD number_of_smbios_structure;
	BYTE smbios_bcd_revision;
} _smbios_entry_point;

typedef struct {
	BYTE type;
	BYTE length;
	WORD handle;
} _smbios_structure_header;

typedef struct {
	// needed for mapping caches to processors
	unsigned short handle;
	// 0: Unknown, 1: Other, 2: Write Back, 3: Write Through, 4: Varies with Memory Address
	unsigned short operational_mode;
	bool enabled;
	// 0: Unknown, 1: Other, 2: Not Applicable, 3: Primary, 4: Secondary, 5: Tertiary
	unsigned short cache_level;
	// in bytes
	unsigned long long granularity;
	// in granularity unit
	unsigned long long installed_size;
	// 0: Unknown, 1: Other, 2: Instruction, 3: Data, 4: Unified
	unsigned short system_cache_type;
	// 0: Unknown, 1: Other, 2: Direct Mapped, 3: 2-way Set-Associative, 4: 4-way Set-Associative,
	// 5: Fully Associative, 6: 8-way Set-Associative, 7: 16-way Set-Associative
	unsigned short associativity;
} _cache_information;

typedef struct {
	// 1: Others, 2: Unknown, 3: Central Processor, 4: Math Processor, 5: DSP Processor, 6: Video Processor
	unsigned char processor_type;
	// 2: unknown
	unsigned short processor_family;
	bool has_voltage;
	// in milliVolts
	long voltage;
	// in MHz
	unsigned long external_clock;
	// in MHz
	unsigned long max_speed;
	// 0: Unknown, 1: CPU Enabled, 2: CPU Disabled by User via BIOS Setup, 3: CPU Disabled By BIOS (POST Error),
	// 4: CPU is Idle, wating to be enabled, 5-6: Reserved, 7: Other
	unsigned short status;
	// 1: Other, 2- : See SMBIOS reference specification
	unsigned short processor_upgrade;
	// 0: Unknown
	unsigned short core_enabled;
	// Empty: Unknown
	vector<unsigned short> processor_characteristics;
	bool has_l1_cache;
	unsigned short l1_cache_handle;
	_cache_information l1_cache;
	bool has_l2_cache;
	unsigned short l2_cache_handle;
	_cache_information l2_cache;
	bool has_l3_cache;
	unsigned short l3_cache_handle;
	_cache_information l3_cache;
} _processor_information;

typedef struct {
  char* string[256];
} _string_map;

typedef struct _memory_device
{
	BYTE type;
	BYTE length;
	WORD handle;
	WORD physical_memory;
	WORD memory_error;
	WORD total_width;
	WORD data_width;
	WORD size;
	BYTE form_factor;
	BYTE device_set;
	BYTE device_locator;
	BYTE bank_locator;
	BYTE memory_type;
	WORD type_detail;
	WORD speed;
	BYTE manufacturer;
	BYTE serial_number;
	BYTE asset_tag;
	BYTE part_number;
	BYTE attributes;
}memory_device;

typedef struct _memory_array_mapped_address
{
	BYTE type;
	BYTE length;
	WORD handle;
	DWORD starting_address;
	DWORD ending_address;
	WORD memory_array_handle;
	BYTE partition_width;
}memory_array_mapped_address;

typedef struct _memory_error_information
{
	BYTE type;
	BYTE length;
	WORD handle;
	BYTE error_type;
	BYTE error_granularity;
	BYTE error_operation;
	DWORD vendor_syndrome;
	DWORD memory_array_error_address;
	DWORD device_error_address;
	DWORD error_resolution;
}memory_error_information;

typedef struct _UUID
{
	DWORD time_low;
	WORD time_mid;
	WORD time_hi_and_version;
	BYTE clock_seq_hi_and_reserved;
	BYTE clock_seq_low;
	BYTE node[6];
}UUID;

typedef struct _system_information
{
	BYTE type;
	BYTE length;
	WORD handle;
	BYTE manufacturer;
	BYTE product_name;
	BYTE version;
	BYTE serial_number;
	UUID uuid;
	BYTE wake_up_type;
	BYTE sku_number;
	BYTE family;
}system_information;

enum MemoryType
{
	Other_MemoryType = 0x1,
	Unknown_MemoryType,
	DRAM,
	EDRAM,
	VRAM,
	SRAM,
	RAM,
	ROM,
	FLASH,
	EEPROM,
	FEPROM,
	EPROM,
	CDRAM,
	_3DRAM,
	SDRAM,
	SGRAM,
	RDRAM,
	DDR,
	DDR2,
	DDR2_FB_DIMM
};

enum MemoryTypeDetail
{
	Other_MemoryTypeDetail = 0x1,
	Unknown_MemoryTypeDetail,
	Fast_paged,
	Static_Column,
	Pseudo_static,
	RAMBus,
	Synchronous,
	CMOS,
	EDO,
	Window_DRAM,
	Cache_DRAM,
	Non_Volatile
};

enum MemoryErrorType
{
	Other_MemoryErrorType = 0x1,
	Unknown_MemoryErrorType,
	MemoryErrorType_OK,
	Bad_read,
	Parity_error,
	Single_bit_error,
	Double_bit_error,
	Multi_bit_error,
	Nibble_error,
	Checksum_error,
	CRC_error,
	Corrected_single_bit_error,
	Corrected_error,
	Uncorrectable_error
};

enum MemoryErrorGranularity
{
	Other_MemoryErrorGranularity = 0x1,
	Unknown_MemoryErrorGranularity,
	Device_level,
	Memory_partition_level
};

enum MemoryErrorOperation
{
	Other_MemoryErrorOperation = 0x1,
	Unknown_MemoryErrorOperation,
	Read,
	Write,
	Partial_write,
};

string SMBIOS_processorInformationToString(const _processor_information& processor_information, const string& identation);

string SMBIOS_cacheInformationToString(const _cache_information& cache_information, const string& identation);

void SMBIOS_cacheInformationToDefault(_cache_information& cache_information);

void SMBIOS_processorInformationToDefault(_processor_information& processor_information);

int SMBIOS_findEntryPoint(int& dev_mem, void** ptrptr, _smbios_anchor** smbios_anchor_ptrptr, string& errorMessage);

char* SMBIOS_nextSMBIOSStructureHeader(char* ptr, _string_map *string_map_ptr);

int SMBIOS_getProcessorsInformation(vector<_processor_information>& processors_information, string& errorMessage);

void *SMBIOS_getRawData(size_t base, size_t len, string& errorMessage);

_smbios_entry_point *SMBIOS_getEntryPoint(void *start);

int SMBIOS_getStructure(vector<void*>& output,char *tableAddress, u16 structureCount, u8 Type);

const char *SMBIOS_getDmiString(_smbios_structure_header *dm, BYTE s);

int SMBIOS_getSystemInformation(system_information& system_info,vector<string>& dmi_strings,string& errorMessage);

#endif /*SMBIOS_H_*/
