/* hbciabstracttrans.h

 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef HBCIABSTRACTTRANS_H
#define HBCIABSTRACTTRANS_H

/** @file abstracttrans.h
 *
 * @short Abstract base class HBCI::AbstractTrans definition. No C
 * wrappers.  */

#include <openhbci/date.h>
#include <openhbci/value.h>


#ifdef __cplusplus
#include <string>
#include <list>

namespace HBCI {

/** @short Abstract base class for transactions and similar business
 * cases in HBCI.
 *
 * This class is the abstract base class for transactions and similar
 * business cases in HBCI. This helps in unifying all their interfaces.
 *
 * @author Christian Stimming <stimming@tuhh.de> */
class DLLIMPORT AbstractTrans {
public:
    AbstractTrans();
    virtual ~AbstractTrans();

    /** @name Our account */
    /*@{*/
    /** country code of your institute -- MT940: not in MT940 */
    virtual int ourCountryCode() const = 0;
    /** country code of your institute -- MT940: not in MT940 */
    virtual void setOurCountryCode(int i) = 0;

    /** bank code of your institute (BLZ or S.W.I.F.T. code) -- MT940: BLZ in
      :25: */
    virtual const string &ourBankCode() const = 0;
    /** bank code of your institute (BLZ or S.W.I.F.T. code) -- MT940: BLZ in
      :25: */
    virtual void setOurBankCode(const string &code) = 0;

    /** Your account number; up to 23 characters. Note: possibly
     * including the currency (yes, that's what MT940 says) --
     * MT940: Kontonummer in :25: */
    virtual const string &ourAccountId() const = 0;
    /** Your account number; up to 23 characters. Note: possibly
     * including the currency (yes, that's what MT940 says) --
     * MT940: Kontonummer in :25: */
    virtual void setOurAccountId(const string &id) = 0;

    /** If your account has subaccounts which are distinguished by
     * different suffixes, then this is that suffix. Otherwise it's
     * empty. -- MT940: not in MT940 */
    virtual const string &ourSuffix() const = 0;
    /** If your account has subaccounts which are distinguished by
     * different suffixes, then this is that suffix. Otherwise it's
     * empty. -- MT940: not in MT940 */
    virtual void setOurSuffix(const string &id) = 0;
    /*@}*/

    /** @name Value */
    /*@{*/
    /** The value of this transaction in the account's currency. --
     * MT940: Betrag in :61: */
    virtual const Value &value() const = 0;
    /** The value of this transaction in the account's currency. --
     * MT940: Betrag in :61: */
    virtual void setValue(const Value &v) = 0;
    /*@}*/

    /* Now follows the obnoxious "Mehrzweckfeld :86:" */

    /** @name Transaction Description */
    /*@{*/
    /** A 3 digit numerical transaction code, defined for all kinds of
     * different HBCI actions. -- MT940: Geschaeftsvorfallcode, first field
     * in field :86:
     *
     * For a normal transfer you should set it to 51. For debit notes the
     * values 05 or 06 may be used. For other values please refer to your
     * credit institute.
     */
    virtual int transactionCode() const = 0;
    /** A 3 digit numerical transaction code, defined for all kinds of
     * different HBCI actions. -- MT940: Geschaeftsvorfallcode, first field
     * in field :86:
     *
     * For a normal transfer you should set it to 51. For debit notes the
     * values 05 or 06 may be used. For other values please refer to your
     * credit institute.
     */
    virtual void setTransactionCode(int c) = 0;

    /** Description of what this transaction is for, and how the Bank
     * describes this transaction.
     * 
     * According to HBCI Spec chapter IX.2.8, somewhere in these lines
     * this might contain the equivalent transaction value in DEM
     * vs. EUR or vice versa. Also, if the originalValue() and
     * charge() above was unused but this in fact had a original value
     * and charges, then that information might be contained somewhere
     * in these description lines. (Good luck.)
     * 
     * -- Verwendungszweck; MT940: fields 20-29 and 60-63 in field :86:  */
    virtual const list<string> &description() const = 0;
    /** Description of what this transaction is for, and how the Bank
     * describes this transaction.
     * 
     * According to HBCI Spec chapter IX.2.8, somewhere in these lines
     * this might contain the equivalent transaction value in DEM
     * vs. EUR or vice versa. Also, if the originalValue() and
     * charge() above was unused but this in fact had a original value
     * and charges, then that information might be contained somewhere
     * in these description lines. (Good luck.)
     * 
     * -- Verwendungszweck; MT940: fields 20-29 and 60-63 in field :86:  */
    virtual void addDescription(const string &s) = 0;
    /*@}*/

    /** @name Other account */
    /*@{*/
    /** Country code (BLZ) of the other party's institute -- MT940:
     * not in MT940/field :86: */
    virtual int otherCountryCode() const = 0;
    /** Country code (BLZ) of the other party's institute -- MT940:
     * not in MT940/field :86: */
    virtual void setOtherCountryCode(int i) = 0;

    /** Institute code (BLZ) of the other party's institute -- BLZ;
     * MT940: field 30 in field :86: */
    virtual const string &otherBankCode() const = 0;
    /** Institute code (BLZ) of the other party's institute -- BLZ;
     * MT940: field 30 in field :86: */
    virtual void setOtherBankCode(const string &s) = 0;

    /** Account number of the other party -- Kontonummer; MT940: field
     * 31 in field :86: */
    virtual const string &otherAccountId() const = 0;
    /** Account number of the other party -- Kontonummer; MT940: field
     * 31 in field :86: */
    virtual void setOtherAccountId(const string &s) = 0;

    /** If the account of the other party has subaccounts which are
     * distinguished by different suffixes, then this is that suffix
     * of the other party. Otherwise it's empty. -- MT940: *not* in
     * MT940 / field :86: */
    virtual const string &otherSuffix() const = 0;
    /** If the account of the other party has subaccounts which are
     * distinguished by different suffixes, then this is that suffix
     * of the other party. Otherwise it's empty. -- MT940: *not* in
     * MT940 / field :86: */
    virtual void setOtherSuffix(const string &id) = 0;

    /** Name of the other party -- MT940: fields 32 and 33 in field
     * :86: */
    virtual const list<string> &otherName() const = 0;
    /** Name of the other party -- MT940: fields 32 and 33 in field
     * :86: */
    virtual void addOtherName(const string &s) = 0;
    /*@}*/
};
} /* namespace HBCI */
extern "C" {   
#endif /* __cplusplus */

#ifdef __cplusplus
} /* extern "C" */
#endif


#endif
