/***************************************************************************
 $RCSfile: adminjobs.h,v $
                             -------------------
    cvs         : $Id: adminjobs.h,v 1.24 2003/06/10 17:48:40 aquamaniac Exp $
    begin       : Mon Nov 19 2001
    copyright   : (C) 2001 by Martin Preuss
    email       : openhbci@aquamaniac.de

 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/

/*
 */

#ifndef ADMINJOBS_H
#define ADMINJOBS_H

#ifdef __cplusplus

#include <string>

#include <openhbci/pointer.h>
#include <openhbci/job.h>
#include <openhbci/rsakey.h>
#include <openhbci/user.h>
#include <openhbci/userparams.h>
#include <openhbci/bankparams.h>
#include <openhbci/statusreport.h>


namespace HBCI {
/**
 * This job initializes a dialog with your credit institute.
 * Please not, that you MUST NOT use this job together with any other
 * job in ONE single MessageBox !!
 * This job needs to be th only one when sending a message to the server !
 * @author Martin Preuss<openhbci@aquamaniac.de>
 */
class DLLIMPORT JOBDialogInit :public Job {
private:
  bool _anonymous;
  bool _sync;
  bool _crypt;
  bool _sign;
  bool _getkeys;
  Pointer<userParams> _upd;
  Pointer<bankParams> _bpd;
  Pointer<RSAKey> _signKey;
  Pointer<RSAKey> _cryptKey;
public:
  /**
   * Constructor.
   * @param cust customer who opens the dialog
   * @param anonymous this is normally set to true. But in some cases (e.g.
   * if you dont have the institute's public keys or the institute does
   * not have yours) you have to contact the server anonymously, indicating
   * that neither signing nor encryption is to be perfomed
   * @author Martin Preuss<openhbci@aquamaniac.de>
   */
  JOBDialogInit(Pointer<Customer> cust,
                bool anonymous,
		bool needcrypt,
		bool needsign,
		bool getkeys,
		bool sync);
  ~JOBDialogInit();

  /**
   * internal
   */
  bool needsToBeSigned() const;

  /**
   * internal
   */
  bool needsToBeEncrypted() const;

  /**
   * internal
   * @author Martin Preuss<openhbci@aquamaniac.de>
   */
  string toString(int firstseg);

  /**
   * internal
   * @author Martin Preuss<openhbci@aquamaniac.de>
   */
  void parseResponse(const string& response);

  /**
   * This overwrites the general result handler of Job.
   * This is needed if your server does not send his public sign key.
   * In this case it would send an error message. But since that actually
   * is not an error we fake the result code for this segment in that case.
   * @author Martin Preuss<openhbci@aquamaniac.de>
   */
  void jobSuccess(const string& response);

  /**
   */
  Pointer<userParams> upd() const { return _upd;};
  Pointer<bankParams> bpd() const { return _bpd;};
  Pointer<RSAKey> serverSignKey() const { return _signKey;};
  Pointer<RSAKey> serverCryptKey() const { return _cryptKey;};
  void clearKeys() { _signKey=0; _cryptKey=0;};
};


/**
 * This job synchronizes some data between you and the credit institute.
 * For now this is only used to get our system id, which has to be assigned
 * to us by the credit institute, and which is necessary for most other jobs.
 * @author Martin Preuss<openhbci@aquamaniac.de>
 */
class DLLIMPORT JOBSynchronize :public Job {
private:
    int _syncmode;
    int _signatureId;
    int _msgNumber;
    string _sysid;

public:
    /**
     * Constructor.
     * @param cust customer who opened the dialog
     * @param syncmode Possible values are:<br>
     * - HBCI_SYNC_SYSTEMID    to sync the system id
     * - HBCI_SYNC_MSGNUMBER   to sync the msg number (not used for now)
     * - HBCI_SYNC_SIGNATUREID to sync the signature id (not used for now)
     * @author Martin Preuss<openhbci@aquamaniac.de>
     */
    JOBSynchronize(Pointer<Customer> cust,
                   int syncmode);

    ~JOBSynchronize();

    /**
     * internal
     */
    bool needsToBeSigned() const { return true;};

    /**
     * internal
     */
    bool needsToBeEncrypted() const { return true;};

     /**
     * internal
     * @author Martin Preuss<openhbci@aquamaniac.de>
     */
    string toString(int firstseg);

    /**
     * internal
     * @author Martin Preuss<openhbci@aquamaniac.de>
     */
    void parseResponse(const string& response);

    int signatureId() const { return _signatureId;};
    int msgNumber() const { return _msgNumber;};
    string sysid() const { return _sysid;};

};


/**
 * This job opens a dialog with the creadit institute and sends your
 * private keys. Only do this if your institute does not have your keys !
 * You should close the dialog directly after this job with a JOBDialogEnd
 * using encryption but without signing.
 * @author Martin Preuss<openhbci@aquamaniac.de>
 */
class DLLIMPORT JOBFirstInit :public Job {
private:

public:
    /**
     * Constructor.
     * @param cust the customer whose keys are to be transmitted.
     * @author Martin Preuss<openhbci@aquamaniac.de>
     */
    JOBFirstInit(Pointer<Customer> cust);
    ~JOBFirstInit();

    /**
     * internal
     */
    bool needsToBeSigned() const { return true;};

    /**
     * internal
     */
    bool needsToBeEncrypted() const { return true;};

    /**
     * internal
     * @author Martin Preuss<openhbci@aquamaniac.de>
     */
    string toString(int firstseg);

    /**
     * internal
     * @author Martin Preuss<openhbci@aquamaniac.de>
     */
    void parseResponse(const string& response);
};


/**
 * This job closes a dialog.
 * @author Martin Preuss<openhbci@aquamaniac.de>
 */
class DLLIMPORT JOBDialogEnd :public Job {
private:
    bool _crypt;
    bool _sign;
    string _dialogid;

public:
    /**
     * Constructor.
     * In most cases encryption and signing is needed. However, there are
     * cases when they are not (e.g. after a JOBFirstInit job).
     * @author Martin Preuss<openhbci@aquamaniac.de>
     * @param cust the customer who opened the dialog
     * @param dialogid guess what ;-)
     * @param crypt if true encryption is needed
     * @param sign if true signing is needed
     */
    JOBDialogEnd(Pointer<Customer> cust,
                 const string& dialogid,
                 bool crypt=true,
                 bool sign=true);

    ~JOBDialogEnd();

    /**
     * internal
     */
    bool needsToBeSigned() const { return _sign;};

    /**
     * internal
     */
    bool needsToBeEncrypted() const { return _crypt;};

     /**
     * internal
     * @author Martin Preuss<openhbci@aquamaniac.de>
     */
    string toString(int firstseg);

    /**
     * internal
     * @author Martin Preuss<openhbci@aquamaniac.de>
     */
    void parseResponse(const string& response);
};



/**
 * This job disables your public keys at the credit institute.
 * Use this job if your keys are either compromised or lost for other
 * reasons.
 * @author Martin Preuss<openhbci@aquamaniac.de>
 */
class DLLIMPORT JOBPublicKeysDisable :public Job {
private:
  int _keyVersion;
  int _keyNumber;

public:
  /**
   * Constructor.
   * If you still have access to your keys, simply ignore
   * keyNumber and keyVersion.
   * If your keys are lost, the only way to disable them anyway is,
   * to tell the bank which keynumber/version they shall lock. If you
   * don't know the number/version, use "999".
   * @author Fabian Kaiser<fabian@openhbci.de>
   * @param cust the customer whose public keys are to be disabled
   * @param keyNumber Number of your signature key, only used if you
   * don't have access to your keys anymore.
   * @param keyVersion Version number of your signature key, only used
   * if you don't have access to your keys anymore.
   */
  JOBPublicKeysDisable(Pointer<Customer> cust,
		       int keyNumber=0, int keyVersion=0);

  ~JOBPublicKeysDisable();

  /**
   * internal
   */
  bool needsToBeSigned() const;

  /**
   * internal
   */
  bool needsToBeEncrypted() const { return true;};

  /**
   * internal
   */
  string toString(int firstseg);

  /**
   * internal
   */
  void parseResponse(const string& response) {};
};


/**
 * This job allows you to change your public cryto- and signature key.
 * @author Fabian Kaiser<fabian@openhbci.de>
 */
class DLLIMPORT JOBPublicKeysChange :public Job {
public:
  /**
   *
   */
  JOBPublicKeysChange(Pointer<Customer> cust);

  ~JOBPublicKeysChange();

  /**
   * internal
   */
  bool needsToBeSigned() const { return true; };

  /**
   * internal
   */
  bool needsToBeEncrypted() const { return true;};

  /**
   * internal
   */
  string toString(int firstseg);

  /**
   * internal
   */
  void parseResponse(const string& response) {};
};


/**
 * @author Martin Preuss<openhbci@aquamaniac.de>
 */
class DLLIMPORT JOBGetStatusReport: public Job {
private:
  Date _fromdate;
  Date _todate;
  int _maxEntries;
  string _attachPoint;
  list<StatusReport> _reports;

public:
  /**
   * Constructor.
   * @param cust customer whose account you like to inspect
   * @param acc pointer to the @ref accountParams describing your account
   * @author Martin Preuss<openhbci@aquamaniac.de>
   */
  JOBGetStatusReport(Pointer<Customer> cust,
		     const Date &fromDate,
		     const Date &toDate,
		     int maxentries);

  JOBGetStatusReport(Pointer<Customer> cust,
		     const Date &fromDate,
		     const Date &toDate,
		     int maxentries,
		     Pointer<Job> lastJob);

  ~JOBGetStatusReport();

  /**
   * internal
   */
  bool needsToBeSigned() const { return true; };

  /**
   * internal
   */
  bool needsToBeEncrypted() const { return true; };

  /**
   * internal
   * @author Martin Preuss<openhbci@aquamaniac.de>
   */
  string toString(int firstseg);

  /**
   * internal
   * @author Martin Preuss<openhbci@aquamaniac.de>
   */
  void parseResponse(const string& response);

  /**
   * internal
   */
  bool attachMore();

  /**
   * Returns the status reports gathered so far
   */
  const list<StatusReport> &statusReports() const { return _reports;};

};


} /* namespace HBCI */
#endif /* __cplusplus */



#endif






