/***************************************************************************
 $RCSfile: customer.h,v $
                             -------------------
    cvs         : $Id: customer.h,v 1.9 2003/04/13 21:29:52 aquamaniac Exp $
    begin       : Sun Nov 18 2001
    copyright   : (C) 2001 by Martin Preuss
    email       : openhbci@aquamaniac.de

 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/


#ifndef HBCICUSTOMER_H
#define HBCICUSTOMER_H

/** @file customer.h
 *
 * @short Definitions of HBCI::Customer and its @ref HBCI_Customer C
 * wrapper. */

#ifdef __cplusplus
#include <string>
#include <list>
using namespace std;
#include <openhbci/pointer.h>
namespace HBCI {
	class Customer;
}
/* list declaration necessary for C wrappers */
typedef struct HBCI::Customer HBCI_Customer;
typedef list<HBCI::Pointer<HBCI::Customer> > list_HBCI_Customer;
typedef list<HBCI::Pointer<HBCI::Customer> >::const_iterator list_HBCI_Customer_iter;

#else /* __cplusplus */
typedef struct HBCI_Customer HBCI_Customer;
typedef struct list_HBCI_Customer list_HBCI_Customer;
typedef struct list_HBCI_Customer_iter list_HBCI_Customer_iter;
#endif /* __cplusplus */

#include <openhbci/user.h>

#ifdef __cplusplus

namespace HBCI {
/** @short Represents the Customer of a Bank in %HBCI (which is a
 * specific role of a User).
 *
 * %HBCI knows two kinds of id:<ul>
 * <li>user id</li>
 * <li>customer id</li>
 * </ul>
 * The User is the owner of the security medium, wheras the Customer
 * defines the role the user wants to play. Your bank may decide to assign
 * multiple customer ids to one user id.
 *
 * Example: You are running a business and want to manage your accounts
 * by means of %HBCI. You have one business account and one private account. 
 * In this case your bank may present you this situation:<ul>
 * <li> one <i>user id</i> for you as a person</li>
 * <li> one <i>customer id</i> for you as the boss of your office,
 * allowing access to your business account</li>
 * <li> one <i>customer id</i> for you as a private person, granting yourself
 * access to your private account.</li>
 * </ul>
 *
 * However, in most cases (private home banking) you will only have
 * <i>one</i> user id and <i>one</i> customer id, and the two may or
 * may not be equal.  Nevertheless %HBCI allows more complex
 * situations.
 *
 * @author Martin Preuss <openhbci@aquamaniac.de> */
class DLLIMPORT Customer {
private:
    Pointer<User> _user;
    string _custId;
    string _custName;

public:
    /** 
     *  Creates a new customer.
     *
     * @author Martin Preuss<openhbci@aquamaniac.de>
     * @param u The user this customer (role) belongs to
     * @param id The customer ID that the bank assigned to this customer.
     * @param custName The name of this customer. You can choose this
     * name as you like, e.g. "business", "private" or whatever you
     * prefer, and you can set this to any arbitrary string you consider
     * helpful.  */
    Customer(Pointer<User> u,
	     const string &id,
	     const string &custName="");

    /**  Copy constructor. */
    Customer(const Customer &cu);
    /**  Default destructor. */
    ~Customer();

    /** 
     *  Returns a pointer to the User this customer belongs to.
     *
     * @author Martin Preuss<openhbci@aquamaniac.de>
     */
    Pointer<User> user() const { return _user;};

    /**  Returns the customer id assigned to this Customer by
     * the Bank.
     *
     * The customer ID that identifies this customer to his
     * bank. Usually this string is assigned to the customer by the
     * bank.  */
    const string& custId() const { return _custId;};

    /**
     *  Returns the name of this customer.
     *
     * Returns the name of the customer, i.e. the role this customer
     * plays. You can choose this name as you like, e.g. "secretary",
     * "private" or whatever you prefer.  This member makes it easier
     * for you to distinguish this customers from others, since the
     * raw customer id is quite cryptic in most cases.
     */
    const string& custName() const { return _custName;};
    /**
     *  Sets the name of this customer.
     *
     * Sets the name of this customer, i.e. the role this customer
     * plays. You can set this to any arbitrary string you consider
     * helpful.
     * @param n Sets this to the new name of the customer.
     */
    void setCustName(const string &n) { _custName = n; };

    /** 
     * Equality operator
     */
    /* bool operator==(const HBCICustomer &c) const; */
};
} /* namespace HBCI */

#endif /* __cplusplus */

#ifdef __cplusplus
extern "C" {
#endif
    /** @name HBCI_Customer methods */
    /*@{*/
    /** 
     *  Returns a pointer to the User this customer belongs to.
     *
     * @param h The object to perform this action on.
     */
    extern const HBCI_User* HBCI_Customer_user(const HBCI_Customer *h);

    /**  Returns the customer id assigned to this Customer by
     * the Bank.
     *
     * The customer ID that identifies this customer to his
     * bank. Usually this string is assigned to the customer by the
     * bank.  
     * @param h The object to perform this action on.
     */
    extern const char* HBCI_Customer_custId(const HBCI_Customer *h);

    /**
     *  Returns the name of this customer.
     *
     * Returns the name of the customer, i.e. the role this customer
     * plays. You can choose this name as you like, e.g. "secretary",
     * "private" or whatever you prefer.  This member makes it easier
     * for you to distinguish this customers from others, since the
     * raw customer id is quite cryptic in most cases.
     * @param h The object to perform this action on.
     */
    extern const char* HBCI_Customer_custName(const HBCI_Customer *h);
    /**
     *  Sets the name of this customer.
     *
     * Sets the name of this customer, i.e. the role this customer
     * plays. You can set this to any arbitrary string you consider
     * helpful.
     * @param h The object to perform this action on.
     * @param n Sets this to the new name of the customer.
     */
    extern void HBCI_Customer_setCustName(HBCI_Customer *h, const char *n);
    /*@}*/

    /** @name C wrapper for list of Customer
     *
     * C wrapper for STL's list<Customer> type. To browse through a
     * list, get an iterator by list_HBCI_Customer_begin, increment it
     * by list_HBCI_Customer_iter_next, access its elements by
     * list_HBCI_Customer_iter_get, and see whether you're at the end
     * of the list by comparing with list_HBCI_Customer_iter_equal
     * against list_HBCI_Customer_end. */
    /*@{*/
    extern void list_HBCI_Customer_delete(list_HBCI_Customer *l);
    extern int list_HBCI_Customer_size(const list_HBCI_Customer *l);
    extern void list_HBCI_Customer_iter_delete(list_HBCI_Customer_iter *l);
    extern list_HBCI_Customer_iter* 
    list_HBCI_Customer_begin(const list_HBCI_Customer *l);
    extern list_HBCI_Customer_iter* 
    list_HBCI_Customer_end(const list_HBCI_Customer *l);
    extern const HBCI_Customer* 
    list_HBCI_Customer_iter_get(const list_HBCI_Customer_iter *l);
    extern void
    list_HBCI_Customer_iter_next(list_HBCI_Customer_iter *l);
    extern int 
    list_HBCI_Customer_iter_equal(const list_HBCI_Customer_iter *l1, 
				  const list_HBCI_Customer_iter *l2);
    /** Callback function for list_HBCI_Customer_foreach. */
    typedef void *(*list_HBCI_Customer_cb) (const HBCI_Customer *customer, 
					    void *user_data);
    /** Traverses the list, calling the callback function 'func' on
     * each list element.  Traversal will stop when 'func' returns a
     * non-NULL value, and the routine will return with that
     * value. Otherwise the routine will return NULL. 
     * @param list The list to traverse.
     * @param func The function to be called with each list element.
     * @param user_data A pointer passed on to the function 'func'.
     * @return The non-NULL pointer returned by 'func' as soon as it
     * returns one. Otherwise (i.e. 'func' always returns NULL)
     * returns NULL.
     * @author Christian Stimming <stimming@tuhh.de> */
    extern void *list_HBCI_Customer_foreach (const list_HBCI_Customer *list, 
					     list_HBCI_Customer_cb func,
					     void *user_data);
    /*@}*/
#ifdef __cplusplus
} /* __cplusplus */
#endif



#endif




