/* hbcistringlist.h
 *
 * Some helper functions for string list manipulation in C.
 *
 * Copyright (C) 2002 Christian Stimming
 */

/***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef HBCISTRINGLIST_H
#define HBCISTRINGLIST_H

/** @file hbcistringlist.h
 *
 * @short C wrapper for list of strings, @ref list_string. */

#ifdef __cplusplus
#include <string>
#include <list>
using namespace std;

typedef list<string> list_string;
typedef list<string>::const_iterator list_string_iter;

extern "C" {
    
#else /* __cplusplus */
    typedef struct list_string list_string;
    typedef struct list_string_iter list_string_iter;
#endif /* __cplusplus */

    extern void list_string_delete(list_string *l);
    extern void list_string_iter_delete(list_string_iter *l);
    extern int list_string_size(const list_string *l);
    extern list_string_iter* list_string_begin(const list_string *l);
    extern list_string_iter* list_string_end(const list_string *l);
    extern const char* list_string_iter_get(const list_string_iter *l);
    extern void list_string_iter_next(list_string_iter *l);
    extern int list_string_iter_equal(const list_string_iter *l1, 
				      const list_string_iter *l2);
    /** Callback function for list_string_foreach. */
    typedef void *(*list_string_cb) (const char *string, 
				     void *user_data);
    /** Traverses the list, calling the callback function 'func' on
     * each list element.  Traversal will stop when 'func' returns a
     * non-NULL value, and the routine will return with that
     * value. Otherwise the routine will return NULL. 
     * @param list The list to traverse.
     * @param func The function to be called with each list element.
     * @param user_data A pointer passed on to the function 'func'.
     * @return The non-NULL pointer returned by 'func' as soon as it
     * returns one. Otherwise (i.e. 'func' always returns NULL)
     * returns NULL.
     * @author Christian Stimming <stimming@tuhh.de> */
    extern void *list_string_foreach (const list_string *list, 
				      list_string_cb func,
				      void *user_data);
    /* Returns a char* of all list entries concatenated together, with
     * no separation in between. 
     *
     * The returned pointer will be owned by the caller and has to be
     * free'd when no longer in use.*/
    extern char *list_string_concat (const list_string *list);
    /* Returns a char* of all list entries concatenated together, with
     * the string 'delim' in between each single list element. 
     *
     * The returned pointer will be owned by the caller and has to be
     * free'd when no longer in use.*/
    extern char *list_string_concat_delim (const list_string *list,
					   const char *delim);

#ifdef __cplusplus
}
#endif

#endif
