/* listwrappers.h
 *
 * Some helper functions for STL-list manipulation from C.
 *
 * Copyright (C) 2002 Christian Stimming
 */

/***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef LISTWRAPPERS_H
#define LISTWRAPPERS_H

/** @file listwrappers.h
 *
 * @short C wrappers for STL lists, e.g. @ref list_int. */

#ifdef __cplusplus
#include <list>
using namespace std;

typedef std::list<int> list_int;
typedef std::list<int>::const_iterator list_int_iter;
extern "C" {
#else /* __cplusplus */
    typedef struct list_int list_int;
    typedef struct list_int_iter list_int_iter;
#endif /* __cplusplus */

    /** @name list_int 
     *
     * Wrapper for STL's list<int> class. */
    /*@{*/
    extern void list_int_delete(list_int *l);
    extern void list_int_iter_delete(list_int_iter *l);
    extern int list_int_size(const list_int *l);
    extern list_int_iter* list_int_begin(const list_int *l);
    extern list_int_iter* list_int_end(const list_int *l);
    extern int list_int_iter_get(const list_int_iter *l);
    extern void list_int_iter_next(list_int_iter *l);
    extern int list_int_iter_equal(const list_int_iter *l1, 
				   const list_int_iter *l2);
    /** Callback function for list_int_foreach. */
    typedef void *(*list_int_cb) (int value, 
				  void *user_data);
    /** Traverses the list, calling the callback function 'func' on
     * each list element.  Traversal will stop when 'func' returns a
     * non-NULL value, and the routine will return with that
     * value. Otherwise the routine will return NULL. 
     * @param list The list to traverse.
     * @param func The function to be called with each list element.
     * @param user_data A pointer passed on to the function 'func'.
     * @return The non-NULL pointer returned by 'func' as soon as it
     * returns one. Otherwise (i.e. 'func' always returns NULL)
     * returns NULL.
     * @author Christian Stimming <stimming@tuhh.de> */
    extern void *list_int_foreach (const list_int *list, 
				   list_int_cb func,
				   void *user_data);
    /*@}*/

#ifdef __cplusplus
}
#endif

#endif
