/*
 * syncml plugin - A syncml plugin for OpenSync
 * Copyright (C) 2008-2009  Michael Bell <michael.bell@opensync.org>
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 */

#include "syncml_ds_server.h"
#include "syncml_callbacks.h"
#include "syncml_devinf.h"
#include <libsyncml/data_sync_api/callbacks.h>

SmlBool ds_server_init_databases(SmlPluginEnv *env, OSyncPluginInfo *info, OSyncError **oerror)
{
	osync_trace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, env, info, oerror);
	SmlDatabase *database = NULL;
	SmlError *error = NULL;
	OSyncPluginConfig *config = osync_plugin_info_get_config(info);
	OSyncFormatEnv *formatenv = osync_plugin_info_get_format_env(info);
	OSyncList *l, *list = osync_plugin_info_get_objtype_sinks(info);

	for (l=list; l; l = l->next) {
		OSyncObjTypeSink *sink = (OSyncObjTypeSink *) l->data;
		osync_bool sinkEnabled = osync_objtype_sink_is_enabled(sink);
		osync_trace(TRACE_INTERNAL, "%s: enabled => %d", __func__, sinkEnabled);
		if (!sinkEnabled)
			continue;

		osync_objtype_sink_set_connect_func(sink, syncml_connect);
		osync_objtype_sink_set_connect_done_func(sink, syncml_connect_done);
                osync_objtype_sink_set_get_changes_func(sink, ds_server_get_changeinfo);
		osync_objtype_sink_set_commit_func(sink, ds_server_commit);
		osync_objtype_sink_set_committed_all_func(sink, ds_server_committed_all);

		// Request a state database from the framework.
		osync_objtype_sink_enable_state_db(sink, TRUE);

		const char *objtype = osync_objtype_sink_get_name(sink);
		OSyncPluginResource *res = osync_plugin_config_find_active_resource(config, objtype);
		if (!(database = syncml_config_parse_database(env, res, oerror)))
			goto oerror;
                
                database->sink = sink;
		osync_objtype_sink_ref(sink);

		/* TODO: Handle all available format sinks! */
		OSyncList *fs = osync_plugin_resource_get_objformat_sinks(res);
		OSyncObjFormatSink *fmtsink = osync_list_nth_data(fs, 0);
		const char *objformat = osync_objformat_sink_get_objformat(fmtsink);

		database->objformat = osync_format_env_find_objformat(formatenv, objformat);

		database->alert_type_mutex = g_mutex_new();
		database->alert_type_cond = g_cond_new();


		/* TODO: Handle error about missing objformat in a nice way. */
		g_assert(database->objformat);

		osync_objformat_ref(database->objformat);

                osync_objtype_sink_set_userdata(sink, database);

		env->databases = g_list_append(env->databases, database);

		if (!smlDataSyncAddDatastore(
                                        env->dsObject1,
					get_database_pref_content_type(database, oerror),
                                        NULL,
                                        database->url,
                                        &error))
                                goto error;
	}
	
	osync_list_free(list);
	list = NULL;

	osync_trace(TRACE_EXIT, "%s - TRUE", __func__);
	return TRUE;
error:
	osync_error_set(oerror, OSYNC_ERROR_GENERIC, "%s", smlErrorPrint(&error));
	smlErrorDeref(&error);
oerror:
	if (list)
		osync_list_free(list);

	osync_trace(TRACE_EXIT_ERROR, "%s - %s", __func__, osync_error_print(oerror));
	return FALSE;
}

void ds_server_get_changeinfo(OSyncObjTypeSink *sink, OSyncPluginInfo *info, OSyncContext *ctx, osync_bool slow_sync, void *data)
{
	osync_trace(TRACE_ENTRY, "%s(%p, %p, %p)", __func__, data, info, ctx);

        SmlDatabase *database = (SmlDatabase*)data;
	SmlPluginEnv *env = database->env;

	database->slowsync = slow_sync;

        database->getChangesCtx = ctx;
        osync_context_ref(database->getChangesCtx);

	smlDataSyncRegisterChangeCallback(env->dsObject1, _recv_change, env);

	osync_trace(TRACE_EXIT, "%s", __func__);
	return;
}

void ds_server_commit(OSyncObjTypeSink *sink, OSyncPluginInfo *info, OSyncContext *ctx, OSyncChange *change, void *data)
{
    osync_trace(TRACE_ENTRY, "%s", __func__);
    g_assert(ctx);

    SmlError  *error = NULL;
    OSyncError  *oserror = NULL;
    SmlDatabase *database = (SmlDatabase*)data;

    database->pendingCommits++;
    osync_trace(TRACE_INTERNAL, "%s - %i changes present to send",
	__func__, database->pendingCommits);

    struct commitContext *tracer = osync_try_malloc0(sizeof(struct commitContext), &oserror);
    if (!tracer)
        goto oserror;
    tracer->change = change;
    tracer->context = ctx;
    tracer->database = database;

    osync_change_ref(change);
    osync_context_ref(ctx);

    // prepare data
    OSyncData *odata = osync_change_get_data(change);
    char *buf = NULL;
    unsigned int size = 0;
    osync_data_get_data(odata, &buf, &size);

    osync_trace(TRACE_INTERNAL, "%s: Committing entry \"%s\": \"%s\"",
                __func__, osync_change_get_uid(change), buf);
    if (!smlDataSyncAddChange(
		database->env->dsObject1,
		database->url,
		_get_changetype(change),
		osync_change_get_uid(change),
		buf, size,
		tracer, &error))
        goto error;

    osync_trace(TRACE_EXIT, "%s", __func__);
    return;

error:
    osync_error_set(&oserror, OSYNC_ERROR_GENERIC, "%s", smlErrorPrint(&error));
    smlErrorDeref(&error);
oserror:
    if (tracer) {
        osync_change_unref(change);
        osync_context_unref(ctx);
        safe_free((gpointer *) tracer);
    }
    osync_trace(TRACE_EXIT_ERROR, "%s: %s", __func__, osync_error_print(&oserror));
    report_error_on_context(&ctx, &oserror, TRUE);
}

void ds_server_committed_all(OSyncObjTypeSink *sink, OSyncPluginInfo *info, OSyncContext *ctx, void *data)
{
    osync_trace(TRACE_ENTRY, "%s", __func__);
    g_assert(ctx);

    SmlError  *error = NULL;
    OSyncError  *oserror = NULL;
    SmlDatabase *database = (SmlDatabase*)data;

    database->env->gotDatabaseCommits++;

    database->commitCtx = ctx;
    osync_context_ref(database->commitCtx);

    g_assert(database->pendingChanges == 0); 

    if (g_list_length(database->env->databases) == database->env->gotDatabaseCommits &&
        !smlDataSyncSendChanges(database->env->dsObject1, &error))
		goto error;

    osync_trace(TRACE_EXIT, "%s", __func__);
    return;

error:
    osync_error_set(&oserror, OSYNC_ERROR_GENERIC, "%s", smlErrorPrint(&error));
    smlErrorDeref(&error);
    osync_trace(TRACE_EXIT_ERROR, "%s: %s", __func__, osync_error_print(&oserror));
    report_error_on_context(&(database->commitCtx), &oserror, TRUE);
}

