#include "osl/record/record.h"
#include "osl/record/csaIOError.h"
#include "osl/record/kanjiCode.h"
#include "osl/misc/iconvConvert.h"
#include <boost/algorithm/string/classification.hpp>
#include <boost/algorithm/string/split.hpp>
#include <boost/algorithm/string/replace.hpp>
#include <boost/lexical_cast.hpp>
#include <boost/foreach.hpp>
#include <stack>
#include <iostream>

osl::record::
MoveRecord::MoveRecord(const Move& mv, int ni)
  : move(mv), nodeIndex(ni), time(0)
{
}

namespace osl
{
namespace record
{
  IRecordStream::~IRecordStream(){}
  
  ORecordStream::~ORecordStream(){}
  
  const Move MoveRecord::getMove() const { return move; }
  
  int MoveRecord::getNodeIndex() const { return nodeIndex; }
  
  void MoveRecord::setTime(int t){
    time=t;
  }
  
  void NodeRecord::addMoveRecord(int moveIndex){
    moves.push_back(moveIndex);
  }
  
  
  Record::Record(){ init(); }
  Record::Record(const SimpleState& initial, const vector<Move>& moves)
  {
    init();
    NumEffectState copy(initial);
    RecordVisitor visitor;
    visitor.setState(&copy);
    visitor.setRecord(this);
    BOOST_FOREACH(Move move, moves) {
      visitor.addMoveAndAdvance(move);
    }
  }  
  void Record::init(){
    version="";
    playerNames[0]=playerNames[1]="";
    nrs.clear();
    nrs.push_back(NodeRecord());
    initialState.init(HIRATE);
    result = UNKNOWN;
    start_date = boost::gregorian::date();
  }

  void Record::load(IRecordStream& irs){
    irs.load(this);
  }
  void Record::save(ORecordStream& /*irs*/){
    // irs.save(this);
  }
  
  void Record::setVersion(const std::string& str){
    version=str;
  }
  void Record::setPlayer(Player player,const std::string& str){
    playerNames[player]=str;
  }
  const std::string& Record::getPlayer(Player player) const{
    return playerNames[player];
  }
  void Record::setInitialState(const SimpleState& state){
    initialState=state;
    initialState.initPawnMask();
  }
  const NumEffectState Record::getInitialState() const {
    if (! initialState.isConsistent(true))
    {
      const char *msg = "Record: bad initial state";
      std::cerr << msg << " " << __FILE__ << " " << __LINE__ << "\n";
      throw CsaIOError(msg);
    }
    return NumEffectState(initialState);
  }
  int Record::addNodeRecord(){
    nrs.push_back(NodeRecord());
    return nrs.size()-1;
  }
  int Record::addMoveRecord(const MoveRecord& moveRecord){
    mrs.push_back(moveRecord);
    return mrs.size()-1;
  }
  NodeRecord* Record::nodeOf(int index){
    return &nrs.at(index);
  }
  const NodeRecord* Record::nodeOf(int index) const{
    return &nrs.at(index);
  }
  MoveRecord* Record::moveOf(int index){
    if (static_cast<size_t>(index) >= mrs.size())
      return NULL;
    else
      return &mrs.at(index);
  }
  const MoveRecord* Record::moveOf(int index) const {
    if (static_cast<size_t>(index) >= mrs.size())
      return NULL;
    else
      return &mrs.at(index);
  }
  NodeRecord& Record::operator[](int index){
    return nrs.at(index);
  }
  void Record::setDate(const std::string& date_str)
  {
    std::vector<std::string> values;
    boost::algorithm::split(values, date_str, boost::algorithm::is_any_of("/"));
    if (values.size() < 3) {
      std::cerr << "ERROR: Invalid date format found: "
#ifndef MINIMAL
		<< IconvConvert::eucToLang(date_str)
#endif
		<< "\n";
      return;
    } else if (values.size() > 3) {
      std::cerr << "WARNING: Invalid date format found: "
#ifndef MINIMAL
		<< IconvConvert::eucToLang(date_str)
#endif
		<< "\n";
      // go through
    }
    BOOST_FOREACH(std::string& value, values) {
      static const CArray<const char *,9> kanji = {{
	  K_R1, K_R2, K_R3, K_R4, K_R5, K_R6, K_R7, K_R8, K_R9, 
	}};
      for (size_t i=0; i<kanji.size(); ++i)
	boost::algorithm::replace_all(value, kanji[i], std::string(1, char('0'+i)));
    }
    int year  = 0;
    int month = 0;
    int day   = 0;
    try {
      year  = boost::lexical_cast<int>(values[0]);
      month = boost::lexical_cast<int>(values[1]);
      if (month == 0) month = 1;
      if ("??" == values[2]) {
        std::cerr << "WARNING: Invalid date format found: "
#ifndef MINIMAL
		  << IconvConvert::eucToLang(values[2])
#endif
		  << "\n";
        // go through
        day = 1;
      } else if (values[2].size() > 2) {
        std::cerr << "WARNING: Invalid date format found: "
#ifndef MINIMAL
		  << IconvConvert::eucToLang(values[2])
#endif
		  << "\n";
        // go through
        day = boost::lexical_cast<int>(values[2].substr(0,2));
      } else {
        day = boost::lexical_cast<int>(values[2]);
      }
      if (day == 0) day = 1;
      start_date = boost::gregorian::date(year, month, day);
      assert(!start_date.is_special());
    } catch (boost::bad_lexical_cast& e) {
      std::cerr << "Invalid date format found: "
#ifndef MINIMAL
		<< IconvConvert::eucToLang(date_str)
#endif
		<< "\n"
                << e.what() << std::endl;
    } catch (boost::gregorian::bad_day_of_month& ebdm) {
      std::cerr << "Bad day of month: "
#ifndef MINIMAL
		<< IconvConvert::eucToLang(date_str)
#endif
		<< "\n"
                << ebdm.what() << std::endl;
    }
    return;
  }
  void Record::setDate(const boost::gregorian::date& date)
  {
    start_date = date;
  }
  boost::gregorian::date Record::getDate() const
  {
    return start_date;
  }
  
  void RecordVisitor::addMoveAndAdvance(Move move){
    assert(state->isValidMove(move));

    int newNode=rec->addNodeRecord();
    int newMove=rec->addMoveRecord(MoveRecord(move,newNode));
    (*rec)[nodeIndex].addMoveRecord(newMove);
    nodeIndex=newNode;
    lastMoveIndex=newMove;

    assert(state->isConsistent() || ((std::cerr << move <<"\n"<< *state),0));
    NumEffectState copy(*state);
    copy.makeMove(move);	// XXX: RecordVisitor should have NumEffectState
    *state = copy;
    assert(state->isConsistent() || ((std::cerr << move <<"\n"<< *state),0));
    for(boost::ptr_vector<record::RecordVisitorObserver>::iterator each = observers.begin(); each != observers.end(); ++each){ 
      each->update(this); 
    }
  }
  
  
  std::ostream& operator<<(std::ostream& os,const MoveRecord & mr){
    return os << "MoveRecord(" << // mr.getMove() << ',' <<
      mr.getNodeIndex()  << ')';
  }
#ifndef MINIMAL  
  std::ostream& operator<<(std::ostream& os,Record & r){
    os << "Record(";
    os << "version=" << r.getVersion()
       << ",BLACK=" << r.getPlayer(BLACK)
       << ",WHITE=" << r.getPlayer(WHITE);
    os << ",initial=" << std:: endl << r.getInitialState() << std::endl;
    SimpleState initial_state=r.getInitialState();
    NumEffectState state(initial_state);
    RecordVisitor visitor;
    visitor.setState(&state);
    visitor.setRecord(&r);
    NodeRecord* node=visitor.getNode();
    while(node->size()>0){
      int moveIndex=node->at(0);
      MoveRecord* mr=r.moveOf(moveIndex);
      Move move=mr->getMove();
      os << move << "," << mr->getTime() << "," << mr->getComment() << std::endl;
      node=r.nodeOf(mr->getNodeIndex());
      state.makeMove(move);
      assert(state.isConsistent());
    }
    os << state;
    os << initial_state;
    return os << ')';
  }
#endif  
  
  const vector<Move> Record::getMoves() const {
    vector<Move> moves;
    vector<int> dummy_time;
    getMoves(moves, dummy_time);
    return moves;
  }

  int readInt(std::istream& is)
  {
    int ret=0;
    CArray<char,4> cs;
    is.read(&cs[0],4);
    for (int i=0;i<4;i++) {
      ret = (ret<<8)|(cs[i]&255);
    }
    return ret;
  }
  
  void
  writeInt(std::ostream& os, int n)
  {
    CArray<char,4> buf;
    for (int i = 0; i < 4; i++)
      {
	buf[i] = (n >> (8 * (4 - i - 1))) & 255;
      }
    os.write(&buf[0], 4);
  }

} // namespace record
} // namespace osl

void osl::record::
Record::getMoves(vector<Move>& moves, vector<int>& times,
		 vector<std::string>& comments,
		 vector<SearchInfo>& info) const 
{
  const NodeRecord* node=nodeOf(0);
  while(node->size()>0){
    const int moveIndex=node->at(0);
    const MoveRecord* mr=moveOf(moveIndex);
    const Move move=mr->getMove();
    moves.push_back(move);
    times.push_back(mr->getTime());
    comments.push_back(mr->getComment());
    info.push_back(mr->info);

    node=nodeOf(mr->getNodeIndex());
  }
}

void osl::record::
Record::getMoves(vector<Move>& moves, vector<int>& times) const 
{
  vector<std::string> dummy_comments;
  vector<SearchInfo> dummy_info;
  getMoves(moves, times, dummy_comments, dummy_info);
}

osl::record::
RecordVisitor::~RecordVisitor()
{
}

// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
